/*
 * Decompiled with CFR 0.152.
 */
package org.apache.beam.sdk.io.hdfs;

import java.io.FileNotFoundException;
import java.io.IOException;
import java.io.OutputStream;
import java.net.URI;
import java.nio.ByteBuffer;
import java.nio.channels.Channels;
import java.nio.channels.ReadableByteChannel;
import java.nio.channels.SeekableByteChannel;
import java.nio.channels.WritableByteChannel;
import java.nio.file.FileAlreadyExistsException;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.HashSet;
import java.util.List;
import java.util.Set;
import org.apache.beam.sdk.io.FileSystem;
import org.apache.beam.sdk.io.fs.CreateOptions;
import org.apache.beam.sdk.io.fs.MatchResult;
import org.apache.beam.sdk.io.fs.ResourceId;
import org.apache.beam.sdk.io.hdfs.HadoopResourceId;
import org.apache.beam.vendor.guava.v20_0.com.google.common.annotations.VisibleForTesting;
import org.apache.beam.vendor.guava.v20_0.com.google.common.collect.ImmutableList;
import org.apache.hadoop.conf.Configuration;
import org.apache.hadoop.fs.FSDataInputStream;
import org.apache.hadoop.fs.FileStatus;
import org.apache.hadoop.fs.FileUtil;
import org.apache.hadoop.fs.Path;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

class HadoopFileSystem
extends FileSystem<HadoopResourceId> {
    private static final Logger LOG = LoggerFactory.getLogger(HadoopFileSystem.class);
    @VisibleForTesting
    static final String LOG_CREATE_DIRECTORY = "Creating directory %s";
    @VisibleForTesting
    static final String LOG_DELETING_EXISTING_FILE = "Deleting existing file %s";
    @VisibleForTesting
    final org.apache.hadoop.fs.FileSystem fileSystem;

    HadoopFileSystem(Configuration configuration) throws IOException {
        this.fileSystem = org.apache.hadoop.fs.FileSystem.newInstance((Configuration)configuration);
    }

    protected List<MatchResult> match(List<String> specs) {
        ImmutableList.Builder resultsBuilder = ImmutableList.builder();
        for (String spec : specs) {
            try {
                HashSet<MatchResult.Metadata> metadata = new HashSet<MatchResult.Metadata>();
                if (spec.contains("**")) {
                    int index = spec.indexOf("**");
                    metadata.addAll(this.matchRecursiveGlob(spec.substring(0, index + 1), spec.substring(index + 1)));
                } else {
                    FileStatus[] fileStatuses = this.fileSystem.globStatus(new Path(spec));
                    if (fileStatuses != null) {
                        for (FileStatus fileStatus : fileStatuses) {
                            metadata.add(this.toMetadata(fileStatus));
                        }
                    }
                }
                if (metadata.isEmpty()) {
                    resultsBuilder.add((Object)MatchResult.create((MatchResult.Status)MatchResult.Status.NOT_FOUND, Collections.emptyList()));
                    continue;
                }
                resultsBuilder.add((Object)MatchResult.create((MatchResult.Status)MatchResult.Status.OK, new ArrayList(metadata)));
            }
            catch (IOException e) {
                resultsBuilder.add((Object)MatchResult.create((MatchResult.Status)MatchResult.Status.ERROR, (IOException)e));
            }
        }
        return resultsBuilder.build();
    }

    private Set<MatchResult.Metadata> matchRecursiveGlob(String directorySpec, String fileSpec) throws IOException {
        HashSet<MatchResult.Metadata> metadata = new HashSet<MatchResult.Metadata>();
        if (directorySpec.contains("*")) {
            FileStatus[] directoryStatuses;
            for (FileStatus directoryStatus : directoryStatuses = this.fileSystem.globStatus(new Path(directorySpec))) {
                if (!directoryStatus.isDirectory()) continue;
                metadata.addAll(this.matchRecursiveGlob(directoryStatus.getPath().toUri().toString(), fileSpec));
            }
        } else {
            FileStatus[] directoryStatuses;
            FileStatus[] fileStatuses;
            for (FileStatus fileStatus : fileStatuses = this.fileSystem.globStatus(new Path(directorySpec + "/" + fileSpec))) {
                if (!fileStatus.isFile()) continue;
                metadata.add(this.toMetadata(fileStatus));
            }
            for (FileStatus directoryStatus : directoryStatuses = this.fileSystem.globStatus(new Path(directorySpec + "/*"))) {
                if (!directoryStatus.isDirectory()) continue;
                metadata.addAll(this.matchRecursiveGlob(directoryStatus.getPath().toUri().toString(), fileSpec));
            }
            if (fileSpec.contains("**")) {
                int index = fileSpec.indexOf("**");
                metadata.addAll(this.matchRecursiveGlob(directorySpec + "/" + fileSpec.substring(0, index + 1), fileSpec.substring(index + 1)));
            }
        }
        return metadata;
    }

    private MatchResult.Metadata toMetadata(FileStatus fileStatus) {
        URI uri = HadoopFileSystem.dropEmptyAuthority(fileStatus.getPath().toUri().toString());
        return MatchResult.Metadata.builder().setResourceId((ResourceId)new HadoopResourceId(uri)).setIsReadSeekEfficient(true).setSizeBytes(fileStatus.getLen()).setLastModifiedMillis(fileStatus.getModificationTime()).build();
    }

    protected WritableByteChannel create(HadoopResourceId resourceId, CreateOptions createOptions) throws IOException {
        return Channels.newChannel((OutputStream)this.fileSystem.create(resourceId.toPath()));
    }

    protected ReadableByteChannel open(HadoopResourceId resourceId) throws IOException {
        FileStatus fileStatus = this.fileSystem.getFileStatus(resourceId.toPath());
        return new HadoopSeekableByteChannel(fileStatus, this.fileSystem.open(resourceId.toPath()));
    }

    protected void copy(List<HadoopResourceId> srcResourceIds, List<HadoopResourceId> destResourceIds) throws IOException {
        for (int i = 0; i < srcResourceIds.size(); ++i) {
            boolean success = FileUtil.copy((org.apache.hadoop.fs.FileSystem)this.fileSystem, (Path)srcResourceIds.get(i).toPath(), (org.apache.hadoop.fs.FileSystem)this.fileSystem, (Path)destResourceIds.get(i).toPath(), (boolean)false, (boolean)true, (Configuration)this.fileSystem.getConf());
            if (success) continue;
            throw new IOException(String.format("Unable to copy resource %s to %s. No further information provided by underlying filesystem.", srcResourceIds.get(i).toPath(), destResourceIds.get(i).toPath()));
        }
    }

    protected void rename(List<HadoopResourceId> srcResourceIds, List<HadoopResourceId> destResourceIds) throws IOException {
        for (int i = 0; i < srcResourceIds.size(); ++i) {
            Path src = srcResourceIds.get(i).toPath();
            Path dest = destResourceIds.get(i).toPath();
            this.mkdirs(dest);
            boolean success = this.fileSystem.rename(src, dest);
            if (!success && this.fileSystem.exists(src) && this.fileSystem.exists(dest)) {
                LOG.debug(String.format(LOG_DELETING_EXISTING_FILE, Path.getPathWithoutSchemeAndAuthority((Path)dest)));
                this.fileSystem.delete(dest, false);
                success = this.fileSystem.rename(src, dest);
            }
            if (success) continue;
            if (!this.fileSystem.exists(src)) {
                throw new FileNotFoundException(String.format("Unable to rename resource %s to %s as source not found.", src, dest));
            }
            if (this.fileSystem.exists(dest)) {
                throw new FileAlreadyExistsException(String.format("Unable to rename resource %s to %s as destination already exists and couldn't be deleted.", src, dest));
            }
            throw new IOException(String.format("Unable to rename resource %s to %s. No further information provided by underlying filesystem.", src, dest));
        }
    }

    private void mkdirs(Path filePath) throws IOException {
        Path targetDirectory = filePath.getParent();
        if (!this.fileSystem.exists(targetDirectory)) {
            LOG.debug(String.format(LOG_CREATE_DIRECTORY, Path.getPathWithoutSchemeAndAuthority((Path)targetDirectory)));
            boolean success = this.fileSystem.mkdirs(targetDirectory);
            if (!success) {
                throw new IOException(String.format("Unable to create target directory %s. No further information provided by underlying filesystem.", targetDirectory));
            }
        }
    }

    protected void delete(Collection<HadoopResourceId> resourceIds) throws IOException {
        for (HadoopResourceId resourceId : resourceIds) {
            this.fileSystem.delete(resourceId.toPath(), false);
        }
    }

    protected HadoopResourceId matchNewResource(String singleResourceSpec, boolean isDirectory) {
        if (singleResourceSpec.endsWith("/") && !isDirectory) {
            throw new IllegalArgumentException(String.format("Expected file path but received directory path %s", singleResourceSpec));
        }
        return !singleResourceSpec.endsWith("/") && isDirectory ? new HadoopResourceId(HadoopFileSystem.dropEmptyAuthority(singleResourceSpec + "/")) : new HadoopResourceId(HadoopFileSystem.dropEmptyAuthority(singleResourceSpec));
    }

    protected String getScheme() {
        return this.fileSystem.getScheme();
    }

    private static URI dropEmptyAuthority(String uriStr) {
        URI uri = URI.create(uriStr);
        String prefix = uri.getScheme() + ":///";
        if (uriStr.startsWith(prefix)) {
            return URI.create(uri.getScheme() + ":/" + uriStr.substring(prefix.length()));
        }
        return uri;
    }

    private static class HadoopSeekableByteChannel
    implements SeekableByteChannel {
        private final FileStatus fileStatus;
        private final FSDataInputStream inputStream;
        private boolean closed;

        private HadoopSeekableByteChannel(FileStatus fileStatus, FSDataInputStream inputStream) {
            this.fileStatus = fileStatus;
            this.inputStream = inputStream;
            this.closed = false;
        }

        @Override
        public int read(ByteBuffer dst) throws IOException {
            if (this.closed) {
                throw new IOException("Channel is closed");
            }
            int read = 0;
            read = dst.hasArray() ? this.inputStream.read(dst.array(), dst.position() + dst.arrayOffset(), dst.remaining()) : this.inputStream.read(dst);
            if (read > 0) {
                dst.position(dst.position() + read);
            }
            return read;
        }

        @Override
        public int write(ByteBuffer src) {
            throw new UnsupportedOperationException();
        }

        @Override
        public long position() throws IOException {
            if (this.closed) {
                throw new IOException("Channel is closed");
            }
            return this.inputStream.getPos();
        }

        @Override
        public SeekableByteChannel position(long newPosition) throws IOException {
            if (this.closed) {
                throw new IOException("Channel is closed");
            }
            this.inputStream.seek(newPosition);
            return this;
        }

        @Override
        public long size() throws IOException {
            if (this.closed) {
                throw new IOException("Channel is closed");
            }
            return this.fileStatus.getLen();
        }

        @Override
        public SeekableByteChannel truncate(long size) {
            throw new UnsupportedOperationException();
        }

        @Override
        public boolean isOpen() {
            return !this.closed;
        }

        @Override
        public void close() throws IOException {
            this.closed = true;
            this.inputStream.close();
        }
    }
}

