package nl.bimbase.bimworks.client;

import java.util.HashMap;
import java.util.HashSet;
import java.util.Map;
import java.util.Set;
import java.util.TreeMap;
import java.util.TreeSet;

import com.fasterxml.jackson.databind.JsonNode;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.fasterxml.jackson.databind.node.ArrayNode;
import com.fasterxml.jackson.databind.node.DoubleNode;
import com.fasterxml.jackson.databind.node.ObjectNode;

import nl.bimbase.bimworks.actions.Discipline;

public class BimQuery {
	public static final BimQuery ALL = new BimQuery();
	private static final ObjectMapper OBJECT_MAPPER = new ObjectMapper();
	private static final int VERSION = 1;
	private int version = 1;
	private Set<String> types;
	private Set<Integer> buildingStoreyLevels;
	private Set<Qid> qids;
	private Map<String, FieldQuery> fields;
	private Map<String, Map<String, PropertyQuery>> propertySets;
	private Set<CalculatedValueQuery> calculatedValues;
	private Set<Discipline> disciplines;

	public BimQuery() {
		
	}
	
	public BimQuery addType(String... types) {
		if (this.types == null) {
			this.types = new TreeSet<>(); // Canonical
		}
		for (String type : types) {
			this.types.add(type);
		}
		return this;
	}
	
	public BimQuery addBuildingStoreyLevel(int level) {
		if (this.buildingStoreyLevels == null) {
			this.buildingStoreyLevels = new HashSet<>();
		}
		this.buildingStoreyLevels.add(level);
		return this;
	}
	
	public ObjectNode toJson() {
		ObjectNode queryNode = OBJECT_MAPPER.createObjectNode();
		queryNode.put("version", version);
		if (types != null) {
			ArrayNode typesNode = OBJECT_MAPPER.createArrayNode();
			for (String type : types) {
				typesNode.add(type);
			}
			queryNode.set("types", typesNode);
		}
		if (buildingStoreyLevels != null) {
			ArrayNode buildingStoreyLevelsNode = OBJECT_MAPPER.createArrayNode();
			for (int level : buildingStoreyLevels) {
				buildingStoreyLevelsNode.add(level);
			}
			queryNode.set("buildingStoreyLevels", buildingStoreyLevelsNode);
		}
		if (qids != null) {
			ArrayNode qidsNode = OBJECT_MAPPER.createArrayNode();
			for (Qid qid : qids) {
				qidsNode.add(qid.toString());
			}
			queryNode.set("qids", qidsNode);
		}
		if (disciplines != null) {
			ArrayNode disciplinesNode = OBJECT_MAPPER.createArrayNode();
			for (Discipline discipline : disciplines) {
				disciplinesNode.add(discipline.name());
			}
			queryNode.set("disciplines", disciplinesNode);
		}
		if (fields != null) {
			ArrayNode fieldsNode = OBJECT_MAPPER.createArrayNode();
			for (String fieldName : fields.keySet()) {
				FieldQuery fieldQuery = fields.get(fieldName);
				fieldsNode.add(fieldQuery.toJson());
			}
			queryNode.set("fields", fieldsNode);
		}
		if (propertySets != null) {
			ArrayNode propertiesNode = OBJECT_MAPPER.createArrayNode();
			for (String propertySetName : propertySets.keySet()) {
				Map<String, PropertyQuery> map = propertySets.get(propertySetName);
				for (String propertyName : map.keySet()) {
					propertiesNode.add(map.get(propertyName).toJson());
				}
			}
			queryNode.set("properties", propertiesNode);
		}
		if (calculatedValues != null) {
			ArrayNode calculatedValuesNode = OBJECT_MAPPER.createArrayNode();
			for (CalculatedValueQuery calculatedValueQuery : calculatedValues) {
				calculatedValuesNode.add(calculatedValueQuery.toJson());
			}
			queryNode.set("calculated", calculatedValuesNode);
		}
		return queryNode;
	}
	
	public BimQuery addProperty(String propertySetName, String propertyName, Operator operator, JsonNode value) {
		if (this.propertySets == null) {
			this.propertySets = new HashMap<>();
		}
		Map<String, PropertyQuery> map = this.propertySets.get(propertySetName);
		if (map == null) {
			map = new HashMap<>();
			this.propertySets.put(propertySetName, map);
		}
		map.put(propertyName, new PropertyQuery(propertySetName, propertyName, operator, value));
		return this;
	}

	public static BimQuery of(JsonNode jsonNode) throws UnsupportedBimQueryVersion, BimQueryValidationException {
		if (jsonNode == null) {
			return BimQuery.ALL;
		}
		BimQuery bimQuery = new BimQuery();
		if (!jsonNode.has("version")) {
			throw new BimQueryValidationException("version field is required");
		}
		bimQuery.version = jsonNode.get("version").asInt();
		if (bimQuery.version != VERSION) {
			throw new UnsupportedBimQueryVersion(bimQuery.version, VERSION);
		}
		if (jsonNode.has("types")) {
			ArrayNode typesNode = (ArrayNode)jsonNode.get("types");
			for (JsonNode typeNode : typesNode) {
				bimQuery.addType(typeNode.asText());
			}
		}
		if (jsonNode.has("buildingStoreyLevels")) {
			ArrayNode buildingStoreyLevelsNode = (ArrayNode) jsonNode.get("buildingStoreyLevels");
			for (JsonNode buildingStoreyLevelNode : buildingStoreyLevelsNode) {
				bimQuery.addBuildingStoreyLevel(buildingStoreyLevelNode.asInt());
			}
		}
		if (jsonNode.has("qids")) {
			ArrayNode qidsNode = (ArrayNode) jsonNode.get("qids");
			for (JsonNode qidNode : qidsNode) {
				bimQuery.addQid(Qid.of(qidNode.asText()));
			}
		}
		if (jsonNode.has("fields")) {
			ArrayNode fieldsNode = (ArrayNode) jsonNode.get("fields");
			for (JsonNode fieldNode : fieldsNode) {
				bimQuery.addField(fieldNode.get("name").asText(), Operator.bySymbol(fieldNode.get("operator").asText()), fieldNode.get("value"));
			}
		}
		if (jsonNode.has("disciplines")) {
			ArrayNode disciplinesNode = (ArrayNode) jsonNode.get("disciplines");
			for (JsonNode disciplineNode : disciplinesNode) {
				bimQuery.addDiscipline(Discipline.valueOf(disciplineNode.asText().toLowerCase()));
			}
		}
		if (jsonNode.has("properties")) {
			ArrayNode propertiesNode = (ArrayNode) jsonNode.get("properties");
			for (JsonNode propertyNode : propertiesNode) {
				String propertySetName = propertyNode.has("propertySetName") ? propertyNode.get("propertySetName").asText() : null;
				if (propertyNode.has("value")) {
					bimQuery.addProperty(propertySetName, propertyNode.get("name").asText(), Operator.bySymbol(propertyNode.get("operator").asText()), propertyNode.get("value"));
				} else {
					bimQuery.addProperty(propertySetName, propertyNode.get("name").asText());
				}
			}
		}
		if (jsonNode.has("calculated")) {
			ArrayNode calculatedValuesNode = (ArrayNode) jsonNode.get("calculated");
			for (JsonNode calculatedValueNode : calculatedValuesNode) {
				if (calculatedValueNode.has("name")) {
					String[] path = calculatedValueNode.get("name").asText().split("\\.");
					bimQuery.addCalculatedValue(new CalculatedValueQuery(path, Operator.bySymbol(calculatedValueNode.get("operator").asText()), calculatedValueNode.get("value")));
				}
			}
		}
		return bimQuery;
	}
	
	public void addDiscipline(Discipline discipline) {
		if (disciplines == null) {
			this.disciplines = new HashSet<>();
		}
		this.disciplines.add(discipline);
	}

	@Override
	public String toString() {
		return toJson().toString();
	}
	
	private void addQid(Qid qid) {
		if (this.qids == null) {
			this.qids = new TreeSet<>();
		}
		this.qids.add(qid);
	}

	public void addField(String fieldName, Operator operator, JsonNode value) {
		if (this.fields == null) {
			this.fields = new TreeMap<>();
		}
		this.fields.put(fieldName, new FieldQuery(fieldName, operator, value));
	}

	public void addField(String fieldName, Operator operator, double value) {
		if (this.fields == null) {
			this.fields = new TreeMap<>();
		}
		this.fields.put(fieldName, new FieldQuery(fieldName, operator, DoubleNode.valueOf(value)));
	}

	public BimQuery addProperty(String propertySetName, String propertyName) {
		if (this.propertySets == null) {
			this.propertySets = new TreeMap<>();
		}
		Map<String, PropertyQuery> map = this.propertySets.get(propertySetName);
		if (map == null) {
			map = new TreeMap<>();
			this.propertySets.put(propertySetName, map);
		}
		map.put(propertyName, new PropertyQuery(propertySetName, propertyName, Operator.EXISTS, null));
		return this;
	}

	public BimQuery addCalculatedValue(CalculatedValueQuery calculatedValueQuery) {
		if (this.calculatedValues == null) {
			this.calculatedValues = new TreeSet<>();
		}
		calculatedValues.add(calculatedValueQuery);
		return this;
	}
	
	public Set<Discipline> getDisciplines() {
		return disciplines;
	}
}