/*
 * Decompiled with CFR 0.152.
 */
package net.snowflake.client.core.crl;

import java.security.cert.X509CRL;
import java.security.cert.X509Certificate;
import java.time.LocalDate;
import java.time.ZoneId;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Date;
import java.util.List;
import java.util.stream.Collectors;
import net.snowflake.client.log.SFLogger;
import net.snowflake.client.log.SFLoggerFactory;
import org.bouncycastle.asn1.ASN1OctetString;
import org.bouncycastle.asn1.ASN1Primitive;
import org.bouncycastle.asn1.DERIA5String;
import org.bouncycastle.asn1.x509.CRLDistPoint;
import org.bouncycastle.asn1.x509.DistributionPoint;
import org.bouncycastle.asn1.x509.DistributionPointName;
import org.bouncycastle.asn1.x509.GeneralName;
import org.bouncycastle.asn1.x509.GeneralNames;
import org.bouncycastle.asn1.x509.IssuingDistributionPoint;

class CRLValidationUtils {
    private static final SFLogger logger = SFLoggerFactory.getLogger(CRLValidationUtils.class);
    private static final Date MARCH_15_2024 = Date.from(LocalDate.of(2024, 3, 15).atStartOfDay(ZoneId.of("UTC")).toInstant());
    private static final Date MARCH_15_2026 = Date.from(LocalDate.of(2026, 3, 15).atStartOfDay(ZoneId.of("UTC")).toInstant());

    CRLValidationUtils() {
    }

    static List<String> extractCRLDistributionPoints(X509Certificate cert) {
        ArrayList<String> crlUrls = new ArrayList<String>();
        try {
            byte[] extensionBytes = cert.getExtensionValue("2.5.29.31");
            if (extensionBytes == null) {
                logger.debug("No CRL Distribution Points extension found for certificate: {}", cert.getSubjectX500Principal());
                return crlUrls;
            }
            ASN1OctetString octetString = (ASN1OctetString)ASN1Primitive.fromByteArray((byte[])extensionBytes);
            CRLDistPoint crlDistPoint = CRLDistPoint.getInstance((Object)ASN1Primitive.fromByteArray((byte[])octetString.getOctets()));
            DistributionPoint[] distributionPoints = crlDistPoint.getDistributionPoints();
            if (distributionPoints != null) {
                for (DistributionPoint dp : distributionPoints) {
                    DistributionPointName dpName = dp.getDistributionPoint();
                    if (dpName == null || dpName.getType() != 0) continue;
                    GeneralNames generalNames = (GeneralNames)dpName.getName();
                    for (GeneralName generalName : generalNames.getNames()) {
                        String url;
                        if (generalName.getTagNo() != 6 || !(url = ((DERIA5String)generalName.getName()).getString()).toLowerCase().startsWith("http://") && !url.toLowerCase().startsWith("https://")) continue;
                        logger.debug("Found CRL URL: {}", url);
                        crlUrls.add(url);
                    }
                }
            }
        }
        catch (Exception e) {
            logger.debug("Failed to extract CRL distribution points from certificate {}: {}", cert.getSubjectX500Principal(), e.getMessage());
        }
        logger.debug("Extracted {} CRL URLs for certificate: {}", crlUrls.size(), cert.getSubjectX500Principal());
        return crlUrls;
    }

    static boolean isShortLived(X509Certificate cert) {
        Date notBefore = cert.getNotBefore();
        Date notAfter = cert.getNotAfter();
        if (notBefore.before(MARCH_15_2024)) {
            return false;
        }
        long maxValidityPeriodMs = notBefore.before(MARCH_15_2026) ? 864000000L : 604800000L;
        long actualValidityPeriodMs = notAfter.getTime() - notBefore.getTime();
        return actualValidityPeriodMs <= (maxValidityPeriodMs += 60000L);
    }

    static boolean verifyIssuingDistributionPoint(X509CRL crl, X509Certificate cert, String crlUrl) {
        try {
            byte[] extensionBytes = crl.getExtensionValue("2.5.29.28");
            if (extensionBytes == null) {
                logger.debug("No IDP extension found - CRL covers all certificates", new Object[0]);
                return true;
            }
            ASN1OctetString octetString = (ASN1OctetString)ASN1Primitive.fromByteArray((byte[])extensionBytes);
            IssuingDistributionPoint idp = IssuingDistributionPoint.getInstance((Object)ASN1Primitive.fromByteArray((byte[])octetString.getOctets()));
            if (idp.onlyContainsUserCerts() && cert.getBasicConstraints() != -1) {
                logger.debug("CRL only covers user certificates, but certificate is a CA certificate", new Object[0]);
                return false;
            }
            if (idp.onlyContainsCACerts() && cert.getBasicConstraints() == -1) {
                logger.debug("CRL only covers CA certificates, but certificate is not a CA certificate", new Object[0]);
                return false;
            }
            DistributionPointName dpName = idp.getDistributionPoint();
            if (dpName != null && dpName.getType() == 0) {
                GeneralNames generalNames = (GeneralNames)dpName.getName();
                boolean foundMatch = false;
                for (GeneralName generalName : generalNames.getNames()) {
                    String idpUrl;
                    if (generalName.getTagNo() != 6 || !(idpUrl = ((DERIA5String)generalName.getName()).getString()).equals(crlUrl)) continue;
                    foundMatch = true;
                    break;
                }
                if (!foundMatch) {
                    logger.debug("CRL URL {} not found in IDP distribution points - this CRL is not authorized for this certificate", crlUrl);
                    return false;
                }
            }
            logger.debug("IDP extension verification passed", new Object[0]);
            return true;
        }
        catch (Exception e) {
            logger.debug("Failed to verify IDP extension: {}", e.getMessage());
            return false;
        }
    }

    static String getCertChainSubjects(List<X509Certificate[]> certificateChains) {
        return certificateChains.stream().flatMap(Arrays::stream).map(cert -> cert.getSubjectX500Principal().getName()).collect(Collectors.joining(", "));
    }
}

