/*
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *    http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package net.shibboleth.shared.httpclient;

import java.net.InetAddress;
import java.net.ProxySelector;
import java.net.UnknownHostException;
import java.nio.charset.Charset;
import java.time.Duration;
import java.util.List;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;

import org.apache.hc.client5.http.HttpRequestRetryStrategy;
import org.apache.hc.client5.http.SchemePortResolver;
import org.apache.hc.client5.http.auth.AuthScope;
import org.apache.hc.client5.http.auth.CredentialsProvider;
import org.apache.hc.client5.http.auth.UsernamePasswordCredentials;
import org.apache.hc.client5.http.classic.HttpClient;
import org.apache.hc.client5.http.config.ConnectionConfig;
import org.apache.hc.client5.http.config.RequestConfig;
import org.apache.hc.client5.http.impl.DefaultSchemePortResolver;
import org.apache.hc.client5.http.impl.auth.CredentialsProviderBuilder;
import org.apache.hc.client5.http.impl.io.ManagedHttpClientConnectionFactory;
import org.apache.hc.client5.http.impl.io.PoolingHttpClientConnectionManagerBuilder;
import org.apache.hc.client5.http.io.HttpClientConnectionManager;
import org.apache.hc.client5.http.io.ManagedHttpClientConnection;
import org.apache.hc.client5.http.routing.HttpRoutePlanner;
import org.apache.hc.client5.http.socket.LayeredConnectionSocketFactory;
import org.apache.hc.core5.http.HttpHost;
import org.apache.hc.core5.http.HttpRequestInterceptor;
import org.apache.hc.core5.http.HttpResponseInterceptor;
import org.apache.hc.core5.http.config.CharCodingConfig;
import org.apache.hc.core5.http.config.Http1Config;
import org.apache.hc.core5.http.io.HttpConnectionFactory;
import org.apache.hc.core5.util.TimeValue;
import org.apache.hc.core5.util.Timeout;

import net.shibboleth.shared.annotation.constraint.NotLive;
import net.shibboleth.shared.annotation.constraint.Unmodifiable;
import net.shibboleth.shared.collection.CollectionSupport;
import net.shibboleth.shared.logic.Constraint;
import net.shibboleth.shared.primitive.StringSupport;

//TODO retry attempts, keep alive strategy

/**
 * Builder used to construct {@link HttpClient} objects configured with particular settings.
 * 
 * <p>
 * When using the single-arg constructor variant to wrap an existing instance of
 * {@link org.apache.hc.client5.http.impl.classic.HttpClientBuilder}, there are several caveats of which to be aware:
 * </p>
 * 
 * <p>
 * Instances of the following will be unconditionally overwritten by this builder when
 * {@link #buildClient()} is called:
 * </p>
 * 
 * <ul>
 * <li>{@link HttpClientConnectionManager} which includes the following sub-components:
 *   <ul>
 *     <li>Default {@link ConnectionConfig}</li>
 *     <li>{@link HttpConnectionFactory}</li>
 *     <li>{@link LayeredConnectionSocketFactory} used as the <code>SSLSocketFactory</code></li>
 *   </ul>
 * </li>
 * <li>{@link HttpRoutePlanner}</li>
 * <li>Default {@link RequestConfig}</li>
 * <li>Default {@link CredentialsProvider}</li>
 * </ul>
 * 
 * <p>
 * This is due to the unfortunate fact that the Apache builder does not currently provide accessor methods to obtain the
 * default instances currently set on the builder. Therefore, if you need to set any config parameters which are not
 * exposed by this builder, then you must use the Apache builder directly and may not use this builder.
 * </p>
 */
public class HttpClientBuilder {

    /** Local IP address used when establishing connections. Default value: system default local address */
    @Nullable private InetAddress socketLocalAddress;

    /** Maximum period inactivity between two consecutive data packets. Default value: (60 seconds) */
    @Nonnull private Duration socketTimeout;

    /** Socket buffer size in bytes. Default size is 8192 bytes. */
    private int socketBufferSize;

    /** Maximum length of time to wait for the connection to be established. Default value: (60 seconds) */
    @Nonnull private Duration connectionTimeout;
    
    /**
     * Maximum length of time to wait for a connection to be returned from the connection manager.
     * Default value: (60 seconds);
     */
    @Nonnull private Duration connectionRequestTimeout;
    
    /** Determines the timeout until arrival of a response from the opposite endpoint. */
    @Nonnull private Duration responseTimeout;
    
    /**
     * Max total simultaneous connections allowed by the pooling connection manager.
     */
    private int maxConnectionsTotal;
    
    /**
     * Max simultaneous connections per route allowed by the pooling connection manager.
     */
    private int maxConnectionsPerRoute;

    /** Whether the SSL/TLS certificates used by the responder should be ignored. Default value: false */
    private boolean connectionDisregardTLSCertificate;
    
    /** The TLS socket factory to use.  Optional, defaults to null. */
    @Nullable private LayeredConnectionSocketFactory tlsSocketFactory;

    /** Whether to instruct the server to close the connection after it has sent its response. Default value: true */
    private boolean connectionCloseAfterResponse;

    /**
     * Sets period after inactivity after which persistent connections must be checked to ensure they are still valid.
     */
    @Nullable private Duration validateAfterInactivity;

    /** Host name of the HTTP proxy server through which connections will be made. Default value: null. */
    @Nullable private String connectionProxyHost;
    
    /** Apache UserAgent. */
    @Nullable private String userAgent;

    /** Port number of the HTTP proxy server through which connections will be made. Default value: 8080. */
    private int connectionProxyPort;

    /** Username used to connect to the HTTP proxy server. Default value: null. */
    @Nullable private String connectionProxyUsername;

    /** Password used to connect to the HTTP proxy server. Default value: null. */
    @Nullable private String connectionProxyPassword;

    /** Whether to follow HTTP redirects. Default value: true */
    private boolean httpFollowRedirects;

    /** Character set used for HTTP entity content. Default value: UTF-8 */
    @Nullable private String httpContentCharSet;

    /** Strategy which determines whether and how a retry should be attempted. */
    @Nullable private HttpRequestRetryStrategy retryStrategy;
    
    /** Resolver for port based on a scheme. */
    @Nullable private SchemePortResolver schemePortResolver;
    
    /** Flag for disabling auth caching.*/
    private boolean disableAuthCaching;

    /** Flag for disabling automatic retries.*/
    private boolean disableAutomaticRetries;

    /** Flag for disabling connection state.*/
    private boolean disableConnectionState;

    /** Flag for disabling content compression.*/
    private boolean disableContentCompression;

    /** Flag for disabling cookie management.*/
    private boolean disableCookieManagement;

    /** Flag for disabling redirect handling.*/
    private boolean disableRedirectHandling;
    
    /** Flag for enabling use of system properties.*/
    private boolean useSystemProperties;

    /** Flag for evicting expired connections from the connection pool using a background thread. */
    private boolean evictExpiredConnections;
    
    /** Flag for evicting expired connections from the connection pool using a background thread. */
    private boolean evictIdleConnections;

    /** Max idle time allowed for an idle connection before it is evicted. */
    @Nonnull private Duration connectionMaxIdleTime;

    /** List of request interceptors to add first. */
    @Nonnull @Unmodifiable @NotLive private List<HttpRequestInterceptor> requestInterceptorsFirst;

    /** List of request interceptors to add last. */
    @Nonnull @Unmodifiable @NotLive private List<HttpRequestInterceptor> requestInterceptorsLast;

    /** List of response interceptors to add first. */
    @Nonnull @Unmodifiable @NotLive private List<HttpResponseInterceptor> responseInterceptorsFirst;

    /** List of response interceptors to add last. */
    @Nonnull @Unmodifiable @NotLive private List<HttpResponseInterceptor> responseInterceptorsLast;
    
    /** List of static context handlers. */
    @Nonnull @Unmodifiable @NotLive private List<HttpClientContextHandler> staticContextHandlers;

    /** The Apache HttpClientBuilder 4.3+ instance over which to layer this builder. */
    @Nonnull private final org.apache.hc.client5.http.impl.classic.HttpClientBuilder apacheBuilder;

    /** Constructor. */
    public HttpClientBuilder() {
        this(org.apache.hc.client5.http.impl.classic.HttpClientBuilder.create());
    }

    /**
     * Constructor.
     * 
     * @param builder the Apache HttpClientBuilder 4.3+ instance over which to layer this builder
     */
    public HttpClientBuilder(@Nonnull final org.apache.hc.client5.http.impl.classic.HttpClientBuilder builder) {
        apacheBuilder = Constraint.isNotNull(builder, "Apache HttpClientBuilder may not be null");
        
        // Defaults are duplicated to avoid static null analyzer issues.
        maxConnectionsTotal = -1;
        maxConnectionsPerRoute = -1;
        socketLocalAddress = null;
        socketBufferSize = 8192;
        socketTimeout = Duration.ofSeconds(60);
        responseTimeout = Duration.ofNanos(60);
        connectionTimeout = Duration.ofSeconds(60);
        connectionRequestTimeout = Duration.ofSeconds(60);
        connectionDisregardTLSCertificate = false;
        connectionCloseAfterResponse = true;
        connectionProxyHost = null;
        connectionProxyPort = 8080;
        connectionProxyUsername = null;
        connectionProxyPassword = null;
        httpFollowRedirects = true;
        httpContentCharSet = "UTF-8";
        userAgent = null;
        validateAfterInactivity = null;
        retryStrategy = null;
        schemePortResolver = null;
        
        disableAuthCaching = false;
        disableAutomaticRetries = false;
        disableConnectionState = false;
        disableContentCompression = false;
        disableCookieManagement = false;
        disableRedirectHandling = false;
        useSystemProperties = false;

        evictExpiredConnections = false;
        evictIdleConnections = false;
        connectionMaxIdleTime = Duration.ofMinutes(30);
        
        requestInterceptorsFirst = CollectionSupport.emptyList();
        requestInterceptorsLast = CollectionSupport.emptyList();
        responseInterceptorsFirst = CollectionSupport.emptyList();
        responseInterceptorsLast = CollectionSupport.emptyList();
        staticContextHandlers = CollectionSupport.emptyList();
    }

    /** Resets all builder parameters to their defaults. */
    public void resetDefaults() {
        
        // If changed, change constructor above.
        
        maxConnectionsTotal = -1;
        maxConnectionsPerRoute = -1;
        socketLocalAddress = null;
        socketBufferSize = 8192;
        socketTimeout = Duration.ofSeconds(60);
        responseTimeout = Duration.ofNanos(60);
        connectionTimeout = Duration.ofSeconds(60);
        connectionRequestTimeout = Duration.ofSeconds(60);
        connectionDisregardTLSCertificate = false;
        connectionCloseAfterResponse = true;
        connectionProxyHost = null;
        connectionProxyPort = 8080;
        connectionProxyUsername = null;
        connectionProxyPassword = null;
        httpFollowRedirects = true;
        httpContentCharSet = "UTF-8";
        userAgent = null;
        validateAfterInactivity = null;
        retryStrategy = null;
        schemePortResolver = null;
        
        disableAuthCaching = false;
        disableAutomaticRetries = false;
        disableConnectionState = false;
        disableContentCompression = false;
        disableCookieManagement = false;
        disableRedirectHandling = false;
        useSystemProperties = false;
        
        evictExpiredConnections = false;
        evictIdleConnections = false;
        connectionMaxIdleTime = Duration.ofMinutes(30);

        requestInterceptorsFirst = CollectionSupport.emptyList();
        requestInterceptorsLast = CollectionSupport.emptyList();
        responseInterceptorsFirst = CollectionSupport.emptyList();
        responseInterceptorsLast = CollectionSupport.emptyList();
        staticContextHandlers = CollectionSupport.emptyList();
    }

    /**
     * Gets the max total simultaneous connections allowed by the pooling connection manager.
     * 
     * @return the max total connections
     */
    public int getMaxConnectionsTotal() {
        return maxConnectionsTotal;
    }

    /**
     * Sets the max total simultaneous connections allowed by the pooling connection manager.
     * 
     * @param max the max total connection
     */
    public void setMaxConnectionsTotal(final int max) {
        maxConnectionsTotal = max;
    }

    /**
     * Gets the max simultaneous connections per route allowed by the pooling connection manager.
     * 
     * @return the max connections per route
     */
    public int getMaxConnectionsPerRoute() {
        return maxConnectionsPerRoute;
    }

    /**
     * Sets the max simultaneous connections per route allowed by the pooling connection manager.
     * 
     * @param max the max connections per route
     */
    public void setMaxConnectionsPerRoute(final int max) {
        maxConnectionsPerRoute = max;
    }

    /**
     * Gets the local IP address used when making requests.
     * 
     * @return local IP address used when making requests
     */
    public InetAddress getSocketLocalAddress() {
        return socketLocalAddress;
    }

    /**
     * Sets the local IP address used when making requests.
     * 
     * @param address local IP address used when making requests
     */
    public void setSocketLocalAddress(final InetAddress address) {
        socketLocalAddress = address;
    }

    /**
     * Sets the local IP address used when making requests.
     * 
     * @param ipOrHost IP address or hostname, never null
     * 
     * @throws UnknownHostException thrown if the given IP or hostname can not be resolved
     */
    public void setSocketLocalAddress(final String ipOrHost) throws UnknownHostException {
        socketLocalAddress = InetAddress.getByName(Constraint.isNotNull(ipOrHost, "IP or hostname may not be null"));
    }

    /**
     * Gets the timeout until arrival of a response from the opposite endpoint.
     * 
     * @return the response timeout
     */
    @Nonnull public Duration getResponseTimeout() {
        return responseTimeout;
    }

    /**
     * Gets the timeout until arrival of a response from the opposite endpoint.
     * 
     * @param timeout the response timeout
     */
    public void setResponseTimeout(@Nonnull final Duration timeout) {
        Constraint.isNotNull(timeout, "Timeout cannot be null");
        Constraint.isLessThanOrEqual(Integer.MAX_VALUE, timeout.toMillis(), "Timeout too large");

        responseTimeout = timeout;
    }

    /**
     * Gets the maximum period inactivity between two consecutive data packets. A value of less than 1 ms
     * indicates no timeout.
     * 
     * @return maximum period inactivity between two consecutive data packets
     */
    @Nonnull public Duration getSocketTimeout() {
        return socketTimeout;
    }

    /**
     * Sets the maximum period inactivity between two consecutive data packets. A value of less than 1 ms
     * indicates no timeout.
     * 
     * @param timeout maximum period inactivity between two consecutive data packets
     */
    public void setSocketTimeout(@Nonnull final Duration timeout) {
        Constraint.isNotNull(timeout, "Timeout cannot be null");
        Constraint.isLessThanOrEqual(Integer.MAX_VALUE, timeout.toMillis(), "Timeout too large");

        socketTimeout = timeout;
    }

    /**
     * Gets the size of the socket buffer, in bytes, used for request/response buffering.
     * 
     * @return size of the socket buffer, in bytes, used for request/response buffering
     */
    public int getSocketBufferSize() {
        return socketBufferSize;
    }

    /**
     * Sets size of the socket buffer, in bytes, used for request/response buffering.
     * 
     * @param size size of the socket buffer, in bytes, used for request/response buffering; must be greater than 0
     */
    public void setSocketBufferSize(final int size) {
        socketBufferSize = Constraint.isGreaterThan(0, size, "Socket buffer size must be greater than 0");
    }

    /**
     * Gets the maximum length of time to wait for the connection to be established. A value of less
     * than 1 ms indicates no timeout.
     * 
     * @return maximum length of time to wait for the connection to be established
     */
    @Nonnull public Duration getConnectionTimeout() {
        return connectionTimeout;
    }

    /**
     * Sets the maximum length of time to wait for the connection to be established. A value of less
     * than 1 ms indicates no timeout.
     * 
     * @param timeout maximum length of time to wait for the connection to be established
     */
    public void setConnectionTimeout(@Nonnull final Duration timeout) {
        Constraint.isNotNull(timeout, "Connection timeout cannot be null");
        Constraint.isLessThanOrEqual(Integer.MAX_VALUE, timeout.toMillis(), "Connection timeout too large");

        connectionTimeout = timeout;
    }

    /**
     * Gets the maximum length of time to wait for a connection to be returned from the connection
     * manager. A value of less than 1 ms indicates no timeout.
     * 
     * @return maximum length of time to wait for the connection to be established
     */
    @Nonnull public Duration getConnectionRequestTimeout() {
        return connectionRequestTimeout;
    }

    /**
     * Sets the maximum length of time to wait for a connection to be returned from the connection
     * manager. A value of less than 1 ms indicates no timeout.
     * 
     * @param timeout maximum length of time to wait for the connection to be established
     */
    public void setConnectionRequestTimeout(@Nonnull final Duration timeout) {
        Constraint.isNotNull(timeout, "Connection request timeout cannot be null");
        Constraint.isLessThanOrEqual(Integer.MAX_VALUE, timeout.toMillis(), "Connection request timeout too large");
        
        connectionRequestTimeout = timeout;
    }

    /**
     * Gets whether the responder's SSL/TLS certificate should be ignored.
     * 
     * <p>
     * This flag is overridden and ignored if a custom TLS socket factory is specified via
     * {@link #setTLSSocketFactory}.
     * </p>
     * 
     * @return whether the responder's SSL/TLS certificate should be ignored
     */
    public boolean isConnectionDisregardTLSCertificate() {
        return connectionDisregardTLSCertificate;
    }

    /**
     * Sets whether the responder's SSL/TLS certificate should be ignored.
     * 
     * <p>
     * This flag is overridden and ignored if a custom TLS socket factory is specified via
     * {@link #setTLSSocketFactory}.
     * </p>
     * 
     * @param disregard whether the responder's SSL/TLS certificate should be ignored
     */
    public void setConnectionDisregardTLSCertificate(final boolean disregard) {
        connectionDisregardTLSCertificate = disregard;
    }

    /**
     * Get the TLS socket factory to use.
     * 
     * @return the socket factory, or null.
     */
    @Nullable public LayeredConnectionSocketFactory getTLSSocketFactory() {
        return tlsSocketFactory;
    }

    /**
     * Set the TLS socket factory to use.
     * 
     * @param factory the new socket factory, may be null
     */
    public void setTLSSocketFactory(@Nullable final LayeredConnectionSocketFactory factory) {
        tlsSocketFactory = factory;
    }

    /**
     * Gets whether to instruct the server to close the connection after it has sent its response.
     * 
     * @return whether to instruct the server to close the connection after it has sent its response
     */
    public boolean isConnectionCloseAfterResponse() {
        return connectionCloseAfterResponse;
    }

    /**
     * Sets whether to instruct the server to close the connection after it has sent its response.
     * 
     * @param close whether to instruct the server to close the connection after it has sent its response
     */
    public void setConnectionCloseAfterResponse(final boolean close) {
        connectionCloseAfterResponse = close;
    }
    
    /**
     * Gets period after inactivity after which persistent
     * connections must be checked to ensure they are still valid.
     * 
     * @return the duration value
     */
    @Nullable Duration getValidateAfterInactivity() {
        return validateAfterInactivity;
    }

    /**
     * Sets period after inactivity after which persistent
     * connections must be checked to ensure they are still valid.
     * 
     * @param duration the duration value
     */
    public void setValidateAfterInactivity(@Nullable final Duration duration) {
        validateAfterInactivity = duration;
    }

    /**
     * Gets the hostname of the default proxy used when making connection. A null indicates no default proxy.
     * 
     * @return hostname of the default proxy used when making connection
     */
    @Nullable public String getConnectionProxyHost() {
        return connectionProxyHost;
    }

    /**
     * Sets the hostname of the default proxy used when making connection. A null indicates no default proxy.
     * 
     * @param host hostname of the default proxy used when making connection
     */
    public void setConnectionProxyHost(@Nullable final String host) {
        connectionProxyHost = StringSupport.trimOrNull(host);
    }

    /**
     * Gets the port of the default proxy used when making connection.
     * 
     * @return port of the default proxy used when making connection
     */
    public int getConnectionProxyPort() {
        return connectionProxyPort;
    }

    /**
     * Sets the port of the default proxy used when making connection.
     * 
     * @param port port of the default proxy used when making connection; must be greater than 0 and less than 65536
     */
    public void setConnectionProxyPort(final int port) {
        connectionProxyPort =
                (int) Constraint.numberInRangeExclusive(0, 65536, port,
                        "Proxy port must be between 0 and 65536, exclusive");
    }

    /**
     * Gets the username to use when authenticating to the proxy.
     * 
     * @return username to use when authenticating to the proxy
     */
    @Nullable public String getConnectionProxyUsername() {
        return connectionProxyUsername;
    }

    /**
     * Sets the username to use when authenticating to the proxy.
     * 
     * @param usename username to use when authenticating to the proxy; may be null
     */
    public void setConnectionProxyUsername(@Nullable final String usename) {
        connectionProxyUsername = usename;
    }

    /**
     * Gets the password used when authenticating to the proxy.
     * 
     * @return password used when authenticating to the proxy
     */
    @Nullable public String getConnectionProxyPassword() {
        return connectionProxyPassword;
    }

    /**
     * Sets the password used when authenticating to the proxy.
     * 
     * @param password password used when authenticating to the proxy; may be null
     */
    public void setConnectionProxyPassword(@Nullable final String password) {
        connectionProxyPassword = password;
    }

    /**
     * Gets whether HTTP redirects will be followed.
     * 
     * @return whether HTTP redirects will be followed
     */
    public boolean isHttpFollowRedirects() {
        return httpFollowRedirects;
    }

    /**
     * Gets whether HTTP redirects will be followed.
     * 
     * @param followRedirects true if redirects are followed, false otherwise
     */
    public void setHttpFollowRedirects(final boolean followRedirects) {
        httpFollowRedirects = followRedirects;
    }

    /**
     * Gets the character set used with the HTTP entity (body).
     * 
     * @return character set used with the HTTP entity (body)
     */
    @Nullable public String getHttpContentCharSet() {
        return httpContentCharSet;
    }

    /**
     * Sets the character set used with the HTTP entity (body).
     * 
     * @param charSet character set used with the HTTP entity (body)
     */
    public void setHttpContentCharSet(@Nullable final String charSet) {
        httpContentCharSet = charSet;
    }

    /**
     * Gets user agent.
     * 
     * @return The user agent.
     */
    @Nullable public String getUserAgent() {
        return userAgent;
    }

    /**
     * Sets user agent.
     * 
     * @param what what to set.  If this is null Apache will use the default.
     */
    public void setUserAgent(@Nullable final String what) {
        userAgent = what;
    }

    /**
     * Get the strategy which determines whether and how a retry should be attempted.
     * 
     * @return strategy which determines if a request should be retried
     */
    @Nullable public HttpRequestRetryStrategy getHttpRequestRetryStrategy() {
        return retryStrategy;
    }

    /**
     * Set the strategy which determines whether and how a retry should be attempted.
     * 
     * @param strategy handler which determines if a request should be retried
     */
    public void setHttpRequestRetryStrategy(@Nullable final HttpRequestRetryStrategy strategy) {
        retryStrategy = strategy;
    }
    
    /**
     * Get the resolver for port based on a scheme.
     * 
     * @return the resolver, or null
     */
    @Nullable public SchemePortResolver getSchemePortResolver() {
        return schemePortResolver;
    }
    
    /**
     * Set the resolver for port based on a scheme.
     * 
     * @param resolver the resolver, or null
     */
    public void setSchemePortResolver(@Nullable final SchemePortResolver resolver) {
        schemePortResolver = resolver;
    }
    
    /** 
     * Get the flag for disabling auth caching.
     * 
     * @return true if disabled, false if not
     */
    public boolean isDisableAuthCaching() {
        return disableAuthCaching;
    }

    /** 
     * Set the flag for disabling auth caching.
     * 
     * @param flag true if disabled, false if not
     */
    public void setDisableAuthCaching(final boolean flag) {
        disableAuthCaching = flag;
    }

    /** 
     * Get the flag for disabling automatic retries.
     * 
     * @return true if disabled, false if not
     */
    public boolean isDisableAutomaticRetries() {
        return disableAutomaticRetries;
    }

    /** 
     * Set the flag for disabling automatic retries.
     * 
     * @param flag true if disabled, false if not
     */
    public void setDisableAutomaticRetries(final boolean flag) {
        disableAutomaticRetries = flag;
    }

    /** 
     * Get the flag for disabling connection state.
     * 
     * @return true if disabled, false if not
     */
    public boolean isDisableConnectionState() {
        return disableConnectionState;
    }
    
    /** 
     * Set the flag for disabling connection state.
     * 
     * @param flag true if disabled, false if not
     */
 
    public void setDisableConnectionState(final boolean flag) {
        disableConnectionState = flag;
    }

    /** 
     * Get the flag for disabling content compression.
     * 
     * @return true if disabled, false if not
     */
    public boolean isDisableContentCompression() {
        return disableContentCompression;
    }

    /** 
     * Set the flag for disabling content compression.
     * 
     * @param flag true if disabled, false if not
     */
    public void setDisableContentCompression(final boolean flag) {
        disableContentCompression = flag;
    }

    /** 
     * Get the flag for disabling cookie management.
     * 
     * @return true if disabled, false if not
     */
    public boolean isDisableCookieManagement() {
        return disableCookieManagement;
    }

    /** 
     * Set the flag for disabling cookie management.
     * 
     * @param flag true if disabled, false if not
     */
    public void setDisableCookieManagement(final boolean flag) {
        disableCookieManagement = flag;
    }

    /** 
     * Get the flag for disabling redirect handling.
     * 
     * @return true if disabled, false if not
     */
    public boolean isDisableRedirectHandling() {
        return disableRedirectHandling;
    }

    /** 
     * Set the flag for disabling redirect handling.
     * 
     * @param flag true if disabled, false if not
     */
    public void setDisableRedirectHandling(final boolean flag) {
        disableRedirectHandling = flag;
    }

    /**
     * Get the flag enabling use of system properties.
     * 
     * @return true if enabled, false if not
     */
    public boolean isUseSystemProperties() {
        return useSystemProperties;
    }

    /**
     * Set the flag enabling use of system properties.
     * 
     * @param flag true if enabled, false if not
     */
    public void setUseSystemProperties(final boolean flag) {
        useSystemProperties = flag;
    }

    /**
     * Get the flag for evicting expired connections from the connection pool using a background thread. 
     * 
     * @return true if enabled, false if not
     */
    public boolean isEvictExpiredConnections() {
        return evictExpiredConnections;
    }

    /**
     * Set the flag for evicting expired connections from the connection pool using a background thread. 
     * 
     * @param flag true if enabled, false if not
     */
    public void setEvictExpiredConnections(final boolean flag) {
        evictExpiredConnections = flag;
    }

    /**
     * Get the flag for evicting idle connections from the connection pool using a background thread. 
     * 
     * @return true if enabled, false if not
     */
    public boolean isEvictIdleConnections() {
        return evictIdleConnections;
    }

    /**
     * Set the flag for evicting idle connections from the connection pool using a background thread. 
     * 
     * @param flag true if enabled, false if not
     */
    public void setEvictIdleConnections(final boolean flag) {
        evictIdleConnections = flag;
    }

    /**
     * Get the max idle time allowed for an idle connection before it is evicted. 
     * 
     * @return max idle time
     */
    @Nonnull Duration getConnectionMaxIdleTime() {
        return connectionMaxIdleTime;
    }

    /**
     * Set the max idle time allowed for an idle connection before it is evicted. 
     * 
     * @param duration the max idle time
     */
    public void setConnectionMaxIdleTime(@Nonnull final Duration duration) {
        Constraint.isNotNull(duration, "Connection max idle time cannot be null");
        Constraint.isLessThanOrEqual(Integer.MAX_VALUE, duration.toMillis(), "Connection max idle time too large");

        connectionMaxIdleTime = duration;
    }

    /**
     * Get the list of request interceptors to add first.
     * 
     * @return the list of interceptors
     */
    @Nonnull @NotLive @Unmodifiable public List<HttpRequestInterceptor> getFirstRequestInterceptors() {
        return requestInterceptorsFirst;
    }
    
    /**
     * Set the list of request interceptors to add first.
     * 
     * @param interceptors the list of interceptors, may be null
     */
    public void setFirstRequestInterceptors(@Nullable final List<HttpRequestInterceptor> interceptors) {
        if (interceptors != null) {
            requestInterceptorsFirst = CollectionSupport.copyToList(interceptors);
        } else {
            requestInterceptorsFirst = CollectionSupport.emptyList();
        }
    }

    /**
     * Get the list of request interceptors to add last.
     * 
     * @return the list of interceptors
     */
    @Nonnull @NotLive @Unmodifiable public List<HttpRequestInterceptor> getLastRequestInterceptors() {
        return requestInterceptorsLast;
    }

    /**
     * Set the list of request interceptors to add last.
     * 
     * @param interceptors the list of interceptors, may be null
     */
    public void setLastRequestInterceptors(@Nullable final List<HttpRequestInterceptor> interceptors) {
        if (interceptors != null) {
            requestInterceptorsLast = CollectionSupport.copyToList(interceptors);
        } else {
            requestInterceptorsLast = CollectionSupport.emptyList();
        }
    }

    /**
     * Get the list of response interceptors to add first.
     * 
     * @return the list of interceptors
     */
    @Nonnull @NotLive @Unmodifiable public List<HttpResponseInterceptor> getFirstResponseInterceptors() {
        return responseInterceptorsFirst;
    }

    /**
     * Set the list of response interceptors to add first.
     * 
     * @param interceptors the list of interceptors, may be null
     */
    public void setFirstResponseInterceptors(@Nullable final List<HttpResponseInterceptor> interceptors) {
        if (interceptors != null) {
            responseInterceptorsFirst = CollectionSupport.copyToList(interceptors);
        } else {
            responseInterceptorsFirst = CollectionSupport.emptyList();
        }
    }

    /**
     * Get the list of response interceptors to add last.
     * 
     * @return the list of interceptors
     */
    @Nonnull @NotLive @Unmodifiable public List<HttpResponseInterceptor> getLastResponseInterceptors() {
        return responseInterceptorsLast;
    }

    /**
     * Set the list of response interceptors to add last.
     * 
     * @param interceptors the list of interceptors, may be null
     */
    public void setLastResponseInterceptors(@Nullable final List<HttpResponseInterceptor> interceptors) {
        if (interceptors != null) {
            responseInterceptorsLast = CollectionSupport.copyToList(interceptors);
        } else {
            responseInterceptorsLast = CollectionSupport.emptyList();
        }
    }

    /**
     * Get the list of static {@link HttpClientContextHandler}.
     * 
     * @return the list of handlers
     */
    @Nonnull @NotLive @Unmodifiable public List<HttpClientContextHandler> getStaticContextHandlers() {
        return staticContextHandlers;
    }

    /**
     * Set the list of static {@link HttpClientContextHandler}.
     * 
     * @param handlers the list of handlers, may be null
     */
    public void setStaticContextHandlers(@Nullable final List<HttpClientContextHandler> handlers) {
        if (handlers != null) {
            staticContextHandlers = CollectionSupport.copyToList(handlers);
        } else {
            staticContextHandlers = CollectionSupport.emptyList();
        }
    }

    /**
     * Constructs an {@link HttpClient} using the settings of this builder.
     * 
     * @return the constructed client
     * 
     * @throws Exception if there is any problem building the new client instance
     */
    @Nonnull public HttpClient buildClient() throws Exception {
        decorateApacheBuilder();
        return new ContextHandlingHttpClient(getApacheBuilder().build(), getStaticContextHandlers());
    }

    /**
     * Decorate the Apache builder as determined by this builder's parameters. Subclasses will likely add additional
     * decoration.
     * 
     * @throws Exception if there is a problem decorating the Apache builder
     */
    protected void decorateApacheBuilder() throws Exception {
        final org.apache.hc.client5.http.impl.classic.HttpClientBuilder builder = getApacheBuilder();
        
        builder.setConnectionManager(buildConnectionManager());

        builder.setDefaultRequestConfig(buildDefaultRequestConfig());
        
        HttpHost proxyHost = null;
        if (connectionProxyHost != null) {
            proxyHost = new HttpHost(connectionProxyHost, connectionProxyPort);
            builder.setProxy(proxyHost);
        }

        builder.setRoutePlanner(buildRoutePlanner(proxyHost));

        builder.setDefaultCredentialsProvider(buildDefaultCredentialsProvider());
        
        handleFluentProperties(builder);
        
        if (connectionCloseAfterResponse) {
            if (!getFirstRequestInterceptors().stream().anyMatch(RequestConnectionClose.class::isInstance)
                    && !getLastRequestInterceptors().stream().anyMatch(RequestConnectionClose.class::isInstance)) {
                builder.addRequestInterceptorLast(new RequestConnectionClose());
            }
        }
        
        getFirstRequestInterceptors().forEach(builder::addRequestInterceptorFirst);
        getLastRequestInterceptors().forEach(builder::addRequestInterceptorLast);
        getFirstResponseInterceptors().forEach(builder::addResponseInterceptorFirst);
        getLastResponseInterceptors().forEach(builder::addResponseInterceptorLast);

        if (retryStrategy != null) {
            builder.setRetryStrategy(retryStrategy);
        }
 
        if (null != userAgent) {
            builder.setUserAgent(userAgent);
        }
    }
    
    /**
     * Build the instance of {@link HttpRoutePlanner}.
     * 
     * This is only necessary if we have a configured local address via
     * (@link {@link #getSocketLocalAddress()}).
     * 
     * @param proxyHost the proxy host, or null
     * @return the route planner instance, or null
     */
    @Nullable protected HttpRoutePlanner buildRoutePlanner(@Nullable final HttpHost proxyHost) {
        // By default we only need to build this if we have an explicitly-configured local address
        if (socketLocalAddress == null) {
            return null;
        }
        
        // This logic for the selection of the route planner impl to use is mirrored from
        // the Apache HttpClientBuilder#build().
        final SchemePortResolver resolver = schemePortResolver != null ?
                schemePortResolver : DefaultSchemePortResolver.INSTANCE;
        
        if (proxyHost != null) {
            return new LocalAddressProxyRoutePlanner(socketLocalAddress, proxyHost, resolver);
        } else if (isUseSystemProperties()) {
            return new LocalAddressSystemRoutePlanner(socketLocalAddress, resolver, ProxySelector.getDefault());
        } else {
            return new LocalAddressRoutePlanner(socketLocalAddress, resolver);
        }
    }

    /**
     * Handle the fluent (non-DI-friendly) builder properties.
     * 
     * @param builder the Apache HttpClientBuilder
     */
    protected void handleFluentProperties(
            @Nonnull final org.apache.hc.client5.http.impl.classic.HttpClientBuilder builder ) {
        // These boolean, interceptor, and eviction properties can otherwise only be supplied
        // to the Apache builder via a fluent-style API.
        
        if (isDisableAuthCaching()) {
            builder.disableAuthCaching();
        }

        if (isDisableAutomaticRetries()) {
            builder.disableAutomaticRetries();
        }

        if (isDisableConnectionState()) {
           builder.disableConnectionState();
        }

        if (isDisableContentCompression()) {
            builder.disableContentCompression();
        }

        if (isDisableCookieManagement()) {
            builder.disableCookieManagement();
        }

        if (isDisableRedirectHandling()) {
            builder.disableRedirectHandling();
        }

        if (isUseSystemProperties()) {
            builder.useSystemProperties();
        }
        
        if (isEvictExpiredConnections()) {
            builder.evictExpiredConnections();
        }

        if (isEvictIdleConnections()) {
            builder.evictIdleConnections(TimeValue.ofMilliseconds(connectionMaxIdleTime.toMillis()));
        }
    }
    
    /**
     * Build default {@link CredentialsProvider}.
     * 
     * @return the default credentials provider, or null
     */
    @Nullable protected CredentialsProvider buildDefaultCredentialsProvider() {
        if (connectionProxyHost != null && connectionProxyUsername != null && connectionProxyPassword != null) {
            // Note proxy HttpHost is set separately
            final char[] proxyPassBytes = connectionProxyPassword.toCharArray();
            return CredentialsProviderBuilder.create()
                    .add(new AuthScope(connectionProxyHost, connectionProxyPort),
                            new UsernamePasswordCredentials(connectionProxyUsername, proxyPassBytes))
                    .build();
        }
        
        return null;
    }
    
    /**
     * Build the default instance of {@link RequestConfig}
     * 
     * @return the request config instance
     */
    @Nonnull protected RequestConfig buildDefaultRequestConfig() {
        final RequestConfig.Builder requestConfigBuilder = RequestConfig.custom();

        if (!connectionRequestTimeout.isNegative()) {
            requestConfigBuilder.setConnectionRequestTimeout(
                    Timeout.ofMilliseconds(connectionRequestTimeout.toMillis()));
        }

        if (!responseTimeout.isNegative()) {
            requestConfigBuilder.setResponseTimeout(Timeout.ofMilliseconds(responseTimeout.toMillis()));
        }

        requestConfigBuilder.setRedirectsEnabled(httpFollowRedirects);
        
        return requestConfigBuilder.build();
    }

    /**
     * Build an instance of {@link HttpClientConnectionManager}
     * 
     * @return the connection manager instance
     */
    @Nonnull protected HttpClientConnectionManager buildConnectionManager() {
        final PoolingHttpClientConnectionManagerBuilder connMgrBuilder =
                PoolingHttpClientConnectionManagerBuilder.create();
        
        connMgrBuilder.setConnectionFactory(buildConnectionFactory());
        
        connMgrBuilder.setDefaultConnectionConfig(buildDefaultConnectionConfig());

        if (getTLSSocketFactory() != null) {
            connMgrBuilder.setSSLSocketFactory(getTLSSocketFactory());
        } else if (connectionDisregardTLSCertificate) {
            connMgrBuilder.setSSLSocketFactory(HttpClientSupport.buildNoTrustTLSSocketFactory());
        } else {
            connMgrBuilder.setSSLSocketFactory(HttpClientSupport.buildStrictTLSSocketFactory());
        }

        if (maxConnectionsTotal > 0) {
            connMgrBuilder.setMaxConnTotal(maxConnectionsTotal);
        }

        if (maxConnectionsPerRoute > 0) {
            connMgrBuilder.setMaxConnPerRoute(maxConnectionsPerRoute);
        }
        
        return connMgrBuilder.build();
    }

    /**
     * Build the default instance of {@link ConnectionConfig}.
     * 
     * @return the default connection config instance
     */
    @Nonnull protected ConnectionConfig buildDefaultConnectionConfig() {
        final ConnectionConfig.Builder builder = ConnectionConfig.custom();
        if (!connectionTimeout.isNegative()) {
            builder.setConnectTimeout(Timeout.ofMilliseconds(connectionTimeout.toMillis()));
        }

        if (!socketTimeout.isNegative()) {
            builder.setSocketTimeout(Timeout.ofMilliseconds(socketTimeout.toMillis()));
        }

        if (validateAfterInactivity != null) {
                builder.setValidateAfterInactivity(TimeValue.ofMilliseconds(validateAfterInactivity.toMillis()));
        }

        return builder.build();
    }

    /**
     * Build an instance of {@link HttpConnectionFactory}.
     * 
     * @return the connection factory instance
     */
    @Nonnull protected HttpConnectionFactory<ManagedHttpClientConnection> buildConnectionFactory() {
        final ManagedHttpClientConnectionFactory.Builder builder = ManagedHttpClientConnectionFactory.builder();

        builder.http1Config(Http1Config.custom()
                .setBufferSize(socketBufferSize)
                .build());

        if (httpContentCharSet != null) {
            builder.charCodingConfig(CharCodingConfig.custom()
                    .setCharset(Charset.forName(httpContentCharSet))
                    .build());
        }

        return builder.build();
    }

    /**
     * Get the Apache {@link org.apache.hc.client5.http.impl.classic.HttpClientBuilder} instance over which this builder
     * will be layered. Subclasses may override to return a specialized subclass.
     * 
     * @return the Apache HttpClientBuilder instance to use
     */
    @Nonnull org.apache.hc.client5.http.impl.classic.HttpClientBuilder getApacheBuilder() {
        return apacheBuilder;
    }

}
