/*
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *    http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package net.shibboleth.shared.httpclient;

import java.io.Closeable;
import java.io.IOException;
import java.util.List;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;

import org.apache.hc.client5.http.classic.HttpClient;
import org.apache.hc.client5.http.protocol.HttpClientContext;
import org.apache.hc.core5.http.ClassicHttpRequest;
import org.apache.hc.core5.http.ClassicHttpResponse;
import org.apache.hc.core5.http.HttpHost;
import org.apache.hc.core5.http.protocol.BasicHttpContext;
import org.apache.hc.core5.http.protocol.HttpContext;
import org.apache.hc.core5.io.CloseMode;
import org.apache.hc.core5.io.ModalCloseable;
import org.slf4j.Logger;

import com.google.common.collect.Lists;

import net.shibboleth.shared.collection.CollectionSupport;
import net.shibboleth.shared.collection.LazyList;
import net.shibboleth.shared.logic.Constraint;
import net.shibboleth.shared.primitive.LoggerFactory;

/**
 * A wrapper implementation of {@link HttpClient} which invokes supplied instances of {@link HttpClientContextHandler}
 * before and after request execution.
 * 
 * <p>
 * By definition the handlers will only be invoked for the {@link HttpClient} execute(...) method variants
 * which take an {@link HttpContext} argument.
 * </p>
 * 
 * <p>
 * The order of execution is:
 * </p>
 * 
 * <ol>
 * <li>Static handlers supplied via the constructor, in original list order</li>
 * <li>Dynamic handlers from the context attribute {@link HttpClientSupport#CONTEXT_KEY_DYNAMIC_CONTEXT_HANDLERS},
 *     in original list order</li>
 * <li>the wrapped client's corresponding execute(...) method</li>
 * <li>Dynamic handlers from the context attribute {@link HttpClientSupport#CONTEXT_KEY_DYNAMIC_CONTEXT_HANDLERS},
 *     in reverse list order</li>
 * <li>Static handlers supplied via the constructor, in reverse list order</li>
 * </ol>
 */
class ContextHandlingHttpClient extends AbstractHttpClient {
    
    /** Logger. */
    @Nonnull private Logger log = LoggerFactory.getLogger(ContextHandlingHttpClient.class);
    
    /** The wrapped client instance. */
    @Nonnull private HttpClient httpClient;
    
    /** Optional list of static handlers supplied to this class instance. */
    @Nonnull private List<HttpClientContextHandler> handlers;
    
    /**
     * Constructor.
     *
     * @param client the wrapped client instance
     */
    public ContextHandlingHttpClient(@Nonnull final HttpClient client) {
        this(client, CollectionSupport.emptyList());
    }

    /**
     * Constructor.
     *
     * @param client the wrapped client instance
     * @param staticHandlers the list of static handlers
     */
    public ContextHandlingHttpClient(@Nonnull final HttpClient client, 
            @Nonnull final List<HttpClientContextHandler> staticHandlers) {
        httpClient = Constraint.isNotNull(client, "HttpClient was null");
        handlers = Constraint.isNotNull(staticHandlers, "staticHandlers was null");
    }
    
    /** {@inheritDoc} */
    @Override
    public void close() throws IOException {
        if (Closeable.class.isInstance(httpClient)) {
            Closeable.class.cast(httpClient).close();
        }
    }
    
    /** {@inheritDoc} */
    @Override
    public void close(final CloseMode closeMode) {
        if (ModalCloseable.class.isInstance(httpClient)) {
            ModalCloseable.class.cast(httpClient).close(closeMode);
        }
    }

    /** {@inheritDoc} */
    @Override
    protected ClassicHttpResponse doExecute(@Nullable final HttpHost target, @Nonnull final ClassicHttpRequest request,
            @Nullable final HttpContext context) throws IOException {
        
        Throwable error = null;
        
        final HttpClientContext clientContext = 
                HttpClientContext.adapt(context != null ? context : new BasicHttpContext());

        assert clientContext != null;

        try {
            invokeBefore(request, clientContext);
            // As of 5.2, using executeOpen(...) b/c all other execute(...) variants are deprecated.
            final ClassicHttpResponse classicResponse = httpClient.executeOpen(target, request, clientContext);
            return classicResponse;
        } catch (final Throwable t) {
            error = t;
            throw t;
        } finally {
            invokeAfter(request, clientContext, error);
        }
    }

    /**
     * Invoke {@link HttpClientContextHandler#invokeBefore(HttpClientContext, ClassicHttpRequest)}
     * for supplied handlers.
     * 
     * @param request the HTTP request
     * @param context the HTTP context
     * @throws IOException if any handler throws an error
     */
    private void invokeBefore(@Nonnull final ClassicHttpRequest request, @Nonnull final HttpClientContext context)
            throws IOException {
        log.trace("In invokeBefore");
        
        final List<Throwable> errors = new LazyList<>();

        for (final HttpClientContextHandler handler : handlers) {
            try {
                log.trace("Invoking static handler invokeBefore: {}", handler.getClass().getName());
                handler.invokeBefore(context, request);
            } catch (final Throwable t) {
                log.warn("Static handler invokeBefore threw: {}", handler.getClass().getName(), t);
                errors.add(t);
            }
        }

        for (final HttpClientContextHandler handler : HttpClientSupport.getDynamicContextHandlerList(context)) {
            try {
                log.trace("Invoking dynamic handler invokeBefore: {}", handler.getClass().getName());
                handler.invokeBefore(context, request);
            } catch (final Throwable t) {
                log.warn("Dynamic handler invokeBefore threw: {}", handler.getClass().getName(), t);
                errors.add(t);
            }
        }
        
        final IOException exception = processHandlerErrors("Invoke Before", errors);
        if (exception != null) {
            throw exception;
        }
        
    }

    /**
     * Invoke {@link HttpClientContextHandler#invokeAfter(HttpClientContext, ClassicHttpRequest)}
     * for all supplied handlers.
     * 
     * @param request the HTTP request
     * @param context the HTTP context
     * @param priorError an error thrown by by either {@link #invokeBefore(ClassicHttpRequest, HttpClientContext)}
     *          or by HttpClient execute(...).
     * 
     * @throws IOException if any handler throws an error, or if priorError is an IOException. If priorError
     *                     is a type of unchecked error (RuntimeException or Error) that will be propagated out
     *                     here as well.
     */
    private void invokeAfter(@Nonnull final ClassicHttpRequest request, @Nonnull final HttpClientContext context, 
            @Nullable final Throwable priorError) throws IOException {
        log.trace("In invokeAfter");
        
        final List<Throwable> errors = new LazyList<>();
            
        for (final HttpClientContextHandler handler 
                : Lists.reverse(HttpClientSupport.getDynamicContextHandlerList(context))) {
            if (handler != null) {
                try {
                    log.trace("Invoking dynamic handler invokeAfter: {}", handler.getClass().getName());
                    handler.invokeAfter(context, request);
                } catch (final Throwable t) {
                    log.warn("Dynamic handler invokeAfter threw: {}", handler.getClass().getName(), t);
                    errors.add(t);
                }
            }
        }

        for (final HttpClientContextHandler handler : Lists.reverse(handlers)) {
            if (handler != null) {
                try {
                    log.trace("Invoking static handler invokeAfter: {}", handler.getClass().getName());
                    handler.invokeAfter(context, request);
                } catch (final Throwable t) {
                    log.warn("Static handler invokeAfter threw: {}", handler.getClass().getName(), t);
                    errors.add(t);
                }
            }
        }
        
        final IOException exception = processHandlerErrors("Invoke After", errors);
        processErrorsForInvokeAfter(exception, priorError);
            
    }
    
    /**
     * Process the error(s) seen during {@link #invokeBefore(ClassicHttpRequest, HttpClientContext)} 
     * or {@link #invokeAfter(ClassicHttpRequest, HttpClientContext, Throwable)}
     * into a single {@link IOException} that will be propagated out of that method.
     * 
     * @param stage the name of the stage, for reporting purposes
     * @param errors all errors seen during the method execution
     * 
     * @return the single exception to be propagated out, will be null if no errors present
     */
    @Nullable private IOException processHandlerErrors(final String stage, @Nonnull final List<Throwable> errors) {
        if (errors.isEmpty()) {
            return null;
        }
        
        if (errors.size() == 1) {
            final Throwable t = errors.get(0);
            if (IOException.class.isInstance(t)) {
                return IOException.class.cast(t);
            }
            return new IOException(
                    String.format("Context handler threw non-IOException Throwable in stage '%s'", stage), t);
        }
        
        final IOException e = new IOException(
                String.format("Multiple context handlers in stage '%s' reported error, see suppressed list", 
                        stage));
        for (final Throwable t : errors) {
            e.addSuppressed(t);
        }
        return e;
    }
    
    /**
     * Process errors for
     * {@link #invokeAfter(ClassicHttpRequest, HttpClientContext, Throwable)}.
     * 
     * @param invokeAfterException the exception thrown by invokeAfter handlers, if any
     * @param priorError an error thrown by by either {@link #invokeBefore(ClassicHttpRequest, HttpClientContext)}
     *          or by HttpClient execute(...), if any.
     * 
     * @throws IOException if invokeAfterException is non-null, or if priorError is an IOException. If priorError
     *                     is a type of unchecked error (RuntimeException or Error) that will be propagated out
     *                     here as well. 
     */
    private void processErrorsForInvokeAfter(@Nullable final IOException invokeAfterException,
            @Nullable final Throwable priorError)
            throws IOException {
        
        if (priorError != null) {
            if (invokeAfterException != null) {
                priorError.addSuppressed(invokeAfterException);
            }
            // Note: The RuntimeException and Error cases below can only occur from the HttpClient execute() method,
            // since a priorError from invokeBefore() is always processed into a checked IOException, and in that case
            // HttpClient execute() wasn't called at all.
            if (IOException.class.isInstance(priorError)) {
                throw IOException.class.cast(priorError);
            } else if (RuntimeException.class.isInstance(priorError)) {
                throw RuntimeException.class.cast(priorError);
            } else if (Error.class.isInstance(priorError)) {
                throw Error.class.cast(priorError);
            } else {
                // This would either be an actual instance of java.lang.Throwable itself (not a subclass), 
                // which really shouldn't ever happen,
                // or some unaccounted case in a future version of Java which adds additional unchecked base types.
                // For safety handle by converting to a RuntimeException.
                throw new RuntimeException(priorError);
            }
        } else if (invokeAfterException != null) {
            throw invokeAfterException;
        }
    }

}