/*
 * Licensed to the University Corporation for Advanced Internet Development,
 * Inc. (UCAID) under one or more contributor license agreements.  See the
 * NOTICE file distributed with this work for additional information regarding
 * copyright ownership. The UCAID licenses this file to You under the Apache
 * License, Version 2.0 (the "License"); you may not use this file except in
 * compliance with the License.  You may obtain a copy of the License at
 *
 *    http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package net.shibboleth.idp.plugin.oidc.op.security.jwt.claims.impl;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;

import org.opensaml.profile.context.ProfileRequestContext;
import org.opensaml.storage.RevocationCache;

import com.nimbusds.jwt.JWTClaimsSet;

import net.shibboleth.idp.plugin.oidc.op.token.support.TokenClaimsSet;
import net.shibboleth.oidc.jwt.claims.AbstractClaimsValidator;
import net.shibboleth.oidc.jwt.claims.JWTValidationException;
import net.shibboleth.utilities.java.support.annotation.constraint.NonnullAfterInit;
import net.shibboleth.utilities.java.support.annotation.constraint.NotEmpty;
import net.shibboleth.utilities.java.support.component.ComponentInitializationException;
import net.shibboleth.utilities.java.support.component.ComponentSupport;
import net.shibboleth.utilities.java.support.logic.Constraint;
import net.shibboleth.utilities.java.support.primitive.StringSupport;

/**
 * Verifies the root identifier ({@link TokenClaimsSet#KEY_ROOT_JTI} from the JWT against revocation via configurable
 * {@link RevocationCache}. If the root identifier is not found, JWT id is used.
 */
public class RootTokenIdRevocationValidator extends AbstractClaimsValidator {

    /** Message revocation cache instance to use. */
    @NonnullAfterInit private RevocationCache revocationCache;
    
    /** Context in revocation cache. */
    @NonnullAfterInit @NotEmpty private String context;
    
    /**
     * Set the revocation cache instance to use.
     * 
     * @param cache revocation cache to set
     */
    public void setRevocationCache(@Nonnull final RevocationCache cache) {
        ComponentSupport.ifInitializedThrowUnmodifiabledComponentException(this);
        revocationCache = Constraint.isNotNull(cache, "RevocationCache cannot be null");
    }
    
    /**
     * Set the revocation cache context that partitions entries.
     * 
     * @param ctx context value
     */
    public void setContext(@Nonnull @NotEmpty final String ctx) {
        ComponentSupport.ifInitializedThrowUnmodifiabledComponentException(this);
        context = Constraint.isNotNull(StringSupport.trimOrNull(ctx), "Context cannot be null or empty");
    }

    /** {@inheritDoc} */
    @Override
    protected void doInitialize() throws ComponentInitializationException {
        super.doInitialize();
        
        if (revocationCache == null) {
            throw new ComponentInitializationException("RevocationCache cannot be null");
        } else if (context == null) {
            throw new ComponentInitializationException("Context cannot be null");
        }
    }

    /** {@inheritDoc} */
    @Override
    protected void doValidate(@Nonnull final JWTClaimsSet claims,
            @Nullable final ProfileRequestContext profileRequestContext) throws JWTValidationException {

        final String rootJti = (String) claims.getClaim(TokenClaimsSet.KEY_ROOT_JTI);
        final String rootJtiToUse;
        if (StringSupport.trimOrNull(rootJti) == null) {
            rootJtiToUse = claims.getJWTID();
        } else {
            rootJtiToUse = rootJti;
        }

        if (StringSupport.trimOrNull(rootJtiToUse) == null) {
            throw new JWTValidationException("Claims set is missing required JWT identifier claim");
        }
        
        if (revocationCache.isRevoked(context, rootJtiToUse)) {
            throw new JWTValidationException("Claims set with root ID '" + rootJtiToUse + "' has been revoked");
        }
    }

}