/*
 * Licensed to the University Corporation for Advanced Internet Development,
 * Inc. (UCAID) under one or more contributor license agreements.  See the
 * NOTICE file distributed with this work for additional information regarding
 * copyright ownership. The UCAID licenses this file to You under the Apache
 * License, Version 2.0 (the "License"); you may not use this file except in
 * compliance with the License.  You may obtain a copy of the License at
 *
 *    http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package net.shibboleth.idp.plugin.oidc.op.profile.impl;

import java.util.Iterator;
import java.util.function.Function;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;

import org.opensaml.messaging.context.navigate.ChildContextLookup;
import org.opensaml.profile.context.ProfileRequestContext;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.nimbusds.oauth2.sdk.ResponseType;
import com.nimbusds.oauth2.sdk.Scope;
import com.nimbusds.openid.connect.sdk.OIDCScopeValue;

import net.shibboleth.idp.plugin.oidc.op.messaging.context.OIDCAuthenticationResponseTokenClaimsContext;
import net.shibboleth.idp.plugin.oidc.op.profile.context.navigate.ClientInfoScopeLookupFunction;
import net.shibboleth.idp.plugin.oidc.op.profile.context.navigate.DefaultOIDCMetadataContextLookupFunction;
import net.shibboleth.idp.plugin.oidc.op.profile.context.navigate.DefaultRequestResponseTypeLookupFunction;
import net.shibboleth.idp.plugin.oidc.op.profile.context.navigate.DefaultRequestedScopeLookupFunction;
import net.shibboleth.idp.plugin.oidc.op.profile.context.navigate.OIDCAuthenticationResponseContextLookupFunction;
import net.shibboleth.idp.profile.context.navigate.RelyingPartyIdLookupFunction;
import net.shibboleth.utilities.java.support.component.ComponentSupport;
import net.shibboleth.utilities.java.support.logic.Constraint;

/**
 * Action that validates requested and previously granted scopes are also registered in client metadata
 * and stores the resulting set in the response context.
 * 
 * <p>Requested scopes come from the inbound message and are possible but optional for both
 * authorization and token requests. They come from lookup functions aware of each message type.</p>
 * 
 * <p>Previously granted scopes are stored in the response context's slot for previous authorization
 * grant claims. In the case where no scopes are explicitly requested, we still filter the previous
 * grants against the metadata.</p>
 * 
 * <p>Explicitly requested scopes are also filtered against, and override, any scopes previously
 * validated as part of an authorization grant claim set. If this occurs, any grant-borne claims
 * are removed because the association to specific scopes is gone by this point.</p>
 * 
 * <p>The "offline_access" scope is ignored and stripped for the authentication endpoint unless the
 * response type includes "code".</p>
 */
public class ValidateScope extends AbstractOIDCAuthenticationResponseAction {

    /** Class logger. */
    @Nonnull private Logger log = LoggerFactory.getLogger(ValidateScope.class);

    /** Strategy used to obtain the relying party ID. */
    @Nonnull private Function<ProfileRequestContext,String> relyingPartyIdLookupStrategy;
    
    /** Strategy used to obtain the requested scope value. */
    @Nullable private Function<ProfileRequestContext,Scope> requestedScopeLookupStrategy;

    /** Strategy used to obtain the scope allowed for the client. */
    @Nonnull private Function<ProfileRequestContext,Scope> allowedScopeLookupStrategy;

    /** Strategy used to locate the {@link OIDCAuthenticationResponseTokenClaimsContext}. */
    @Nonnull
    private Function<ProfileRequestContext,OIDCAuthenticationResponseTokenClaimsContext>
    tokenClaimsContextLookupStrategy;
    
    /** Constructor. */
    public ValidateScope() {
        requestedScopeLookupStrategy = new DefaultRequestedScopeLookupFunction();
        relyingPartyIdLookupStrategy = new RelyingPartyIdLookupFunction();
        allowedScopeLookupStrategy = new ClientInfoScopeLookupFunction().compose(
                new DefaultOIDCMetadataContextLookupFunction());
        tokenClaimsContextLookupStrategy =
                new ChildContextLookup<>(OIDCAuthenticationResponseTokenClaimsContext.class).compose(
                        new OIDCAuthenticationResponseContextLookupFunction());
    }

    /**
     * Set the strategy used to obtain the relying party ID.
     * 
     * @param strategy lookup strategy
     */
    public void setRelyingPartyIdLookupStrategy(@Nonnull final Function<ProfileRequestContext,String> strategy) {
        ComponentSupport.ifInitializedThrowUnmodifiabledComponentException(this);
        
        relyingPartyIdLookupStrategy = Constraint.isNotNull(strategy,
                "Relying party ID lookup strategy cannot be null");
    }
    
    /**
     * Set the strategy used to locate the requested scope to validate.
     * 
     * @param strategy lookup strategy
     */
    public void setRequestedScopeLookupStrategy(@Nullable final Function<ProfileRequestContext,Scope> strategy) {
        ComponentSupport.ifInitializedThrowUnmodifiabledComponentException(this);
        
        requestedScopeLookupStrategy = strategy;
    }


    /**
     * Set the strategy used to locate the allowed scope for the client.
     * 
     * @param strategy lookup strategy
     */
    public void setAllowedScopeLookupStrategy(@Nonnull final Function<ProfileRequestContext,Scope> strategy) {
        ComponentSupport.ifInitializedThrowUnmodifiabledComponentException(this);
        
        allowedScopeLookupStrategy = Constraint.isNotNull(strategy,
                "Allowed scope lookyp strategy cannot be null");
    }
    
    /**
     * Set the strategy used to locate the {@link OIDCAuthenticationResponseTokenClaimsContext} associated with a given
     * {@link ProfileRequestContext}.
     * 
     * @param strategy lookup strategy
     */
    public void setOIDCAuthenticationResponseTokenClaimsContextLookupStrategy(
            @Nonnull final Function<ProfileRequestContext,OIDCAuthenticationResponseTokenClaimsContext> strategy) {
        ComponentSupport.ifInitializedThrowUnmodifiabledComponentException(this);
        
        tokenClaimsContextLookupStrategy = Constraint.isNotNull(strategy,
                "OIDCAuthenticationResponseTokenClaimsContextt lookup strategy cannot be null");
    }
    

// Checkstyle: CyclomaticComplexity OFF
    /** {@inheritDoc} */
    @Override
    protected void doExecute(@Nonnull final ProfileRequestContext profileRequestContext) {
        
        final String clientId = relyingPartyIdLookupStrategy.apply(profileRequestContext);
        
        // These typically come from metadata but may be supplemented or substituted from elsewhere.
        final Scope allowedScopes = allowedScopeLookupStrategy.apply(profileRequestContext);
        if (allowedScopes == null || allowedScopes.isEmpty()) {
            log.debug("{} No allowed scope for client {}, nothing to do", getLogPrefix(), clientId);
            return;
        }
        
        // These come from a previous authorization grant (authz code or access/refresh token).
        Scope previouslyGrantedScopes = null;
        if (getOidcResponseContext().getAuthorizationGrantClaimsSet() != null) {
            previouslyGrantedScopes = getOidcResponseContext().getAuthorizationGrantClaimsSet().getScope();
        }

        // These come from a request object or parameter. Absent by definition on the UserInfo endpoint.
        Scope requestedScopes = requestedScopeLookupStrategy != null ?
                requestedScopeLookupStrategy.apply(profileRequestContext) : null;
        if (requestedScopes == null) {
            // With none requested, simply swap requested for previously granted, if any.
            // Set previous set to null since there's no need to filter against it.
            requestedScopes = previouslyGrantedScopes;
            previouslyGrantedScopes = null;
        }
        
        boolean reducedRequestedScopes = false;
        
        for (Iterator<Scope.Value> i = requestedScopes.iterator(); i.hasNext();) {
            final Scope.Value scope = i.next();
            if (!allowedScopes.contains(scope)) {
                log.warn("{} Removing requested but unregistered scope {} for RP {}", getLogPrefix(), scope.getValue(),
                        clientId);
                i.remove();
            } else if (previouslyGrantedScopes != null && !previouslyGrantedScopes.contains(scope)) {
                log.warn("{} Removing requested but previously ungranted scope {} for RP {}", getLogPrefix(),
                        scope.getValue(), clientId);
                i.remove();
                reducedRequestedScopes = true;
            }
        }
        
        if (requestedScopes.contains(OIDCScopeValue.OFFLINE_ACCESS)) {
            // DefaultRequestResponseTypeLookupFunction returns response type only on authorization end point.
            // It is enough to remove offline_scope in this first validation turn.
            final ResponseType responseType =
                    new DefaultRequestResponseTypeLookupFunction().apply(profileRequestContext);
            if (responseType != null && !responseType.contains(ResponseType.Value.CODE)) {
                requestedScopes.remove(OIDCScopeValue.OFFLINE_ACCESS);
            }
        }
        
        if (!requestedScopes.isEmpty()) {
            getOidcResponseContext().setScope(requestedScopes);
        }
        
        if (reducedRequestedScopes) {
            final OIDCAuthenticationResponseTokenClaimsContext tokenClaimsCtx =
                    tokenClaimsContextLookupStrategy.apply(profileRequestContext);
            if (tokenClaimsCtx != null) {
                log.debug("{} Removing grant-encoded attributes due to reduction of requested scopes",
                        getLogPrefix());
                tokenClaimsCtx.getParent().removeSubcontext(tokenClaimsCtx);
            }
        }
    }
// Checkstyle: CyclomaticComplexity ON
    
}