/*
 * Licensed to the University Corporation for Advanced Internet Development,
 * Inc. (UCAID) under one or more contributor license agreements.  See the
 * NOTICE file distributed with this work for additional information regarding
 * copyright ownership. The UCAID licenses this file to You under the Apache
 * License, Version 2.0 (the "License"); you may not use this file except in
 * compliance with the License.  You may obtain a copy of the License at
 *
 *    http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package net.shibboleth.idp.plugin.oidc.op.profile.impl;

import java.text.ParseException;
import java.util.function.Function;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;

import org.opensaml.messaging.context.navigate.ChildContextLookup;
import org.opensaml.profile.action.ActionSupport;
import org.opensaml.profile.context.ProfileRequestContext;
import org.opensaml.xmlsec.context.SecurityParametersContext;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.nimbusds.jose.JWSAlgorithm;
import com.nimbusds.jwt.JWT;
import com.nimbusds.jwt.JWTClaimsSet;
import com.nimbusds.jwt.PlainJWT;
import com.nimbusds.jwt.SignedJWT;
import com.nimbusds.oauth2.sdk.ResponseType;
import com.nimbusds.oauth2.sdk.id.ClientID;

import net.shibboleth.oidc.jwt.claims.ClaimsValidator;
import net.shibboleth.oidc.jwt.claims.JWTValidationException;
import net.shibboleth.oidc.profile.core.OidcEventIds;
import net.shibboleth.oidc.security.impl.JWTSignatureValidationUtil;
import net.shibboleth.oidc.security.impl.OIDCSignatureValidationParameters;
import net.shibboleth.utilities.java.support.annotation.constraint.NonnullAfterInit;
import net.shibboleth.utilities.java.support.component.ComponentInitializationException;
import net.shibboleth.utilities.java.support.component.ComponentSupport;
import net.shibboleth.utilities.java.support.logic.Constraint;

/**
 * Action validates request object in response context.
 */
public class ValidateRequestObject extends AbstractOIDCAuthenticationResponseAction {

    /** Class logger. */
    @Nonnull private Logger log = LoggerFactory.getLogger(ValidateRequestObject.class);

    /** The signature validation parameters. */
    @Nullable private OIDCSignatureValidationParameters signatureValidationParameters;

    /**
     * Strategy used to locate the {@link SecurityParametersContext} to use for signing.
     */
    @Nonnull private Function<ProfileRequestContext, SecurityParametersContext> securityParametersLookupStrategy;

    /** Request Object. */
    @Nullable private JWT requestObject;

    /** The claims validator to be applied for validating the signed request object. */
    @NonnullAfterInit private ClaimsValidator signedClaimsValidator;

    /** The claims validator to be applied for validating the plain/unsigned request object. */
    @NonnullAfterInit private ClaimsValidator plainClaimsValidator;

    /** Constructor. */
    public ValidateRequestObject() {
        securityParametersLookupStrategy = new ChildContextLookup<>(SecurityParametersContext.class);
    }

    /**
     * Set the strategy used to locate the {@link SecurityParametersContext} to use.
     * 
     * @param strategy lookup strategy
     */
    public void setSecurityParametersLookupStrategy(
            @Nonnull final Function<ProfileRequestContext, SecurityParametersContext> strategy) {
        ComponentSupport.ifInitializedThrowUnmodifiabledComponentException(this);

        securityParametersLookupStrategy =
                Constraint.isNotNull(strategy, "SecurityParameterContext lookup strategy cannot be null");
    }

    /**
     * Set the claims validator used for validating the signed request object.
     * 
     * @param validator What to set
     */
    public void setSignedClaimsValidator(@Nonnull final ClaimsValidator validator) {
        ComponentSupport.ifInitializedThrowUnmodifiabledComponentException(this);
        signedClaimsValidator = Constraint.isNotNull(validator, "Signed claims validator cannot be null");
    }

    /**
     * Set the claims validator used for validating the plain/unsigned request object.
     * 
     * @param validator What to set
     */
    public void setPlainClaimsValidator(@Nonnull final ClaimsValidator validator) {
        ComponentSupport.ifInitializedThrowUnmodifiabledComponentException(this);
        plainClaimsValidator = Constraint.isNotNull(validator, "Plain claims validator cannot be null");
    }

    /** {@inheritDoc} */
    @Override
    protected void doInitialize() throws ComponentInitializationException {
        super.doInitialize();

        if (signedClaimsValidator == null) {
            throw new ComponentInitializationException("ClaimsValidator for signed requests cannot be null");
        }

        if (plainClaimsValidator == null) {
            throw new ComponentInitializationException("ClaimsValidator for plain requests cannot be null");
        }
}

    /** {@inheritDoc} */
    @Override
    protected boolean doPreExecute(@Nonnull final ProfileRequestContext profileRequestContext) {
        if (!super.doPreExecute(profileRequestContext)) {
            return false;
        }
        
        requestObject = getOidcResponseContext().getRequestObject();
        if (requestObject == null) {
            log.debug("{} No request object, nothing to do", getLogPrefix());
            return false;
        }
        return true;
    }
    
    // Checkstyle: CyclomaticComplexity OFF

    /** {@inheritDoc} */
    @Override
    protected void doExecute(@Nonnull final ProfileRequestContext profileRequestContext) {

        // We let "none" to be used only if nothing else has been registered.
        if (requestObject instanceof PlainJWT) {
            if (getMetadataContext() == null) {
                log.error("{} Request object unsigned, no client metadata", getLogPrefix());
                ActionSupport.buildEvent(profileRequestContext, OidcEventIds.INVALID_REQUEST_OBJECT);
                return;
            }
            final JWSAlgorithm requestObjectAlg =
                    getMetadataContext().getClientInformation().getOIDCMetadata().getRequestObjectJWSAlg();
            if (requestObjectAlg != null && !"none".equals(requestObjectAlg.getName())) {
                log.error("{} Request object is not signed, registered alg is {}", getLogPrefix(),
                        requestObjectAlg.getName());
                ActionSupport.buildEvent(profileRequestContext, OidcEventIds.INVALID_REQUEST_OBJECT);
                return;
            }
        }
        
        // Signature of signed request object must be verified
        if (!(requestObject instanceof PlainJWT)) {
            // Verify req object is signed with correct algorithm
            final SecurityParametersContext secParamCtx = securityParametersLookupStrategy.apply(profileRequestContext);
            final String errorEventId = JWTSignatureValidationUtil.validateSignature(secParamCtx,
                    (SignedJWT) requestObject, OidcEventIds.INVALID_REQUEST_OBJECT);
            if (errorEventId != null) {
                ActionSupport.buildEvent(profileRequestContext, errorEventId);
                return;
            }

        }
        final JWTClaimsSet claimsSet;
        // Validate still client_id and response_type values
        try {
            claimsSet = requestObject.getJWTClaimsSet();
            if (claimsSet.getClaims().containsKey("client_id")
                    && !getAuthenticationRequest().getClientID()
                            .equals(new ClientID((String) claimsSet.getClaim("client_id")))) {
                log.error("{} client_id in request object not matching client_id request parameter", getLogPrefix());
                ActionSupport.buildEvent(profileRequestContext, OidcEventIds.INVALID_REQUEST_OBJECT);
                return;
            }
            if (claimsSet.getClaims().containsKey("response_type")
                    && !getAuthenticationRequest().getResponseType().equals(new ResponseType(
                            ((String) requestObject.getJWTClaimsSet().getClaim("response_type")).split(" ")))) {
                log.error("{} response_type in request object not matching response_type request parameter",
                        getLogPrefix());
                ActionSupport.buildEvent(profileRequestContext, OidcEventIds.INVALID_REQUEST_OBJECT);
                return;
            }
        } catch (final ParseException e) {
            log.error("{} Unable to parse request object {}", getLogPrefix(), e.getMessage());
            ActionSupport.buildEvent(profileRequestContext, OidcEventIds.INVALID_REQUEST_OBJECT);
            return;
        }

        try {
            if (requestObject instanceof SignedJWT) {
                signedClaimsValidator.validate(claimsSet, profileRequestContext);
            } else {
                plainClaimsValidator.validate(claimsSet, profileRequestContext);
            }
        } catch (final JWTValidationException e) {
            log.warn("{} JWT validation failed: {}", getLogPrefix(), e.getMessage());
            ActionSupport.buildEvent(profileRequestContext, OidcEventIds.INVALID_REQUEST_OBJECT);
            return;
        }

    }    
    
    // Checkstyle: CyclomaticComplexity ON

}