/*
 * Licensed to the University Corporation for Advanced Internet Development,
 * Inc. (UCAID) under one or more contributor license agreements.  See the
 * NOTICE file distributed with this work for additional information regarding
 * copyright ownership. The UCAID licenses this file to You under the Apache
 * License, Version 2.0 (the "License"); you may not use this file except in
 * compliance with the License.  You may obtain a copy of the License at
 *
 *    http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package net.shibboleth.idp.plugin.oidc.op.profile.impl;

import java.io.IOException;
import java.net.URI;
import java.text.ParseException;
import java.util.Set;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;
import org.apache.http.HttpResponse;
import org.apache.http.HttpStatus;
import org.apache.http.client.HttpClient;
import org.apache.http.client.methods.HttpGet;
import org.apache.http.client.methods.HttpUriRequest;
import org.apache.http.client.protocol.HttpClientContext;
import org.apache.http.util.EntityUtils;
import org.opensaml.profile.action.ActionSupport;
import org.opensaml.profile.context.ProfileRequestContext;
import org.opensaml.security.httpclient.HttpClientSecurityParameters;
import org.opensaml.security.httpclient.HttpClientSecuritySupport;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import com.nimbusds.jwt.JWTParser;
import com.nimbusds.openid.connect.sdk.rp.OIDCClientMetadata;

import net.shibboleth.idp.plugin.oidc.op.messaging.context.OIDCAuthenticationResponseContext;
import net.shibboleth.oidc.profile.core.OidcEventIds;
import net.shibboleth.utilities.java.support.annotation.constraint.NonnullAfterInit;
import net.shibboleth.utilities.java.support.component.ComponentInitializationException;
import net.shibboleth.utilities.java.support.logic.Constraint;

/**
 * Action that stored request object to {@link OIDCAuthenticationResponseContext}. The request
 * object may be given by value with request parameter or by reference with request_uri parameter.
 */
public class SetRequestObjectToResponseContext extends AbstractOIDCAuthenticationResponseAction {

    /** Class logger. */
    @Nonnull private Logger log = LoggerFactory.getLogger(SetRequestObjectToResponseContext.class);
    
    /** HTTP Client used to post the data. */
    @NonnullAfterInit private HttpClient httpClient;

    /** HTTP client security parameters. */
    @Nullable private HttpClientSecurityParameters httpClientSecurityParameters;

    /**
     * Set the {@link HttpClient} to use.
     * 
     * @param client
     *            client to use
     */
    public void setHttpClient(@Nonnull final HttpClient client) {
        httpClient = Constraint.isNotNull(client, "HttpClient cannot be null");
    }

    /**
     * Set the optional client security parameters.
     * 
     * @param params
     *            the new client security parameters
     */
    public void setHttpClientSecurityParameters(@Nullable final HttpClientSecurityParameters params) {
        httpClientSecurityParameters = params;
    }

    /**
     * Build the {@link HttpClientContext} instance to be used by the HttpClient.
     * 
     * @param request
     *            the HTTP client request
     * @return the client context instance
     */
    @Nonnull
    private HttpClientContext buildHttpContext(@Nonnull final HttpUriRequest request) {
        final HttpClientContext clientContext = HttpClientContext.create();
        HttpClientSecuritySupport.marshalSecurityParameters(clientContext, httpClientSecurityParameters, false);
        HttpClientSecuritySupport.addDefaultTLSTrustEngineCriteria(clientContext, request);
        return clientContext;
    }
    
    /** {@inheritDoc} */
    @Override
    protected void doInitialize() throws ComponentInitializationException {
        super.doInitialize();
        Constraint.isNotNull(httpClient, "Httpclient cannot be null");
    }

    /** {@inheritDoc} */
    @Override
    protected boolean doPreExecute(@Nonnull final ProfileRequestContext profileRequestContext) {
        if (!super.doPreExecute(profileRequestContext)) {
            return false;
        }
        
        if (!getAuthenticationRequest().specifiesRequestObject()) {
            log.debug("{} No request_uri or request by value, nothing to do", getLogPrefix());
            return false;
        }
        
        if (getAuthenticationRequest().getRequestObject() != null
                && getAuthenticationRequest().getRequestURI() != null) {
            log.error("{} request_uri and request object cannot be both set", getLogPrefix());
            ActionSupport.buildEvent(profileRequestContext, OidcEventIds.REQUEST_OBJECT_AND_URI);
            return false;
        }
        
        return true;
    }

    /** {@inheritDoc} */
    @Override
    protected void doExecute(@Nonnull final ProfileRequestContext profileRequestContext) {
        if (getAuthenticationRequest().getRequestObject() != null) {
            getOidcResponseContext().setRequestObject(getAuthenticationRequest().getRequestObject());
            log.debug("{} Request object {} by value stored to oidc response context", getLogPrefix(),
                    getOidcResponseContext().getRequestObject().serialize());
            return;
        }
        
        // Request URI must be found in metadata.
        boolean authorized = false;
        if (getMetadataContext() != null && getMetadataContext().getClientInformation() != null) {
            final OIDCClientMetadata metadata = getMetadataContext().getClientInformation().getOIDCMetadata();
            if (metadata != null) {
                final Set<URI> allowedURIs = metadata.getRequestObjectURIs();
                if (allowedURIs != null) {
                    authorized = allowedURIs.contains(getAuthenticationRequest().getRequestURI());
                }
            }
        }
        
        if (!authorized) {
            log.error("{} Unregistered request URI blocked: {}", getLogPrefix(),
                    getAuthenticationRequest().getRequestURI());
            ActionSupport.buildEvent(profileRequestContext, OidcEventIds.INVALID_REQUEST_URI);
            return;
        }
        
        final HttpGet httpRequest = new HttpGet(getAuthenticationRequest().getRequestURI());
        final HttpClientContext httpContext = buildHttpContext(httpRequest);
        try {
            final HttpResponse response = httpClient.execute(httpRequest, httpContext);
            HttpClientSecuritySupport.checkTLSCredentialEvaluated(httpContext, httpRequest.getURI().getScheme());
            if (response.getStatusLine().getStatusCode() == HttpStatus.SC_OK) {
                final String requestObject = EntityUtils.toString(response.getEntity());
                try {
                    getOidcResponseContext().setRequestObject(JWTParser.parse(requestObject));
                    log.debug("{} Request object {} by reference stored to oidc response context", getLogPrefix(),
                            getOidcResponseContext().getRequestObject().serialize());
                    return;
                } catch (final ParseException e) {
                    log.error("{} Unable to parse request object from request_uri, {}", getLogPrefix(), e.getMessage());
                    ActionSupport.buildEvent(profileRequestContext, OidcEventIds.INVALID_REQUEST_URI);
                    return;
                }
            } else {
                log.error("{} Unable to get request object from request_uri, HTTP status {}", getLogPrefix(),
                        response.getStatusLine().getStatusCode());
                ActionSupport.buildEvent(profileRequestContext, OidcEventIds.INVALID_REQUEST_URI);
                return;
            }
        } catch (final IOException e) {
            log.error("{} Unable to get request object from request_uri, {}", getLogPrefix(), e.getMessage());
            ActionSupport.buildEvent(profileRequestContext, OidcEventIds.INVALID_REQUEST_URI);
            return;
        }
    }
}
