/*
 * Licensed to the University Corporation for Advanced Internet Development,
 * Inc. (UCAID) under one or more contributor license agreements.  See the
 * NOTICE file distributed with this work for additional information regarding
 * copyright ownership. The UCAID licenses this file to You under the Apache
 * License, Version 2.0 (the "License"); you may not use this file except in
 * compliance with the License.  You may obtain a copy of the License at
 *
 *    http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package net.shibboleth.idp.plugin.oidc.op.profile.impl;

import java.util.Collections;
import java.util.List;
import java.util.function.Function;
import java.util.function.Predicate;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;

import org.opensaml.profile.action.ActionSupport;
import org.opensaml.profile.action.EventIds;
import org.opensaml.profile.context.ProfileRequestContext;
import org.opensaml.saml.saml2.profile.context.EncryptionContext;
import org.opensaml.xmlsec.EncryptionConfiguration;
import org.opensaml.xmlsec.EncryptionParameters;
import org.opensaml.xmlsec.EncryptionParametersResolver;
import org.opensaml.xmlsec.SecurityConfigurationSupport;
import org.opensaml.xmlsec.criterion.EncryptionConfigurationCriterion;
import org.opensaml.xmlsec.criterion.EncryptionOptionalCriterion;

import net.shibboleth.idp.plugin.oidc.op.criterion.ClientInformationCriterion;
import net.shibboleth.idp.plugin.oidc.op.profile.context.navigate.DefaultOIDCMetadataContextLookupFunction;
import net.shibboleth.idp.profile.AbstractProfileAction;
import net.shibboleth.idp.profile.context.RelyingPartyContext;
import net.shibboleth.oidc.metadata.context.OIDCMetadataContext;
import net.shibboleth.oidc.profile.config.logic.EncryptionOptionalPredicate;
import net.shibboleth.utilities.java.support.annotation.constraint.NonnullAfterInit;
import net.shibboleth.utilities.java.support.annotation.constraint.NonnullElements;
import net.shibboleth.utilities.java.support.component.ComponentInitializationException;
import net.shibboleth.utilities.java.support.component.ComponentSupport;
import net.shibboleth.utilities.java.support.logic.Constraint;
import net.shibboleth.utilities.java.support.resolver.CriteriaSet;
import net.shibboleth.utilities.java.support.resolver.ResolverException;

import org.opensaml.messaging.context.navigate.ChildContextLookup;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * Action that resolves and populates {@link EncryptionParameters} on an {@link EncryptionContext} created/accessed via
 * a lookup function, by default on a {@link RelyingPartyContext} child of the profile request context.
 * 
 * The parameters are used either to encrypt id token / userinfo response or to decrypt request object. For the first
 * case the parameters are set with {@link EncryptionContext#setAssertionEncryptionParameters}, for request object
 * decryption the parameters are set with {@link EncryptionContext#setAttributeEncryptionParameters}. Yes, we are
 * stealing and bit misusing existing Shib context for our own almost similar purposes.
 * 
 * 
 * <p>
 * The OpenSAML default, per-RelyingParty, and default per-profile {@link EncryptionConfiguration} objects are input to
 * the resolution process, along with the relying party's oidc client registration data, which in most cases will be the
 * source of the eventual encryption key.
 * </p>
 * 
 */
public class PopulateOIDCEncryptionParameters extends AbstractProfileAction {

    /** Class logger. */
    @Nonnull private final Logger log = LoggerFactory.getLogger(PopulateOIDCEncryptionParameters.class);

    /** Whether we resolve encryption or decryption parameters. */
    private boolean forDecryption;
    
    /** Predicate to determine how to proceed if no encryption parameters are resolved. */
    @Nonnull private Predicate<ProfileRequestContext> encryptionOptionalPredicate;

    /** Strategy used to look up the {@link EncryptionContext} to store parameters in. */
    @Nonnull private Function<ProfileRequestContext,EncryptionContext> encryptionContextLookupStrategy;

    /** Strategy used to look up a per-request {@link EncryptionConfiguration} list. */
    @NonnullAfterInit private Function<ProfileRequestContext,List<EncryptionConfiguration>> configurationLookupStrategy;

    /** Resolver for parameters to store into context. */
    @NonnullAfterInit private EncryptionParametersResolver encParamsresolver;

    /** Active configurations to feed into resolver. */
    @Nullable @NonnullElements private List<EncryptionConfiguration> encryptionConfigurations;

    /** Strategy used to look up a OIDC metadata context. */
    @Nullable private Function<ProfileRequestContext,OIDCMetadataContext> oidcMetadataContextLookupStrategy;
    
    /** Whether encryption is optional. */
    private boolean encryptionOptional;
    
    /** Context to populate. */
    private EncryptionContext encryptionContext;

    /** Constructor. */
    public PopulateOIDCEncryptionParameters() {
        // Create context by default.
        oidcMetadataContextLookupStrategy = new DefaultOIDCMetadataContextLookupFunction();
        encryptionContextLookupStrategy = new ChildContextLookup<>(EncryptionContext.class, true).compose(
                new ChildContextLookup<>(RelyingPartyContext.class));
        encryptionOptionalPredicate = new EncryptionOptionalPredicate();
    }
    
    /**
     * Whether we resolve encryption or decryption parameters.
     * 
     * @param flag true if we should resolve decryption parameters.
     */
    public void setForDecryption(final boolean flag) {
        ComponentSupport.ifInitializedThrowUnmodifiabledComponentException(this);
        
        forDecryption = flag;
    }
   
    /**
     * Sets the condition to apply to determine how to proceed if encryption parameter resolution fails.
     * 
     * @param condition condition to set
     * 
     * @since 3.1.0
     */
    public void setEncryptionOptionalPredicate(@Nonnull final Predicate<ProfileRequestContext> condition) {
        ComponentSupport.ifInitializedThrowUnmodifiabledComponentException(this);
        
        encryptionOptionalPredicate = Constraint.isNotNull(condition, "Condition cannot be null");
    }

    /**
     * Set the strategy used to look up the {@link EncryptionContext} to set the flags for.
     * 
     * @param strategy lookup strategy
     */
    public void setEncryptionContextLookupStrategy(
            @Nonnull final Function<ProfileRequestContext, EncryptionContext> strategy) {
        ComponentSupport.ifInitializedThrowUnmodifiabledComponentException(this);

        encryptionContextLookupStrategy =
                Constraint.isNotNull(strategy, "EncryptionContext lookup strategy cannot be null");
    }

    /**
     * Set the strategy used to look up the {@link OIDCMetadataContext} to locate client registered encryption
     * parameters.
     * 
     * @param strategy lookup strategy
     */
    public void setOidcMetadataContextLookupStrategy(
            @Nonnull final Function<ProfileRequestContext, OIDCMetadataContext> strategy) {
        ComponentSupport.ifInitializedThrowUnmodifiabledComponentException(this);

        oidcMetadataContextLookupStrategy =
                Constraint.isNotNull(strategy, " OIDCMetadataContext lookup strategy cannot be null");
    }

    /**
     * Set the strategy used to look up a per-request {@link EncryptionConfiguration} list.
     * 
     * @param strategy lookup strategy
     */
    public void setConfigurationLookupStrategy(
            @Nonnull final Function<ProfileRequestContext, List<EncryptionConfiguration>> strategy) {
        ComponentSupport.ifInitializedThrowUnmodifiabledComponentException(this);

        configurationLookupStrategy =
                Constraint.isNotNull(strategy, "EncryptionConfiguration lookup strategy cannot be null");
    }

    /**
     * Set the encParamsresolver to use for the parameters to store into the context.
     * 
     * @param newResolver encParamsresolver to use
     */
    public void setEncryptionParametersResolver(@Nonnull final EncryptionParametersResolver newResolver) {
        ComponentSupport.ifInitializedThrowUnmodifiabledComponentException(this);

        encParamsresolver = Constraint.isNotNull(newResolver, "EncryptionParametersResolver cannot be null");
    }

    /** {@inheritDoc} */
    @Override
    protected void doInitialize() throws ComponentInitializationException {
        super.doInitialize();

        if (encParamsresolver == null) {
            throw new ComponentInitializationException("EncryptionParametersResolver cannot be null");
        } else if (configurationLookupStrategy == null) {
            configurationLookupStrategy = new Function<ProfileRequestContext, List<EncryptionConfiguration>>() {
                public List<EncryptionConfiguration> apply(final ProfileRequestContext input) {
                    return Collections.singletonList(SecurityConfigurationSupport.getGlobalEncryptionConfiguration());
                }
            };
        }
    }

    /** {@inheritDoc} */
    @Override
    protected boolean doPreExecute(@Nonnull final ProfileRequestContext profileRequestContext) {
        if (!super.doPreExecute(profileRequestContext)) {
            return false;
        }
        
        encryptionContext = encryptionContextLookupStrategy.apply(profileRequestContext);
        if (encryptionContext == null) {
            log.debug("{} No EncryptionContext returned by lookup strategy", getLogPrefix());
            ActionSupport.buildEvent(profileRequestContext, EventIds.INVALID_PROFILE_CTX);
            return false;
        }
        
        if (!forDecryption) {
            encryptionOptional = encryptionOptionalPredicate.test(profileRequestContext);
        }
        
        return true;
    }

// Checkstyle: CyclomaticComplexity OFF
    /** {@inheritDoc} */
    @Override
    protected void doExecute(@Nonnull final ProfileRequestContext profileRequestContext) {

        log.debug("{} Resolving EncryptionParameters for {}", getLogPrefix(),
                forDecryption ? "request object decryption" : "response encryption");
        
        try {
            encryptionConfigurations = configurationLookupStrategy.apply(profileRequestContext);
            if (encryptionConfigurations == null || encryptionConfigurations.isEmpty()) {
                throw new ResolverException("No EncryptionConfigurations returned by lookup strategy");
            }
            final CriteriaSet criteria = buildCriteriaSet(profileRequestContext);
            final EncryptionParameters params = encParamsresolver.resolveSingle(criteria);
            
            if (params != null) {
                log.debug("{} Resolved EncryptionParameters for {}", getLogPrefix(),
                        forDecryption ? "request object decryption" : "response encryption");
                if (forDecryption) {
                    // Decryption parameters for request object decryption
                    encryptionContext.setAttributeEncryptionParameters(params);
                } else {
                    // Indicates that id token or userinfo response should be encrypted
                    encryptionContext.setAssertionEncryptionParameters(params);
                }
            } else {
                if (encryptionOptional) {
                    log.debug("{} Resolver returned no EncryptionParameters", getLogPrefix());
                    log.debug("{} Encryption is optional, ignoring inability to encrypt", getLogPrefix());
                } else {
                    log.warn("{} Resolver returned no EncryptionParameters", getLogPrefix());
                    ActionSupport.buildEvent(profileRequestContext, EventIds.INVALID_SEC_CFG);
                }
            }
        } catch (final ResolverException e) {
            log.error("{} Error resolving EncryptionParameters", getLogPrefix(), e);
            if (encryptionOptional) {
                log.debug("{} Encryption is optional, ignoring inability to encrypt", getLogPrefix());
            } else {
                ActionSupport.buildEvent(profileRequestContext, EventIds.INVALID_SEC_CFG);
            }
        }
    }
// Checkstyle: CyclomaticComplexity ON
    
    /**
     * Build the criteria used as input to the {@link EncryptionParametersResolver}.
     * 
     * @param profileRequestContext current profile request context
     * 
     * @return the criteria set to use
     */
    @Nonnull
    private CriteriaSet buildCriteriaSet(@Nonnull final ProfileRequestContext profileRequestContext) {

        final CriteriaSet criteria = new CriteriaSet(new EncryptionConfigurationCriterion(encryptionConfigurations));
        
        if (!forDecryption) {
            criteria.add(new EncryptionOptionalCriterion(encryptionOptional));
        }
        
        final OIDCMetadataContext oidcMetadataCtx = oidcMetadataContextLookupStrategy.apply(profileRequestContext);
        if (oidcMetadataCtx != null && oidcMetadataCtx.getClientInformation() != null) {
            log.debug(
                    "{} Adding OIDC client information to resolution criteria for encryption algorithms",
                    getLogPrefix());
            criteria.add(new ClientInformationCriterion(oidcMetadataCtx.getClientInformation()));
        } else {
            log.debug("{} No OIDC client information available", getLogPrefix());
        }
        return criteria;
    }

}