/*
 * Licensed to the University Corporation for Advanced Internet Development,
 * Inc. (UCAID) under one or more contributor license agreements.  See the
 * NOTICE file distributed with this work for additional information regarding
 * copyright ownership. The UCAID licenses this file to You under the Apache
 * License, Version 2.0 (the "License"); you may not use this file except in
 * compliance with the License.  You may obtain a copy of the License at
 *
 *    http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package net.shibboleth.idp.plugin.oidc.op.profile.impl;

import java.util.function.Function;

import javax.annotation.Nonnull;

import net.shibboleth.idp.plugin.oidc.op.profile.context.navigate.DefaultClientIDLookupFunction;
import net.shibboleth.oidc.metadata.ClientInformationResolver;
import net.shibboleth.oidc.metadata.context.OIDCMetadataContext;
import net.shibboleth.oidc.metadata.criterion.ClientIDCriterion;
import net.shibboleth.utilities.java.support.annotation.constraint.NonnullAfterInit;
import net.shibboleth.utilities.java.support.component.ComponentInitializationException;
import net.shibboleth.utilities.java.support.component.ComponentSupport;
import net.shibboleth.utilities.java.support.logic.Constraint;
import net.shibboleth.utilities.java.support.resolver.CriteriaSet;
import net.shibboleth.utilities.java.support.resolver.ResolverException;
import org.opensaml.messaging.context.MessageContext;
import org.opensaml.messaging.handler.AbstractMessageHandler;
import org.opensaml.messaging.handler.MessageHandlerException;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.nimbusds.oauth2.sdk.id.ClientID;
import com.nimbusds.openid.connect.sdk.rp.OIDCClientInformation;

/**
 * Handler for inbound OIDC protocol messages that attempts to locate OIDC metadata for a rp, and attaches it with a
 * {@link OIDCMetadataContext} as a child of a pre-existing instance of {@link MessageContext}.
 */
public class OIDCMetadataLookupHandler extends AbstractMessageHandler {

    /** Logger. */
    @Nonnull private final Logger log = LoggerFactory.getLogger(OIDCMetadataLookupHandler.class);

    /** Resolver used to look up OIDC client information. */
    @NonnullAfterInit private ClientInformationResolver clientResolver;

    /** Strategy used to obtain the client id value for authorize/token request. */
    @Nonnull
    private Function<MessageContext,ClientID> clientIDLookupStrategy;

    /**
     * Constructor.
     */
    public OIDCMetadataLookupHandler() {
        clientIDLookupStrategy = new DefaultClientIDLookupFunction();
    }

    /**
     * Set the strategy used to locate the client id of the request.
     * 
     * @param strategy lookup strategy
     */
    public void setClientIDLookupStrategy(@Nonnull final Function<MessageContext, ClientID> strategy) {
        ComponentSupport.ifInitializedThrowUnmodifiabledComponentException(this);
        clientIDLookupStrategy =
                Constraint.isNotNull(strategy, "ClientIDLookupStrategy lookup strategy cannot be null");
    }

    /**
     * Set the {@link ClientInformationResolver} to use.
     * 
     * @param resolver The resolver to use.
     */
    public void setClientInformationResolver(@Nonnull final ClientInformationResolver resolver) {
        ComponentSupport.ifInitializedThrowUnmodifiabledComponentException(this);

        clientResolver = Constraint.isNotNull(resolver, "ClientInformationResolver cannot be null");
    }

    /** {@inheritDoc} */
    @Override
    protected void doInitialize() throws ComponentInitializationException {
        super.doInitialize();

        if (clientResolver == null) {
            throw new ComponentInitializationException("ClientInformationResolver cannot be null");
        }
    }

    /** {@inheritDoc} */
    @Override
    protected void doInvoke(@Nonnull final MessageContext messageContext) throws MessageHandlerException {
        ComponentSupport.ifNotInitializedThrowUninitializedComponentException(this);
        
        final ClientID clientId = clientIDLookupStrategy.apply(messageContext);
        if (clientId == null) {
            log.debug("{} No client ID available", getLogPrefix());
            return;
        }
        
        final CriteriaSet criteria = new CriteriaSet(new ClientIDCriterion(clientId));
        try {
            final OIDCClientInformation clientInformation = clientResolver.resolveSingle(criteria);
            if (clientInformation == null) {
                log.debug("{} No client information returned for {}", getLogPrefix(), clientId);
                return;
            }
            final OIDCMetadataContext oidcCtx = new OIDCMetadataContext();
            oidcCtx.setClientInformation(clientInformation);
            messageContext.addSubcontext(oidcCtx);
            // Based on that info we know 1) client is valid 2) we know valid
            // redirect uris
            log.debug("{} {} added to MessageContext as child of {}", getLogPrefix(),
                    OIDCMetadataContext.class.getName(), messageContext.getClass().getName());
        } catch (final ResolverException e) {
            log.error("{} ResolverException thrown during client information lookup", getLogPrefix(), e);
        }
    }
}