/*
 * Licensed to the University Corporation for Advanced Internet Development,
 * Inc. (UCAID) under one or more contributor license agreements.  See the
 * NOTICE file distributed with this work for additional information regarding
 * copyright ownership. The UCAID licenses this file to You under the Apache
 * License, Version 2.0 (the "License"); you may not use this file except in
 * compliance with the License.  You may obtain a copy of the License at
 *
 *    http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package net.shibboleth.idp.plugin.oidc.op.profile.impl;

import javax.annotation.Nonnull;

import org.opensaml.profile.context.ProfileRequestContext;

import com.nimbusds.jose.Payload;
import com.nimbusds.jwt.EncryptedJWT;
import com.nimbusds.jwt.SignedJWT;

import net.shibboleth.idp.plugin.oidc.op.messaging.context.OIDCAuthenticationResponseContext;
import net.shibboleth.idp.plugin.oidc.op.oauth2.profile.impl.AbstractEncryptTokenAction;

/**
 * Action that serves both id token and user info response encryption.
 * 
 * <p>Action assumes the content to be encrypted is located primarily
 * by {@link OIDCAuthenticationResponseContext#getProcessedToken()} returning either signed id token or signed user info
 * response. If such information is not available action assumes the data to be encrypted is
 * {@link OIDCAuthenticationResponseContext#getUserInfo()} containing bare user info response. If neither of the sources
 * for encryption exists the actions fails.</p>
 */
public class EncryptProcessedToken extends AbstractEncryptTokenAction {

    /** {@inheritDoc} */
    @Override
    protected Payload getPayload(@Nonnull final ProfileRequestContext profileRequestContext) {
        
        if (getOidcResponseContext().getProcessedToken() != null) {
            return new Payload((SignedJWT) getOidcResponseContext().getProcessedToken());
        } else if (getOidcResponseContext().getUserInfo() != null) {
            return new Payload(getOidcResponseContext().getUserInfo().toJSONObject());
        } else {
            return null;
        }
    }

    /** {@inheritDoc} */
    @Override
    protected void setProcessedToken(@Nonnull final ProfileRequestContext profileRequestContext,
            @Nonnull final EncryptedJWT token) {
        getOidcResponseContext().setProcessedToken(token);
    }

}