/*
 * Licensed to the University Corporation for Advanced Internet Development,
 * Inc. (UCAID) under one or more contributor license agreements.  See the
 * NOTICE file distributed with this work for additional information regarding
 * copyright ownership. The UCAID licenses this file to You under the Apache
 * License, Version 2.0 (the "License"); you may not use this file except in
 * compliance with the License.  You may obtain a copy of the License at
 *
 *    http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package net.shibboleth.idp.plugin.oidc.op.profile.impl;

import java.net.URI;
import java.util.function.Function;
import java.util.function.Predicate;

import javax.annotation.Nonnull;

import org.opensaml.profile.action.ActionSupport;
import org.opensaml.profile.context.EventContext;
import org.opensaml.profile.context.ProfileRequestContext;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.nimbusds.oauth2.sdk.ErrorObject;
import com.nimbusds.oauth2.sdk.ResponseMode;
import com.nimbusds.oauth2.sdk.ResponseType;
import com.nimbusds.oauth2.sdk.id.Issuer;
import com.nimbusds.openid.connect.sdk.AuthenticationErrorResponse;

import net.shibboleth.idp.plugin.oidc.op.profile.context.navigate.DefaultRequestResponseModeLookupFunction;
import net.shibboleth.idp.plugin.oidc.op.profile.context.navigate.DefaultRequestResponseTypeLookupFunction;
import net.shibboleth.idp.plugin.oidc.op.profile.context.navigate.DefaultRequestStateLookupFunction;
import net.shibboleth.idp.plugin.oidc.op.profile.context.navigate.ValidatedRedirectURILookupFunction;
import net.shibboleth.idp.profile.IdPEventIds;
import net.shibboleth.idp.profile.context.navigate.ResponderIdLookupFunction;
import net.shibboleth.oidc.profile.config.logic.IncludeIssuerInAuthenticationResponsePredicate;
import net.shibboleth.utilities.java.support.component.ComponentSupport;
import net.shibboleth.utilities.java.support.logic.Constraint;
import net.shibboleth.utilities.java.support.primitive.StringSupport;

/**
 * This action reads an event from the configured {@link EventContext} lookup strategy, constructs an OIDC
 * authentication error response message and attaches it as the outbound message.
 */
public class BuildAuthenticationErrorResponseFromEvent
        extends AbstractBuildErrorResponseFromEvent<AuthenticationErrorResponse> {

    /** Class logger. */
    @Nonnull private Logger log = LoggerFactory.getLogger(BuildAuthenticationErrorResponseFromEvent.class);

    /** Strategy used to obtain the response issuer value. */
    @Nonnull private Function<ProfileRequestContext, String> issuerLookupStrategy;

    /** Predicate to signal whether or not to include iss-parameter to the response. */
    @Nonnull private Predicate<ProfileRequestContext> includeIssuerInResponsePredicate;

    /** Issuer value to included in the response message, if configured to be included. */
    private Issuer issuer;

    /**
     * Constructor.
     */
    public BuildAuthenticationErrorResponseFromEvent() {
        issuerLookupStrategy = new ResponderIdLookupFunction();
        includeIssuerInResponsePredicate = new IncludeIssuerInAuthenticationResponsePredicate();
    }

    /**
     * Set the strategy used to locate the issuer value to use.
     * 
     * @param strategy what to set
     */
    public void setIssuerLookupStrategy(@Nonnull final Function<ProfileRequestContext, String> strategy) {
        ComponentSupport.ifInitializedThrowUnmodifiabledComponentException(this);
        
        issuerLookupStrategy = Constraint.isNotNull(strategy, "IssuerLookupStrategy lookup strategy cannot be null");
    }

    /**
     * Set the predicate to signal whether or not to include iss-parameter to the response.
     *
     * @param predicate what to set
     */
    public void setIncludeIssuerInResponsePredicate(@Nonnull final Predicate<ProfileRequestContext> predicate) {
        ComponentSupport.ifInitializedThrowUnmodifiabledComponentException(this);

        includeIssuerInResponsePredicate = Constraint.isNotNull(predicate,
                "Include issuer in response predicate cannot be null");
    }

    /** {@inheritDoc} */
    @Override
    protected boolean doPreExecute(@Nonnull final ProfileRequestContext profileRequestContext) {
        if (!super.doPreExecute(profileRequestContext)) {
            return false;
        }

        if (includeIssuerInResponsePredicate.test(profileRequestContext)) {
            final String issValue = issuerLookupStrategy.apply(profileRequestContext);
            if (StringSupport.trimOrNull(issValue) == null) {
                log.error("{} Could not resolve value for issuer even though it's required", getLogPrefix());
                ActionSupport.buildEvent(profileRequestContext, IdPEventIds.INVALID_RELYING_PARTY_CTX);
                return false;
            }
            issuer = new Issuer(issValue);
        } else {
            issuer = null;
        }
        return true;
    }

    /** {@inheritDoc} */
    @Override
    protected AuthenticationErrorResponse buildErrorResponse(final ErrorObject error,
            final ProfileRequestContext profileRequestContext) {
        final URI redirectURI = new ValidatedRedirectURILookupFunction().apply(profileRequestContext);
        if (redirectURI == null) {
            // No validated redirect uri to return reply to.
            return null;
        }
        final ResponseMode defaultResponseMode = 
                new DefaultRequestResponseModeLookupFunction().apply(profileRequestContext);
        final ResponseType responseType = new DefaultRequestResponseTypeLookupFunction().apply(profileRequestContext);
        final ResponseMode responseMode;
        if (responseType != null && (responseType.impliesHybridFlow() || responseType.impliesImplicitFlow())) {
            responseMode = defaultResponseMode == null ? ResponseMode.FRAGMENT :
                defaultResponseMode == ResponseMode.QUERY ? ResponseMode.FRAGMENT : defaultResponseMode;
        } else {
            responseMode = defaultResponseMode;
        }
        return new AuthenticationErrorResponse(redirectURI, error,
                new DefaultRequestStateLookupFunction().apply(profileRequestContext), issuer, responseMode);
    }

}