/*
 * Licensed to the University Corporation for Advanced Internet Development,
 * Inc. (UCAID) under one or more contributor license agreements.  See the
 * NOTICE file distributed with this work for additional information regarding
 * copyright ownership. The UCAID licenses this file to You under the Apache
 * License, Version 2.0 (the "License"); you may not use this file except in
 * compliance with the License.  You may obtain a copy of the License at
 *
 *    http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package net.shibboleth.idp.plugin.oidc.op.oauth2.profile.impl;

import java.time.Duration;
import java.util.function.Function;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;

import org.opensaml.profile.action.ActionSupport;
import org.opensaml.profile.action.EventIds;
import org.opensaml.profile.context.ProfileRequestContext;
import org.opensaml.storage.RevocationCache;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.nimbusds.jwt.JWTClaimsSet;

import net.shibboleth.idp.plugin.oidc.op.oauth2.messaging.context.OAuth2TokenMgmtResponseContext;
import net.shibboleth.idp.plugin.oidc.op.profile.logic.DefaultChainRevocationLifetimeLookupStrategy;
import net.shibboleth.idp.plugin.oidc.op.profile.logic.DefaultRootTokenIdentifierLookupStrategy;
import net.shibboleth.idp.plugin.oidc.op.profile.logic.DefaultTokenRevocationLifetimeLookupStrategy;
import net.shibboleth.idp.plugin.oidc.op.storage.RevocationCacheContexts;
import net.shibboleth.idp.profile.AbstractProfileAction;
import net.shibboleth.idp.profile.IdPEventIds;
import net.shibboleth.oidc.profile.config.navigate.RevocationMethodLookupFunction;
import net.shibboleth.oidc.profile.core.OidcEventIds;
import net.shibboleth.oidc.profile.oauth2.config.OAuth2TokenRevocationConfiguration.OAuth2TokenRevocationMethod;
import net.shibboleth.utilities.java.support.annotation.constraint.NonnullAfterInit;
import net.shibboleth.utilities.java.support.component.ComponentInitializationException;
import net.shibboleth.utilities.java.support.component.ComponentSupport;
import net.shibboleth.utilities.java.support.logic.Constraint;
import net.shibboleth.utilities.java.support.primitive.StringSupport;

/**
 * Action that revokes a single token or the full chain of tokens, depending on the result of the configured lookup
 * strategy for the revocation method. The full chain of tokens refer to the token that is fed to this action together
 * with all other tokens related to the same root token identifier. For legacy reasons, if the root token identifier
 * is not existing in the claims set but the full chain is to be revoked, the JWT identifier (jti) from the claims set
 * is used as the root token identifier.
 * 
 * @event {@link EventIds#PROCEED_EVENT_ID}
 * @event {@link OidcEventIds#REVOCATION_FAILED}
 */
public class RevokeToken extends AbstractProfileAction {

    /** Class logger. */
    @Nonnull private Logger log = LoggerFactory.getLogger(RevokeToken.class);

    /** Message revocation cache instance to use. */
    @NonnullAfterInit private RevocationCache revocationCache;

    /** 
     * Which revocation method should be used when revoking a token. 
     * Supported values are CHAIN and TOKEN. The default is CHAIN. 
     */
    @Nonnull private Function<ProfileRequestContext,OAuth2TokenRevocationMethod> revocationMethodLookupStrategy;

    /** Lookup function to supply chain revocation lifetime. */
    @Nonnull private Function<ProfileRequestContext,Duration> chainRevocationLifetimeLookupStrategy;

    /** Lookup function to supply token revocation lifetime. */
    @Nonnull private Function<JWTClaimsSet,Duration> tokenRevocationLifetimeLookupStrategy;

    /** Lookup function to supply root token identifier. */
    @Nonnull private Function<JWTClaimsSet,String> rootTokenIdentifierLookupStrategy;

    /** Revocation method used when revoking a token. */
    private OAuth2TokenRevocationMethod revocationMethod;

    /** Revocation lifetime to use. */
    private Duration revocationLifetime;

    /** The claims set to operate on. */
    private JWTClaimsSet claimsSet;

    /**
     * Constructor.
     */
    public RevokeToken() {
        revocationMethodLookupStrategy = new RevocationMethodLookupFunction();
        chainRevocationLifetimeLookupStrategy = new DefaultChainRevocationLifetimeLookupStrategy();
        tokenRevocationLifetimeLookupStrategy = new DefaultTokenRevocationLifetimeLookupStrategy();
        rootTokenIdentifierLookupStrategy = new DefaultRootTokenIdentifierLookupStrategy();
    }

    /**
     * Set the revocation cache instance to use.
     * 
     * @param cache The revocationCache to set.
     */
    public void setRevocationCache(@Nonnull final RevocationCache cache) {
        ComponentSupport.ifInitializedThrowUnmodifiabledComponentException(this);
        revocationCache = Constraint.isNotNull(cache, "RevocationCache cannot be null");
    }

    /**
     * Set strategy for looking up which revocation method should be used when revoking a token.
     * 
     * @param strategy What to set.
     */
    public void setRevocationMethodLookupStrategy(
            @Nonnull final Function<ProfileRequestContext,OAuth2TokenRevocationMethod> strategy) {
        revocationMethodLookupStrategy = Constraint.isNotNull(strategy, "Lookup strategy cannot be null");        
    }

    /**
     * Set a lookup strategy for the chain revocation lifetime.
     *
     * @param strategy What to set.
     */
    public void setChainRevocationLifetimeLookupStrategy(
            @Nullable final Function<ProfileRequestContext,Duration> strategy) {
        chainRevocationLifetimeLookupStrategy = Constraint.isNotNull(strategy, "Lookup strategy cannot be null");
    }

    /**
     * Set a lookup strategy for the token revocation lifetime.
     *
     * @param strategy What to set.
     */
    public void setTokenRevocationLifetimeLookupStrategy(
            @Nullable final Function<JWTClaimsSet,Duration> strategy) {
        tokenRevocationLifetimeLookupStrategy = Constraint.isNotNull(strategy, "Lookup strategy cannot be null");
    }

    /**
     * Set a lookup strategy for the root token identifier.
     *
     * @param strategy What to set.
     */
    public void setRootTokenIdentifierLookupStrategy(@Nullable final Function<JWTClaimsSet,String> strategy) {
        rootTokenIdentifierLookupStrategy = Constraint.isNotNull(strategy, "Lookup strategy cannot be null");
    }

    /** {@inheritDoc} */
    @Override
    protected void doInitialize() throws ComponentInitializationException {
        super.doInitialize();
        
        if (revocationCache == null) {
            throw new ComponentInitializationException("RevocationCache cannot be null");
        }    
    }

    /** {@inheritDoc} */
    @Override
    protected boolean doPreExecute(@Nonnull final ProfileRequestContext profileRequestContext) {
        if (!super.doPreExecute(profileRequestContext)) {
            return false;
        }

        revocationMethod = revocationMethodLookupStrategy.apply(profileRequestContext);
        if (revocationMethod == null) {
            log.error("{} Unable to obtain revocation method to use", getLogPrefix());
            ActionSupport.buildEvent(profileRequestContext, IdPEventIds.INVALID_PROFILE_CONFIG);
            return false;            
        }

        final OAuth2TokenMgmtResponseContext ctx = profileRequestContext.getOutboundMessageContext().getSubcontext(
                OAuth2TokenMgmtResponseContext.class);
        if (ctx == null || ctx.getTokenClaimsSet() == null) {
            log.debug("{} No token validated for revocation, assumed to be invalid", getLogPrefix());
            return false;
        }
        claimsSet = ctx.getTokenClaimsSet();

        if (OAuth2TokenRevocationMethod.CHAIN.equals(revocationMethod)) {
            revocationLifetime = chainRevocationLifetimeLookupStrategy.apply(profileRequestContext);
        } else if (OAuth2TokenRevocationMethod.TOKEN.equals(revocationMethod)) {
            revocationLifetime = tokenRevocationLifetimeLookupStrategy.apply(claimsSet);
        }

        if (revocationLifetime == null || Duration.ZERO.equals(revocationLifetime)) {
            log.error("{} Unable to obtain revocation lifetime to use", getLogPrefix());
            ActionSupport.buildEvent(profileRequestContext, IdPEventIds.INVALID_PROFILE_CONFIG);
            return false;
        }
        return true;
    }

    /** {@inheritDoc} */
    @Override
    protected void doExecute(@Nonnull final ProfileRequestContext profileRequestContext) {
        final String jti = claimsSet.getJWTID(); 
        if (jti == null) {
            log.error("{} No ID found in token claims set (this should be impossible)", getLogPrefix());
            return;
        }

        if (OAuth2TokenRevocationMethod.CHAIN.equals(revocationMethod)) {
            final String rootJti = rootTokenIdentifierLookupStrategy.apply(claimsSet);
            final String idToRevoke;
            if (StringSupport.trimOrNull(rootJti) == null) {
                log.warn("{} No root token identifier returned, using JWT id", getLogPrefix());
                idToRevoke = jti;
            } else {
                idToRevoke = rootJti;
            }
            if (revocationCache.revoke(RevocationCacheContexts.AUTHORIZATION_CODE, idToRevoke, revocationLifetime)) {
                log.debug("{} Revoked all tokens based on ID '{}'", getLogPrefix(), idToRevoke);
            } else {
                log.warn("{} Failed to revoke tokens based on ID '{}'", getLogPrefix(), idToRevoke);
                ActionSupport.buildEvent(profileRequestContext, OidcEventIds.REVOCATION_FAILED);
            }
        } else if (OAuth2TokenRevocationMethod.TOKEN.equals(revocationMethod)) {
            if (revocationCache.revoke(RevocationCacheContexts.SINGLE_ACCESS_OR_REFRESH_TOKENS, jti,
                    revocationLifetime)) {
                log.debug("{} Revoked the single token with ID '{}'", getLogPrefix(), jti);
            } else {
                log.warn("{} Failed to revoke the single token with ID '{}'", getLogPrefix(), jti);
                ActionSupport.buildEvent(profileRequestContext, OidcEventIds.REVOCATION_FAILED);
            }
        } else {
            log.error("{} Unrecognized revocation method: {}", getLogPrefix(), revocationMethod);
            ActionSupport.buildEvent(profileRequestContext, OidcEventIds.REVOCATION_FAILED);
        }
    }
}