/*
 * Licensed to the University Corporation for Advanced Internet Development,
 * Inc. (UCAID) under one or more contributor license agreements.  See the
 * NOTICE file distributed with this work for additional information regarding
 * copyright ownership. The UCAID licenses this file to You under the Apache
 * License, Version 2.0 (the "License"); you may not use this file except in
 * compliance with the License.  You may obtain a copy of the License at
 *
 *    http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package net.shibboleth.idp.plugin.oidc.op.oauth2.messaging.impl;

import java.util.List;
import java.util.function.Function;

import javax.annotation.Nonnull;

import org.opensaml.messaging.context.MessageContext;
import org.opensaml.messaging.context.navigate.ChildContextLookup;
import org.opensaml.messaging.handler.AbstractMessageHandler;
import org.opensaml.saml.common.messaging.context.SAMLMetadataContext;
import org.opensaml.saml.common.messaging.context.SAMLPeerEntityContext;
import org.opensaml.saml.saml2.metadata.RoleDescriptor;
import org.opensaml.saml.saml2.metadata.SPSSODescriptor;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.nimbusds.openid.connect.sdk.rp.OIDCClientInformation;

import net.shibboleth.oidc.metadata.context.OIDCMetadataContext;
import net.shibboleth.utilities.java.support.component.ComponentSupport;
import net.shibboleth.utilities.java.support.logic.Constraint;

/**
 * A message handler that attempts to locate OIDC client information from the SAML entity descriptor containing role
 * descriptor of type {@link SPSSODescriptor}. If it contains {@link OIDCClientInformation} in the
 * object metadata, it is attached inside {@link OIDCMetadataContext} as a child of the input {@link MessageContext}.
 */
public class PopulateOIDCMetadataContext extends AbstractMessageHandler {

    /** Class logger. */
    @Nonnull
    private final Logger log = LoggerFactory.getLogger(PopulateOIDCMetadataContext.class);
    
    /** Set the strategy used to look up the {@link SAMLMetadataContext} to draw from. */
    @Nonnull private Function<MessageContext,SAMLMetadataContext> samlMetadataContextLookupStrategy;
    
    /**
     * Constructor.
     */
    public PopulateOIDCMetadataContext() {
        samlMetadataContextLookupStrategy = new ChildContextLookup<>(SAMLMetadataContext.class).compose(
                new ChildContextLookup<>(SAMLPeerEntityContext.class));
    }
    
    /**
     * Set the strategy used to look up the {@link SAMLMetadataContext} to draw from.
     * 
     * @param strategy strategy used to look up the {@link SAMLMetadataContext}
     */
    public void setSAMLMetadataContextLookupStrategy(
            @Nonnull final Function<MessageContext,SAMLMetadataContext> strategy) {
        ComponentSupport.ifInitializedThrowUnmodifiabledComponentException(this);

        samlMetadataContextLookupStrategy =
                Constraint.isNotNull(strategy, "SAMLMetadataContext lookup strategy cannot be null");
    }
    
    /** {@inheritDoc} */
    @Override
    protected void doInvoke(@Nonnull final MessageContext messageContext) {
        
        final SAMLMetadataContext samlMetadataCtx = samlMetadataContextLookupStrategy.apply(messageContext);
        if (samlMetadataCtx == null) {
            log.error("Unable to locate SAMLMetadataContext");
            return;
        }

        final List<RoleDescriptor> roleDescriptors 
            = samlMetadataCtx.getEntityDescriptor().getRoleDescriptors(SPSSODescriptor.DEFAULT_ELEMENT_NAME);
        for (final RoleDescriptor roleDescriptor : roleDescriptors) {
            if (roleDescriptor instanceof SPSSODescriptor) {
                final List<OIDCClientInformation> clientInformations 
                    = roleDescriptor.getObjectMetadata().get(OIDCClientInformation.class);
                if (clientInformations != null && clientInformations.size() > 0) {
                    final OIDCMetadataContext oidcCtx = new OIDCMetadataContext();
                    oidcCtx.setClientInformation(clientInformations.get(0));
                    messageContext.addSubcontext(oidcCtx);
                    log.debug("{} Client information found and attached", getLogPrefix());
                    return;
                }   
            }
        }
        
        log.debug("{} No client information found to be attached into OIDC metadata context.", getLogPrefix());
    }

}