/*
 * Licensed to the University Corporation for Advanced Internet Development,
 * Inc. (UCAID) under one or more contributor license agreements.  See the
 * NOTICE file distributed with this work for additional information regarding
 * copyright ownership. The UCAID licenses this file to You under the Apache
 * License, Version 2.0 (the "License"); you may not use this file except in
 * compliance with the License.  You may obtain a copy of the License at
 *
 *    http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package net.shibboleth.idp.plugin.oidc.op.audit.impl;

import java.util.function.Function;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;

import org.opensaml.profile.context.ProfileRequestContext;

import net.shibboleth.idp.plugin.oidc.op.messaging.context.OIDCClientRegistrationResponseContext;
import net.shibboleth.idp.plugin.oidc.op.profile.context.navigate.OIDCRegistrationResponseContextLookupFunction;
import net.shibboleth.utilities.java.support.logic.Constraint;

/**
 * Looks up the client ID value from the OIDC client registration response context.
 */
public class ClientIdRegistrationAuditExtractor implements Function<ProfileRequestContext,String> {

    /** Lookup strategy for the context to find the subject value from. */
    @Nonnull
    private final Function<ProfileRequestContext, OIDCClientRegistrationResponseContext> ctxLookupStrategy;

    /** Constructor. */
    public ClientIdRegistrationAuditExtractor() {
        this(new OIDCRegistrationResponseContextLookupFunction());
    }

    /**
     * Constructor.
     * 
     * @param strategy context lookup strategy
     */
    public ClientIdRegistrationAuditExtractor(
            final Function<ProfileRequestContext, OIDCClientRegistrationResponseContext> strategy) {
        ctxLookupStrategy =
                Constraint.isNotNull(strategy, "OIDCClientRegistrationResponseContext lookup strategy cannot be null");
    }

    /** {@inheritDoc} */
    @Nullable public String apply(@Nullable final ProfileRequestContext input) {
        final OIDCClientRegistrationResponseContext context = ctxLookupStrategy.apply(input);
        if (context != null) {
            return context.getClientId();
        }

        return null;
    }
}
