/*
 * Licensed to the University Corporation for Advanced Internet Development,
 * Inc. (UCAID) under one or more contributor license agreements.  See the
 * NOTICE file distributed with this work for additional information regarding
 * copyright ownership. The UCAID licenses this file to You under the Apache
 * License, Version 2.0 (the "License"); you may not use this file except in
 * compliance with the License.  You may obtain a copy of the License at
 *
 *    http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package net.shibboleth.idp.plugin.oidc.op.audit.impl;

import java.text.ParseException;
import java.util.List;
import java.util.function.Function;

import javax.annotation.Nonnull;

import org.opensaml.profile.context.ProfileRequestContext;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.nimbusds.openid.connect.sdk.AuthenticationRequest;

import net.shibboleth.idp.plugin.oidc.op.profile.context.navigate.AbstractAuthenticationRequestLookupFunction;
import net.shibboleth.utilities.java.support.annotation.constraint.NotEmpty;

/**
 * A function that resolves a claim value from the authentication request. If request object was involved and contains
 * a value for the claim, it will be returned. Otherwise it's fetched from the authentication request parameters.
 */
public class AuthenticationRequestClaimsAuditExtractor extends AbstractAuthenticationRequestLookupFunction<String>
    implements Function<ProfileRequestContext,String> {
    
    /** Class logger. */
    @Nonnull
    private Logger log = LoggerFactory.getLogger(AuthenticationRequestClaimsAuditExtractor.class);

    /** The claim whose value is to be extracted. */
    @Nonnull @NotEmpty private final String key;
    
    /**
     * Constructor.
     *
     * @param claim The claim whose value is to be resolved from the id_token claims set.
     */
    public AuthenticationRequestClaimsAuditExtractor(@Nonnull @NotEmpty final String claim) {
        key = claim;
    }
    
    /** {@inheritDoc} */
    @Override
    protected String doLookup(@Nonnull final AuthenticationRequest request) {
        try {
            if (getRequestObject() != null && getRequestObject().getJWTClaimsSet().getClaim(key) != null) {
                return getRequestObject().getJWTClaimsSet().getStringClaim(key);
            }
        } catch (final ParseException e) {
            log.error("Unable to parse response mode from request object response_mode value");
            return null;
        }
        final List<String> value = request.toParameters().get(key);
        if (value == null || value.isEmpty()) {
            return null;
        }
        return value.get(0);
    }
}
