/*
 * Licensed to the University Corporation for Advanced Internet Development,
 * Inc. (UCAID) under one or more contributor license agreements.  See the
 * NOTICE file distributed with this work for additional information regarding
 * copyright ownership. The UCAID licenses this file to You under the Apache
 * License, Version 2.0 (the "License"); you may not use this file except in
 * compliance with the License.  You may obtain a copy of the License at
 *
 *    http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package net.shibboleth.idp.plugin.oidc.op.audit.impl;

import java.util.function.Function;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;

import org.opensaml.profile.context.ProfileRequestContext;

import net.shibboleth.idp.plugin.oidc.op.messaging.context.OIDCAuthenticationResponseContext;
import net.shibboleth.idp.plugin.oidc.op.profile.context.navigate.OIDCAuthenticationResponseContextLookupFunction;
import net.shibboleth.utilities.java.support.logic.Constraint;

/**
 * Looks up the 'acr' value from the OIDC authentication response context.
 */
public class AuthenticationContextReferenceAuditExtractor implements Function<ProfileRequestContext, String> {
    
    /** Lookup strategy for the context to find the subject value from. */
    @Nonnull
    private final Function<ProfileRequestContext, OIDCAuthenticationResponseContext> ctxLookupStrategy;

    /**
     * Constructor.
     */
    public AuthenticationContextReferenceAuditExtractor() {
        this(new OIDCAuthenticationResponseContextLookupFunction());
    }

    /**
     * Constructor.
     *
     * @param strategy lookup strategy for message
     */
    public AuthenticationContextReferenceAuditExtractor(
            @Nonnull final Function<ProfileRequestContext, OIDCAuthenticationResponseContext> strategy) {
        ctxLookupStrategy =
                Constraint.isNotNull(strategy, "OIDC authentication response context lookup strategy cannot be null");
    }

    /** {@inheritDoc} */
    @Override
    @Nullable
    public String apply(@Nullable final ProfileRequestContext input) {
        final OIDCAuthenticationResponseContext context = ctxLookupStrategy.apply(input);
        if (context != null && context.getAcr() != null) {
            return context.getAcr().getValue();
        }

        return null;
    }
}
