/*
 * Licensed to the University Corporation for Advanced Internet Development,
 * Inc. (UCAID) under one or more contributor license agreements.  See the
 * NOTICE file distributed with this work for additional information regarding
 * copyright ownership. The UCAID licenses this file to You under the Apache
 * License, Version 2.0 (the "License"); you may not use this file except in
 * compliance with the License.  You may obtain a copy of the License at
 *
 *    http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package net.shibboleth.idp.plugin.oidc.op.servlet;

import java.util.Set;

import javax.annotation.Nonnull;
import javax.servlet.FilterRegistration;
import javax.servlet.ServletContainerInitializer;
import javax.servlet.ServletContext;
import javax.servlet.ServletException;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.web.filter.DelegatingFilterProxy;

import net.shibboleth.utilities.java.support.primitive.StringSupport;

/**
 * A {@link ServletContainerInitializer} implementation that registers dynamic response header filter for enabling
 * configurable headers. The target bean name is {@link RegisterFilterServletContextInitializer#TARGET_BEAN_NAME}.
 *
 * The filter registration can be disabled by setting the system property
 * {@link RegisterFilterServletContextInitializer#SYSTEM_PROPERTY_ACTIVATION} to <pre>disabled</pre>.
 *
 * The filter mappings can be configured via space-separated list on the system property
 * {@link RegisterFilterServletContextInitializer#SYSTEM_PROPERTY_MAPPINGS}. By default the value is
 * <pre>/profile/oidc/* /profile/oauth2/</pre>.
 */
public class RegisterFilterServletContextInitializer implements ServletContainerInitializer {

    /** System property name for the activation flag of this class. */
    public static final String SYSTEM_PROPERTY_ACTIVATION =
            RegisterFilterServletContextInitializer.class.getCanonicalName();

    /** System property name for configuring the filter mappings. */
    public static final String SYSTEM_PROPERTY_MAPPINGS =
            SYSTEM_PROPERTY_ACTIVATION + ".mappings";

    /** The filter name for the dynamic response header filter for the OP's flows. */
    public static final String FILTER_NAME_DYNAMIC_OIDC_RESPONSE_HEADER = "DynamicOidcResponseHeaderFilter";

    /** The target bean name for the dynamic response header filter. */
    public static final String TARGET_BEAN_NAME = "shibboleth.oidc.ResponseHeaderFilter";

    /** The value for the filter mappings, if no custom configuration is set. */
    public static final String DEFAULT_MAPPINGS = "/profile/oidc/* /profile/oauth2/*";

    /** Class logger. */
    @Nonnull private final Logger log = LoggerFactory.getLogger(RegisterFilterServletContextInitializer.class);
    
    /** {@inheritDoc} */
    @Override
    public void onStartup(final Set<Class<?>> c, final ServletContext ctx) throws ServletException {
        final String flag = System.getProperty(SYSTEM_PROPERTY_ACTIVATION);
        log.debug("The value of the flag {}: {}", SYSTEM_PROPERTY_ACTIVATION, flag);
        if ("disabled".equalsIgnoreCase(flag)) {
            log.info("The filter registration is disabled according to the system properties");
            return;
        }

        log.debug("Attempting to register filter {}", FILTER_NAME_DYNAMIC_OIDC_RESPONSE_HEADER);
        final FilterRegistration.Dynamic headerFilter = ctx.addFilter(FILTER_NAME_DYNAMIC_OIDC_RESPONSE_HEADER,
                DelegatingFilterProxy.class);
        final String mappings;
        if (StringSupport.trimOrNull(System.getProperty(SYSTEM_PROPERTY_MAPPINGS)) == null) {
            mappings = DEFAULT_MAPPINGS;
            log.debug("Using default mappings: {}", DEFAULT_MAPPINGS);
        } else {
            mappings = System.getProperty(SYSTEM_PROPERTY_MAPPINGS);
            log.debug("Using custom mappings: {}", mappings);
        }

        for (final String mapping : mappings.split(" ")) {
            headerFilter.addMappingForUrlPatterns(null, false, mapping);
            log.debug("Mapping added for the following pattern: {}", mapping);
        }
        headerFilter.setInitParameter("targetBeanName", TARGET_BEAN_NAME);

        log.info("Registered the filter '{}'.", FILTER_NAME_DYNAMIC_OIDC_RESPONSE_HEADER);

    }
}
