/*
 * Licensed to the University Corporation for Advanced Internet Development,
 * Inc. (UCAID) under one or more contributor license agreements.  See the
 * NOTICE file distributed with this work for additional information regarding
 * copyright ownership. The UCAID licenses this file to You under the Apache
 * License, Version 2.0 (the "License"); you may not use this file except in
 * compliance with the License.  You may obtain a copy of the License at
 *
 *    http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package net.shibboleth.idp.plugin.oidc.op.profile.logic;

import java.util.function.Function;
import java.util.function.Predicate;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;

import org.opensaml.profile.context.ProfileRequestContext;

import com.nimbusds.oauth2.sdk.Scope;
import com.nimbusds.openid.connect.sdk.OIDCScopeValue;

import net.shibboleth.idp.plugin.oidc.op.profile.context.navigate.ValidatedScopeLookupFunction;
import net.shibboleth.utilities.java.support.logic.Constraint;

/**
 * Activation condition returning true if validated scope contains 'openid' scope.
 */
public class IssueIDTokenCondition implements Predicate<ProfileRequestContext> {

    /** Lookup strategy for the validated scope. */
    @Nonnull private Function<ProfileRequestContext, Scope> validatedScopeLookupStrategy;

    /**
     * Constructor.
     */
    public IssueIDTokenCondition() {
        validatedScopeLookupStrategy = new ValidatedScopeLookupFunction();
    }

    /**
     * Set the lookup strategy for the validated scope.
     * 
     * @param strategy What to set.
     */
    public void setValidatedScopeLookupStrategy(final @Nonnull Function<ProfileRequestContext, Scope> strategy) {
        validatedScopeLookupStrategy = Constraint.isNotNull(strategy, "Lookup strategy cannot be null");
    }
    
    /** {@inheritDoc} */
    @Override
    public boolean test(final @Nullable ProfileRequestContext input) {
        if (input == null) {
            return false;
        }
        final Scope validatedScope = validatedScopeLookupStrategy.apply(input);
        if (validatedScope == null || validatedScope.isEmpty()) {
            return false;
        }
        return validatedScope.contains(OIDCScopeValue.OPENID);
    }
}