/*
 * Licensed to the University Corporation for Advanced Internet Development,
 * Inc. (UCAID) under one or more contributor license agreements.  See the
 * NOTICE file distributed with this work for additional information regarding
 * copyright ownership. The UCAID licenses this file to You under the Apache
 * License, Version 2.0 (the "License"); you may not use this file except in
 * compliance with the License.  You may obtain a copy of the License at
 *
 *    http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package net.shibboleth.idp.plugin.oidc.op.profile.impl;

import java.util.Collections;
import java.util.Set;
import java.util.function.Function;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;

import org.opensaml.profile.action.ActionSupport;
import org.opensaml.profile.context.ProfileRequestContext;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import com.nimbusds.oauth2.sdk.AuthorizationGrant;
import com.nimbusds.oauth2.sdk.GrantType;

import net.shibboleth.oidc.metadata.context.OIDCMetadataContext;
import net.shibboleth.oidc.profile.config.navigate.GrantTypesLookupFunction;
import net.shibboleth.oidc.profile.core.OidcEventIds;
import net.shibboleth.utilities.java.support.annotation.constraint.NonnullElements;

/**
 * An action that validates the grant type is registered to the requesting RP. This action is used in Token end point to
 * check if authorization code or refresh token has been registered to be used as a grant.
 */
public class ValidateGrantType extends AbstractOIDCTokenResponseAction {

    /** Class logger. */
    @Nonnull private Logger log = LoggerFactory.getLogger(ValidateGrantType.class);

    /** Strategy to obtain enabled grant types. */
    @Nonnull private Function<ProfileRequestContext,Set<GrantType>> grantTypesLookupStrategy;

    /** Enabled grant types. */
    @Nullable @NonnullElements private Set<GrantType> enabledTypes;
    
    /** Constructor. */
    public ValidateGrantType() {
        grantTypesLookupStrategy = new GrantTypesLookupFunction();
    }
    
    /** {@inheritDoc} */
    @Override
    protected boolean doPreExecute(@Nonnull final ProfileRequestContext profileRequestContext) {
        if (!super.doPreExecute(profileRequestContext)) {
            return false;
        }
        
        enabledTypes = grantTypesLookupStrategy.apply(profileRequestContext);
        if (enabledTypes == null) {
            enabledTypes = Collections.emptySet();
        }
        
        return true;
    }
    
    /** {@inheritDoc} */
    @Override
    protected void doExecute(@Nonnull final ProfileRequestContext profileRequestContext) {
        
        final Set<GrantType> registeredTypes;
        final OIDCMetadataContext metadataCtx = getMetadataContext();
        if (metadataCtx != null) {
            registeredTypes = getMetadataContext().getClientInformation().getMetadata().getGrantTypes();
        } else {
            registeredTypes = null;
        }
        
        final AuthorizationGrant grant = getTokenRequest().getAuthorizationGrant();

        if (registeredTypes != null && !registeredTypes.contains(grant.getType())) {
            log.error("{} Grant type {} not registered for client", getLogPrefix(), grant.getType().getValue());
            ActionSupport.buildEvent(profileRequestContext, OidcEventIds.INVALID_GRANT_TYPE);
        } else if (!enabledTypes.contains(grant.getType())) {
            log.error("{} Grant type {} not enabled in profile configuration", getLogPrefix(),
                    grant.getType().getValue());
            ActionSupport.buildEvent(profileRequestContext, OidcEventIds.INVALID_GRANT_TYPE);
        }
    }

}