/*
 * Licensed to the University Corporation for Advanced Internet Development,
 * Inc. (UCAID) under one or more contributor license agreements.  See the
 * NOTICE file distributed with this work for additional information regarding
 * copyright ownership. The UCAID licenses this file to You under the Apache
 * License, Version 2.0 (the "License"); you may not use this file except in
 * compliance with the License.  You may obtain a copy of the License at
 *
 *    http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package net.shibboleth.idp.plugin.oidc.op.profile.impl;

import java.text.ParseException;
import java.util.Map;
import java.util.function.BiFunction;
import java.util.function.Function;

import javax.annotation.Nonnull;

import org.opensaml.profile.action.ActionSupport;
import org.opensaml.profile.action.EventIds;
import org.opensaml.profile.context.ProfileRequestContext;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.nimbusds.jwt.JWTClaimsSet;
import com.nimbusds.openid.connect.sdk.claims.IDTokenClaimsSet;

import net.shibboleth.idp.profile.IdPEventIds;
import net.shibboleth.oidc.profile.config.navigate.IDTokenManipulationStrategyLookupFunction;
import net.shibboleth.utilities.java.support.component.ComponentSupport;
import net.shibboleth.utilities.java.support.logic.Constraint;

/**
 * An action that can be used for manipulating id_token claims via configurable strategy (bi-function). If the
 * function exists, its result (a Map) is used as-is for building new contents for the upcoming id_token, i.e.
 * the previously built claims set is replaced. If the function returns null, the existing id_token contents
 * are not replaced.
 * 
 * @event {@link EventIds#INVALID_MSG_CTX} If the id_token does not exist.
 * @event {@link IdPEventIds#INVALID_PROFILE_CONFIG} If the new claims set is not compatible with the id_token
 * requirements (e.g. due to missing mandatory claims, or claims having unexpected syntax).
 */
public class ManipulateClaimsForIDToken extends AbstractOIDCAuthenticationResponseAction {

    /** Class logger. */
    @Nonnull private Logger log = LoggerFactory.getLogger(ManipulateClaimsForIDToken.class);

    /** Lookup function to supply strategy bi-function for manipulating id_token claims. */ 
    @Nonnull
    private Function<ProfileRequestContext,BiFunction<ProfileRequestContext,Map<String,Object>,Map<String,Object>>>
        idTokenManipulationStrategyLookupStrategy;

    /** The strategy used for manipulating the id_token. */
    private BiFunction<ProfileRequestContext,Map<String,Object>,Map<String,Object>> manipulationStrategy;

    /** The id_token to operate on. */
    private IDTokenClaimsSet idToken;

    /**
     * Constructor.
     */
    public ManipulateClaimsForIDToken() {
        idTokenManipulationStrategyLookupStrategy = new IDTokenManipulationStrategyLookupFunction();
    }

    /**
     * Set the lookup function to supply strategy bi-function for manipulating id_token claims.
     * 
     * @param strategy What to set
     */
    public void setIDTokenManipulationStrategyLookupStrategy(@Nonnull final
            Function<ProfileRequestContext,BiFunction<ProfileRequestContext,Map<String,Object>,Map<String,Object>>>
            strategy) {
        ComponentSupport.ifInitializedThrowUnmodifiabledComponentException(this);

        idTokenManipulationStrategyLookupStrategy =
                Constraint.isNotNull(strategy, "IDToken manipulation strategy lookup strategy cannot be null");
    }

    /** {@inheritDoc} */
    @Override
    protected boolean doPreExecute(@Nonnull final ProfileRequestContext profileRequestContext) {
        if (!super.doPreExecute(profileRequestContext)) {
            return false;
        }
        
        idToken = getOidcResponseContext().getIDToken();
        if (idToken == null) {
            log.error("{} No id token", getLogPrefix());
            ActionSupport.buildEvent(profileRequestContext, EventIds.INVALID_MSG_CTX);
            return false;
        }
        
        manipulationStrategy = idTokenManipulationStrategyLookupStrategy.apply(profileRequestContext);
        if (manipulationStrategy == null) {
            log.debug("{} No manipulation strategy resolved, nothing to do.", getLogPrefix());
            return false;
        }
    
        return true;
    }

    /** {@inheritDoc} */
    @Override
    protected void doExecute(@Nonnull final ProfileRequestContext profileRequestContext) {
        final Map<String, Object> result = manipulationStrategy.apply(profileRequestContext, idToken.toJSONObject());
        if (result == null) {
            log.debug("{} Manipulation strategy retruned null, leaving id_token claims untouched.", getLogPrefix());
            return;
        }
        log.debug("{} Applying the manipulated claims into the id_token", getLogPrefix());
        final IDTokenClaimsSet newIdToken;
        try {
            newIdToken = new IDTokenClaimsSet(JWTClaimsSet.parse(result));
        } catch (final ParseException | com.nimbusds.oauth2.sdk.ParseException e) {
            log.error("{} The resulted claims set could not be transformed into id_token", getLogPrefix(), e);
            ActionSupport.buildEvent(profileRequestContext, IdPEventIds.INVALID_PROFILE_CONFIG);
            return;
        }
        log.debug("{} Replacing the id_token with the manipulated contents", getLogPrefix());
        getOidcResponseContext().setIDToken(newIdToken);   
    }

}
