/*
 * Licensed to the University Corporation for Advanced Internet Development,
 * Inc. (UCAID) under one or more contributor license agreements.  See the
 * NOTICE file distributed with this work for additional information regarding
 * copyright ownership. The UCAID licenses this file to You under the Apache
 * License, Version 2.0 (the "License"); you may not use this file except in
 * compliance with the License.  You may obtain a copy of the License at
 *
 *    http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package net.shibboleth.idp.plugin.oidc.op.profile.impl;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;

import org.opensaml.profile.action.ActionSupport;
import org.opensaml.profile.action.EventIds;
import org.opensaml.profile.context.ProfileRequestContext;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import com.nimbusds.jwt.JWT;
import com.nimbusds.oauth2.sdk.AccessTokenResponse;
import com.nimbusds.oauth2.sdk.TokenResponse;
import com.nimbusds.oauth2.sdk.token.AccessToken;
import com.nimbusds.oauth2.sdk.token.Tokens;
import com.nimbusds.openid.connect.sdk.OIDCTokenResponse;
import com.nimbusds.openid.connect.sdk.token.OIDCTokens;

/**
 * Action that forms outbound message based on token request and response context. Formed message is set to
 * {@link ProfileRequestContext#getOutboundMessageContext()}.
 * 
 * <p>This is both OIDC and OAuth aware, based on the presence of an ID token in the context.</p>
 */
public class FormOutboundTokenResponseMessage extends AbstractOIDCTokenResponseAction {

    /** Class logger. */
    @Nonnull private Logger log = LoggerFactory.getLogger(FormOutboundTokenResponseMessage.class);

    /** access token for response. */
    @Nullable private AccessToken accessToken;

    /** ID token for response. */
    @Nullable private JWT idToken;

    /** {@inheritDoc} */
    @Override
    protected boolean doPreExecute(@Nonnull final ProfileRequestContext profileRequestContext) {

        if (!super.doPreExecute(profileRequestContext)) {
            return false;
        }
        
        accessToken = getOidcResponseContext().getAccessToken();
        if (accessToken == null) {
            log.error("{} unable to provide required access token", getLogPrefix());
            ActionSupport.buildEvent(profileRequestContext, EventIds.INVALID_PROFILE_CTX);
            return false;
        }
        
        idToken = getOidcResponseContext().getProcessedToken();
        return true;
    }

    /** {@inheritDoc} */
    @Override
    protected void doExecute(@Nonnull final ProfileRequestContext profileRequestContext) {
        
        final TokenResponse resp;
        if (idToken != null) {
            resp = new OIDCTokenResponse(
                    new OIDCTokens(idToken, accessToken, getOidcResponseContext().getRefreshToken()));
        } else {
            // No refresh tokens likely here, but just in case.
            resp = new AccessTokenResponse(new Tokens(accessToken, getOidcResponseContext().getRefreshToken()));
        }
        profileRequestContext.getOutboundMessageContext().setMessage(resp);
    }

}