/*
 * Licensed to the University Corporation for Advanced Internet Development,
 * Inc. (UCAID) under one or more contributor license agreements.  See the
 * NOTICE file distributed with this work for additional information regarding
 * copyright ownership. The UCAID licenses this file to You under the Apache
 * License, Version 2.0 (the "License"); you may not use this file except in
 * compliance with the License.  You may obtain a copy of the License at
 *
 *    http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package net.shibboleth.idp.plugin.oidc.op.profile.impl;

import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Set;
import java.util.function.Function;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;

import net.minidev.json.JSONObject;
import net.shibboleth.idp.attribute.AttributeEncodingException;
import net.shibboleth.idp.attribute.AttributesMapContainer;
import net.shibboleth.idp.attribute.IdPAttribute;
import net.shibboleth.idp.attribute.context.AttributeContext;
import net.shibboleth.idp.attribute.transcoding.AttributeTranscoder;
import net.shibboleth.idp.attribute.transcoding.AttributeTranscoderRegistry;
import net.shibboleth.idp.attribute.transcoding.TranscoderSupport;
import net.shibboleth.idp.attribute.transcoding.TranscodingRule;
import net.shibboleth.idp.plugin.oidc.op.messaging.context.OIDCAuthenticationResponseConsentContext;
import net.shibboleth.idp.plugin.oidc.op.messaging.context.OIDCAuthenticationResponseTokenClaimsContext;
import net.shibboleth.idp.plugin.oidc.op.profile.context.navigate.DefaultResponseClaimsSetLookupFunction;
import net.shibboleth.idp.plugin.oidc.op.profile.context.navigate.OIDCAuthenticationResponseContextLookupFunction;
import net.shibboleth.idp.profile.IdPEventIds;
import net.shibboleth.idp.profile.context.RelyingPartyContext;
import net.shibboleth.oidc.profile.config.navigate.AlwaysIncludedAttributesLookupFunction;
import net.shibboleth.oidc.profile.config.navigate.DeniedUserInfoAttributesLookupFunction;
import net.shibboleth.utilities.java.support.annotation.constraint.Live;
import net.shibboleth.utilities.java.support.annotation.constraint.NonnullAfterInit;
import net.shibboleth.utilities.java.support.annotation.constraint.NonnullElements;
import net.shibboleth.utilities.java.support.component.ComponentInitializationException;
import net.shibboleth.utilities.java.support.component.ComponentSupport;
import net.shibboleth.utilities.java.support.logic.Constraint;
import net.shibboleth.utilities.java.support.service.ReloadableService;
import net.shibboleth.utilities.java.support.service.ServiceableComponent;

import org.opensaml.messaging.context.navigate.ChildContextLookup;
import org.opensaml.profile.action.ActionSupport;
import org.opensaml.profile.action.EventIds;
import org.opensaml.profile.context.ProfileRequestContext;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.nimbusds.oauth2.sdk.ResponseType;
import com.nimbusds.openid.connect.sdk.AuthenticationRequest;
import com.nimbusds.openid.connect.sdk.OIDCResponseTypeValue;
import com.nimbusds.openid.connect.sdk.claims.ClaimsSet;
import com.nimbusds.openid.connect.sdk.claims.IDTokenClaimsSet;
import com.nimbusds.openid.connect.sdk.claims.UserInfo;

/**
 * Action that adds claims to a {@link ClaimsSet}. Claims are formed of resolved attributes having OIDC encoder. Action
 * verifies user has consented to release attribute, if consent information is available. Actions will not add claims
 * listed as reserved.
 */
public class AddAttributesToClaimsSet extends AbstractOIDCResponseAction {

    /** Class logger. */
    @Nonnull private Logger log = LoggerFactory.getLogger(AddAttributesToClaimsSet.class);

    /**
     * Strategy used to locate the {@link AttributeContext} associated with a given {@link ProfileRequestContext}.
     */
    @Nonnull private Function<ProfileRequestContext, AttributeContext> attributeContextLookupStrategy;

    /**
     * Strategy used to locate the response {@link ClaimsSet} associated with a given {@link ProfileRequestContext}.
     */
    @Nonnull private Function<ProfileRequestContext, ClaimsSet> responseClaimsSetLookupStrategy;

    /** Strategy used to locate the {@link OIDCAuthenticationResponseConsentContext}. */
    @Nonnull
    private Function<ProfileRequestContext, OIDCAuthenticationResponseConsentContext> consentContextLookupStrategy;
    
    /** Strategy used to obtain the set of attribute IDs to include in the ID token in all cases. */
    @Nonnull private Function<ProfileRequestContext,Set<String>> alwaysIncludedAttributesLookupStrategy;

    /** Strategy used to obtain the set of attribute IDs to omit from the UserInfo token. */
    @Nonnull private Function<ProfileRequestContext,Set<String>> deniedUserInfoAttributesLookupStrategy;

    /** Transcoder registry service object. */
    @NonnullAfterInit private ReloadableService<AttributeTranscoderRegistry> transcoderRegistry;

    /**
     * Whether attributes that result in an {@link net.shibboleth.idp.attribute.AttributeEncodingException}
     * when being encoded should be ignored or
     * result in an {@link net.shibboleth.idp.profile.IdPEventIds#UNABLE_ENCODE_ATTRIBUTE} transition.
     */
    private boolean ignoringUnencodableAttributes;
    
    /** AttributeContext to use. */
    @Nullable private AttributeContext attributeCtx;

    /** Claims Set to use. */
    @Nullable private ClaimsSet claimsSet;

    /** Whether we can add claims to IDToken by default i.e. response type is "id_token". */
    private boolean addToIDTokenByDefault;

    /** List of claim names that will not be added. */
    @Nullable @NonnullElements private List<String> reservedClaimNames;

    /** Attributes to include in ID token no matter what. */
    @Nullable @NonnullElements private Set<String> alwaysIncludedAttributes;

    /** Attributes to omit from UserInfo token. */
    @Nullable @NonnullElements private Set<String> deniedUserInfoAttributes;

    /** Constructor. */
    AddAttributesToClaimsSet() {
        attributeContextLookupStrategy = new ChildContextLookup<>(AttributeContext.class).compose(
                new ChildContextLookup<>(RelyingPartyContext.class));
        responseClaimsSetLookupStrategy = new DefaultResponseClaimsSetLookupFunction();
        consentContextLookupStrategy =
                new ChildContextLookup<>(OIDCAuthenticationResponseConsentContext.class).compose(
                        new OIDCAuthenticationResponseContextLookupFunction());
        
        alwaysIncludedAttributesLookupStrategy = new AlwaysIncludedAttributesLookupFunction();
        deniedUserInfoAttributesLookupStrategy = new DeniedUserInfoAttributesLookupFunction();
        
        ignoringUnencodableAttributes = true;
    }

    /**
     * Sets the registry of transcoding rules to apply to encode attributes.
     * 
     * @param registry registry service interface
     */
    public void setTranscoderRegistry(@Nonnull final ReloadableService<AttributeTranscoderRegistry> registry) {
        ComponentSupport.ifInitializedThrowUnmodifiabledComponentException(this);
        
        transcoderRegistry = Constraint.isNotNull(registry, "AttributeTranscoderRegistry cannot be null");
    }
    
    /**
     * Set whether the attributes that result in an {@link AttributeEncodingException} when being encoded
     * should be ignored or result in an {@link IdPEventIds#UNABLE_ENCODE_ATTRIBUTE} transition.
     * 
     * @param flag flag to set
     */
    public void setIgnoringUnencodableAttributes(final boolean flag) {
        ComponentSupport.ifInitializedThrowUnmodifiabledComponentException(this);

        ignoringUnencodableAttributes = flag;
    }
    
    /**
     * Set list of claim names that will not be added.
     * 
     * @param claimNames list of claim names that will not be added.
     */
    public void setReservedClaimNames(final List<String> claimNames) {
        ComponentSupport.ifInitializedThrowUnmodifiabledComponentException(this);
        reservedClaimNames = claimNames;
    }

    /**
     * Set the strategy used to locate the response {@link ClaimsSet} associated with a given
     * {@link ProfileRequestContext}.
     * 
     * @param strategy strategy used to locate the response {@link ClaimsSet} associated with a given
     *            {@link ProfileRequestContext}
     */
    public void setResponseClaimsSetLookupStrategy(@Nonnull final Function<ProfileRequestContext, ClaimsSet> strategy) {
        ComponentSupport.ifInitializedThrowUnmodifiabledComponentException(this);

        responseClaimsSetLookupStrategy =
                Constraint.isNotNull(strategy, "Response Claims Set lookup strategy cannot be null");
    }

    /**
     * Set the strategy used to locate the {@link AttributeContext} associated with a given
     * {@link ProfileRequestContext}.
     * 
     * @param strategy strategy used to locate the {@link AttributeContext} associated with a given
     *            {@link ProfileRequestContext}
     */
    public void setAttributeContextLookupStrategy(
            @Nonnull final Function<ProfileRequestContext, AttributeContext> strategy) {
        ComponentSupport.ifInitializedThrowUnmodifiabledComponentException(this);

        attributeContextLookupStrategy =
                Constraint.isNotNull(strategy, "AttributeContext lookup strategy cannot be null");
    }

    /**
     * Set the strategy used to locate the {@link OIDCAuthenticationResponseTokenClaimsContext} associated with a given
     * {@link ProfileRequestContext}.
     * 
     * @param strategy lookup strategy
     */
    public void setOIDCAuthenticationResponseConsentContextLookupStrategy(
            @Nonnull final Function<ProfileRequestContext, OIDCAuthenticationResponseConsentContext> strategy) {
        ComponentSupport.ifInitializedThrowUnmodifiabledComponentException(this);
        consentContextLookupStrategy = Constraint.isNotNull(strategy,
                "OIDCAuthenticationResponseConsentContext lookup strategy cannot be null");
    }

    /**
     * Set the strategy used to obtain the set of attribute IDs always included in ID tokens.
     * 
     * @param strategy lookup strategy
     */
    public void setAlwaysIncludedAttributesLookupStrategy(
            @Nonnull final Function<ProfileRequestContext,Set<String>> strategy) {
        ComponentSupport.ifInitializedThrowUnmodifiabledComponentException(this);
        
        alwaysIncludedAttributesLookupStrategy = Constraint.isNotNull(strategy,
                "Always included attributes lookup strategy cannot be null");
    }

    /**
     * Set the strategy used to obtain the set of attribute IDs to omit from UserInfo tokens.
     * 
     * @param strategy lookup strategy
     */
    public void setDeniedUserInfoAttributesLookupStrategy(
            @Nonnull final Function<ProfileRequestContext,Set<String>> strategy) {
        ComponentSupport.ifInitializedThrowUnmodifiabledComponentException(this);
        
        deniedUserInfoAttributesLookupStrategy = Constraint.isNotNull(strategy,
                "Denied UserInfo attributes lookup strategy cannot be null");
    }
    
    /** {@inheritDoc} */
    @Override
    protected void doInitialize() throws ComponentInitializationException {
        super.doInitialize();
        
        if (transcoderRegistry == null) {
            throw new ComponentInitializationException("AttributeTranscoderRegistry cannot be null");
        }
    }
    
    /** {@inheritDoc} */
    @Override
    protected boolean doPreExecute(@Nonnull final ProfileRequestContext profileRequestContext) {
        if (!super.doPreExecute(profileRequestContext)) {
            return false;
        }
        
        attributeCtx = attributeContextLookupStrategy.apply(profileRequestContext);
        if (attributeCtx == null) {
            log.debug("{} No AttributeSubcontext available, nothing to do", getLogPrefix());
            return false;
        }
        
        claimsSet = responseClaimsSetLookupStrategy.apply(profileRequestContext);
        if (claimsSet == null) {
            log.error("{} No claims set to fill", getLogPrefix());
            ActionSupport.buildEvent(profileRequestContext, EventIds.INVALID_MSG_CTX);
            return false;
        }
        
        if (claimsSet instanceof IDTokenClaimsSet) {
            final Object msg = profileRequestContext.getInboundMessageContext().getMessage();
            if (msg instanceof AuthenticationRequest) {
                final ResponseType responseType = ((AuthenticationRequest) msg).getResponseType();
                addToIDTokenByDefault = responseType.contains(OIDCResponseTypeValue.ID_TOKEN)
                        && responseType.size() == 1;
            }

            alwaysIncludedAttributes = alwaysIncludedAttributesLookupStrategy.apply(profileRequestContext);
            if (alwaysIncludedAttributes == null) {
                alwaysIncludedAttributes = Collections.emptySet();
            }
            deniedUserInfoAttributes = Collections.emptySet();
        } else if (claimsSet instanceof UserInfo) {
            deniedUserInfoAttributes = deniedUserInfoAttributesLookupStrategy.apply(profileRequestContext);
            if (deniedUserInfoAttributes == null) {
                deniedUserInfoAttributes = Collections.emptySet();
            }
            alwaysIncludedAttributes = Collections.emptySet();
        }
        return true;
    }

// Checkstyle: CyclomaticComplexity OFF
    /** {@inheritDoc} */
    @Override
    protected void doExecute(@Nonnull final ProfileRequestContext profileRequestContext) {

        final List<JSONObject> claims = new ArrayList<>();
        
        final OIDCAuthenticationResponseConsentContext consentCtx =
                consentContextLookupStrategy.apply(profileRequestContext);
        
        ServiceableComponent<AttributeTranscoderRegistry> component = null;
        try {
            component = transcoderRegistry.getServiceableComponent();
            if (component == null) {
                log.error("Attribute transoding service unavailable");
                throw new AttributeEncodingException("Attribute transoding service unavailable");
            }
            
            for (final IdPAttribute attribute : attributeCtx.getIdPAttributes().values()) {
                if (attribute != null && !attribute.getValues().isEmpty()) {
                    if (consentCtx != null && !consentCtx.getConsentedAttributes().contains(attribute.getId())) {
                        log.debug("{} Consentable attribute {} has no consent. Not added to claims set",
                                getLogPrefix(), attribute.getId());
                    } else {
                        encodeAttribute(component.getComponent(), profileRequestContext, attribute, claims);
                    }
                }
            }
        } catch (final AttributeEncodingException e) {
            ActionSupport.buildEvent(profileRequestContext, IdPEventIds.UNABLE_ENCODE_ATTRIBUTE);
            return;
        } finally {
            if (null != component) {
                component.unpinComponent();
            }
        }
        
        for (final JSONObject claim : claims) {
            for (final String name : claim.keySet()) {
                if (reservedClaimNames != null && reservedClaimNames.contains(name)) {
                    log.debug("{} Claim has a reserved name ({}), not added to claims set", getLogPrefix(), name);
                    continue;
                }
                
                log.debug("{} Adding claim {} with value {}", getLogPrefix(), name, claim.get(name));
                claimsSet.setClaim(name, claim.get(name));
            }
        }
        
        log.debug("{} Claims set after mapping attributes to claims: {}", getLogPrefix(),
                claimsSet.toJSONObject().toJSONString());
    }

    /**
     * Access the registry of transcoding rules to transform the input attribute into claims.
     * 
     * @param registry  registry of transcoding rules
     * @param profileRequestContext current profile request context
     * @param attribute input attribute
     * @param results collection to add results to
     * 
     * @throws AttributeEncodingException if a non-ignorable error occurs
     */
    private void encodeAttribute(@Nonnull final AttributeTranscoderRegistry registry,
            @Nonnull final ProfileRequestContext profileRequestContext, @Nonnull final IdPAttribute attribute,
            @Nonnull @NonnullElements @Live final Collection<JSONObject> results)
                    throws AttributeEncodingException {
        
        final Collection<TranscodingRule> transcodingRules = registry.getTranscodingRules(attribute, JSONObject.class);
        if (transcodingRules.isEmpty()) {
            log.debug("{} Attribute {} does not have any transcoding rules, nothing to do", getLogPrefix(),
                    attribute.getId());
            return;
        }
        
        for (final TranscodingRule rule : transcodingRules) {
            try {
                // Check for claims to skip based on token type.
                if (claimsSet instanceof IDTokenClaimsSet && !addToIDTokenByDefault
                        && !alwaysIncludedAttributes.contains(attribute.getId())) {
                    final AttributesMapContainer container = getOidcResponseContext().getMappedIdTokenRequestedClaims();
                    if (container != null && container.get().containsKey(attribute.getId())) {
                        log.debug("{} Attribute {} is targeted for ID Token via claims request", getLogPrefix(),
                                attribute.getId());
                    } else {
                        log.debug("{} Attribute {} not targeted for ID Token", getLogPrefix(), attribute.getId());
                        continue;
                    }
                } else if (claimsSet instanceof UserInfo && deniedUserInfoAttributes.contains(attribute.getId())) {
                    log.debug("{} Attribute {} not targeted for Userinfo Token", getLogPrefix(), attribute.getId());
                    continue;
                }
                
                final AttributeTranscoder<JSONObject> transcoder = TranscoderSupport.<JSONObject>getTranscoder(rule);
                final JSONObject encodedAttribute =
                        transcoder.encode(profileRequestContext, attribute, JSONObject.class, rule);
                if (encodedAttribute != null) {
                    results.add(encodedAttribute);
                }
            } catch (final AttributeEncodingException e) {
                log.warn("{} Unable to encode attribute {}", getLogPrefix(), attribute.getId(), e);
                if (!ignoringUnencodableAttributes) {
                    throw e;
                }
            }
        }
    }
// Checkstyle: CyclomaticComplexity ON

}