/*
 * Licensed to the University Corporation for Advanced Internet Development,
 * Inc. (UCAID) under one or more contributor license agreements.  See the
 * NOTICE file distributed with this work for additional information regarding
 * copyright ownership. The UCAID licenses this file to You under the Apache
 * License, Version 2.0 (the "License"); you may not use this file except in
 * compliance with the License.  You may obtain a copy of the License at
 *
 *    http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package net.shibboleth.idp.plugin.oidc.op.oauth2.profile.impl;

import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import java.util.function.Function;
import java.util.function.Predicate;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;

import org.opensaml.messaging.context.navigate.ChildContextLookup;
import org.opensaml.profile.action.ActionSupport;
import org.opensaml.profile.action.EventIds;
import org.opensaml.profile.context.ProfileRequestContext;
import org.opensaml.profile.context.ProxiedRequesterContext;
import org.opensaml.profile.context.navigate.OutboundMessageContextLookup;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import net.shibboleth.idp.plugin.oidc.op.messaging.context.OIDCAuthenticationResponseContext;
import net.shibboleth.idp.plugin.oidc.op.messaging.context.OIDCAuthenticationResponseTokenClaimsContext;
import net.shibboleth.idp.plugin.oidc.op.profile.context.navigate.ClientInfoAudienceLookupFunction;
import net.shibboleth.idp.plugin.oidc.op.profile.context.navigate.DefaultOIDCMetadataContextLookupFunction;
import net.shibboleth.idp.plugin.oidc.op.profile.context.navigate.TokenRequestAudienceLookupFunction;
import net.shibboleth.idp.plugin.oidc.op.profile.impl.AbstractOIDCAuthenticationResponseAction;
import net.shibboleth.idp.plugin.oidc.op.profile.logic.IssueIDTokenCondition;
import net.shibboleth.idp.profile.context.navigate.RelyingPartyIdLookupFunction;
import net.shibboleth.oidc.profile.core.OidcEventIds;
import net.shibboleth.utilities.java.support.component.ComponentSupport;
import net.shibboleth.utilities.java.support.logic.Constraint;

/**
 * Action that validates requested resource/audience values against a computed set of "allowed"
 * values and populates the resulting set into the {@link OIDCAuthenticationResponseContext}
 * and a {@link ProxiedRequesterContext}.
 * 
 * <p>This is an ordered list, so the first allowed value determines the "primary" audience of
 * the eventual token.</p>
 * 
 * <p>Requesting values is optional. If the OP is an implied audience, then no other audience will
 * be established, but if not then at least one audience must be permitted and the first permitted
 * value will be assumed.</p>
 * 
 * @event {@link EventIds#PROCEED_EVENT_ID}
 * @event {@link EventIds#INVALID_PROFILE_CTX}
 * @event {@link OidcEventIds#INVALID_TARGET}
 */
public class ValidateAudience extends AbstractOIDCAuthenticationResponseAction {

    /** Class logger. */
    @Nonnull private Logger log = LoggerFactory.getLogger(ValidateAudience.class);

    /** Strategy used to obtain the relying party ID. */
    @Nonnull private Function<ProfileRequestContext,String> relyingPartyIdLookupStrategy;
    
    /** Strategy used to obtain the requested audience. */
    @Nullable private Function<ProfileRequestContext,List<String>> requestedAudienceLookupStrategy;

    /** Strategy used to obtain the audience allowed for the client. */
    @Nonnull private Function<ProfileRequestContext,List<String>> allowedAudienceLookupStrategy;

    /** Strategy used for locating/creating the proxy context. */
    @Nonnull private Function<ProfileRequestContext,ProxiedRequesterContext> proxiedRequesterContextCreationStrategy;
    
    /** Whether the request includes the OP as an audience. */
    @Nonnull private Predicate<ProfileRequestContext> selfAudienceCondition;
    
    /** Strategy used to locate the {@link OIDCAuthenticationResponseTokenClaimsContext}. */
    @Nonnull
    private Function<ProfileRequestContext,OIDCAuthenticationResponseTokenClaimsContext>
    tokenClaimsContextLookupStrategy;
    
    /** Constructor. */
    public ValidateAudience() {
        requestedAudienceLookupStrategy = new TokenRequestAudienceLookupFunction();
        relyingPartyIdLookupStrategy = new RelyingPartyIdLookupFunction();
        allowedAudienceLookupStrategy = new ClientInfoAudienceLookupFunction().compose(
                new DefaultOIDCMetadataContextLookupFunction());
        proxiedRequesterContextCreationStrategy = new ChildContextLookup<>(ProxiedRequesterContext.class, true).compose(
                new OutboundMessageContextLookup());
        // openid scope -> we're issuing an ID token -> the OP will be an audience for the access token
        selfAudienceCondition = new IssueIDTokenCondition();
    }

    /**
     * Set the strategy used to obtain the relying party ID.
     * 
     * @param strategy lookup strategy
     */
    public void setRelyingPartyIdLookupStrategy(@Nonnull final Function<ProfileRequestContext,String> strategy) {
        ComponentSupport.ifInitializedThrowUnmodifiabledComponentException(this);
        
        relyingPartyIdLookupStrategy = Constraint.isNotNull(strategy,
                "Relying party ID lookup strategy cannot be null");
    }
    
    /**
     * Set the strategy used to locate the requested audience to validate.
     * 
     * @param strategy lookup strategy
     */
    public void setRequestedAudienceLookupStrategy(
            @Nullable final Function<ProfileRequestContext,List<String>> strategy) {
        ComponentSupport.ifInitializedThrowUnmodifiabledComponentException(this);
        
        requestedAudienceLookupStrategy = strategy;
    }


    /**
     * Set the strategy used to locate the allowed audience for the client.
     * 
     * @param strategy lookup strategy
     */
    public void setAllowedAudienceLookupStrategy(@Nonnull final Function<ProfileRequestContext,List<String>> strategy) {
        ComponentSupport.ifInitializedThrowUnmodifiabledComponentException(this);
        
        allowedAudienceLookupStrategy = Constraint.isNotNull(strategy,
                "Allowed scope lookup strategy cannot be null");
    }
    
    /**
     * Set the strategy used to locate or create the ProxiedRequesterContext.
     * 
     * @param strategy lookup/creation strategy
     */
    public void setProxiedRequesterContextCreationStrategy(
            @Nonnull final Function<ProfileRequestContext,ProxiedRequesterContext> strategy) {
        ComponentSupport.ifInitializedThrowUnmodifiabledComponentException(this);
        
        proxiedRequesterContextCreationStrategy = Constraint.isNotNull(strategy,
                "ProxiedRequesterContext lookup strategy cannot be null");
    }
    
    /**
     * Set whether the OP is an implied audience for the token request.
     * 
     * @param condition condition to set
     * 
     * @since 3.2.0
     */
    public void setSelfAudienceCondition(@Nonnull final Predicate<ProfileRequestContext> condition) {
        ComponentSupport.ifInitializedThrowUnmodifiabledComponentException(this);
        
        selfAudienceCondition = Constraint.isNotNull(condition, "Self audience condition cannot be null");
    }
    
// Checkstyle: CyclomaticComplexity|MethodLength OFF
    /** {@inheritDoc} */
    @Override
    protected void doExecute(@Nonnull final ProfileRequestContext profileRequestContext) {
        
        final String clientId = relyingPartyIdLookupStrategy.apply(profileRequestContext);
        final boolean allowNone = selfAudienceCondition.test(profileRequestContext);
        
        // These may come from metadata or be supplemented or substituted from elsewhere.
        final List<String> allowedAudience = allowedAudienceLookupStrategy.apply(profileRequestContext);
        
        // These come from a previous authorization grant (authz code or refresh token).
        List<String> previouslyGrantedAudience = null;
        if (getOidcResponseContext().getAuthorizationGrantClaimsSet() != null) {
            previouslyGrantedAudience = getOidcResponseContext().getAuthorizationGrantClaimsSet().getAudience();
        }

        // These come from a request object or parameter. Absent by definition on the UserInfo endpoint.
        List<String> requestedAudience = requestedAudienceLookupStrategy != null ?
                requestedAudienceLookupStrategy.apply(profileRequestContext) : null;

        if (allowedAudience == null || allowedAudience.isEmpty()) {
            if (allowNone) {
                if (previouslyGrantedAudience != null || requestedAudience != null) {
                    log.warn("{} No allowed audiences for client {}, OP will be sole audience", getLogPrefix(),
                            clientId);
                } else {
                    log.debug("{} No allowed audiences for client {}, OP will be sole audience", getLogPrefix(),
                            clientId);
                }
            } else {
                log.warn("{} No allowed audience for client {}", getLogPrefix(), clientId);
                ActionSupport.buildEvent(profileRequestContext, OidcEventIds.INVALID_TARGET);
            }
            return;
        }

        if (requestedAudience == null) {
            // With none requested, simply swap requested for previously granted, if any.
            // Set previous set to null since there's no need to filter against it.
            requestedAudience = previouslyGrantedAudience;
            previouslyGrantedAudience = null;
        }
        
        if (requestedAudience == null) {
            // Nothing requested or previously granted.
            if (allowNone) {
                log.debug("{} No audience in request for client {}, OP will be sole audience", getLogPrefix(),
                        clientId);
                return;
            }
            log.debug("{} No audience in request for client {}, using first allowed", getLogPrefix(), clientId);
            requestedAudience = Collections.singletonList(allowedAudience.get(0));
        }
        
        final List<String> effectiveAudience = new ArrayList<>();
        
        for (final String aud : requestedAudience) {
            if (!allowedAudience.contains(aud)) {
                log.warn("{} Omitting requested but unregistered audience {} for RP {}", getLogPrefix(), aud,
                        clientId);
            } else if (previouslyGrantedAudience != null && !previouslyGrantedAudience.contains(aud)) {
                log.warn("{} Omitting requested but previously ungranted audience {} for RP {}", getLogPrefix(),
                        aud, clientId);
            } else {
                effectiveAudience.add(aud);
            }
        }
        
        if (effectiveAudience.isEmpty()) {
            if (allowNone) {
                log.debug("{} No allowed audience for client {}, OP will be sole audience", getLogPrefix(), clientId);
            } else {
                log.warn("{} No allowed audience for client {}", getLogPrefix(), clientId);
                ActionSupport.buildEvent(profileRequestContext, OidcEventIds.INVALID_TARGET);
            }
            return;
        }
        
        log.debug("{} Computed audience for client {}: {}", getLogPrefix(), clientId, effectiveAudience);
        getOidcResponseContext().getAudience().addAll(effectiveAudience);
        
        final ProxiedRequesterContext proxyCtx = proxiedRequesterContextCreationStrategy.apply(profileRequestContext);
        if (proxyCtx == null) {
            log.error("{} Unable to locate/create ProxiedRequesterContext", getLogPrefix());
            ActionSupport.buildEvent(profileRequestContext, EventIds.INVALID_PROFILE_CTX);
            return;
        }
        
        proxyCtx.getRequesters().addAll(effectiveAudience);
    }
// Checkstyle: CyclomaticComplexity|MethodLength ON
    
}