/*
 * Licensed to the University Corporation for Advanced Internet Development,
 * Inc. (UCAID) under one or more contributor license agreements.  See the
 * NOTICE file distributed with this work for additional information regarding
 * copyright ownership. The UCAID licenses this file to You under the Apache
 * License, Version 2.0 (the "License"); you may not use this file except in
 * compliance with the License.  You may obtain a copy of the License at
 *
 *    http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package net.shibboleth.idp.plugin.oidc.op.oauth2.profile.impl;

import java.text.ParseException;
import java.time.Duration;
import java.time.Instant;
import java.util.HashMap;
import java.util.Map;
import java.util.function.BiFunction;
import java.util.function.Function;
import java.util.function.Predicate;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;

import org.opensaml.profile.action.EventIds;
import org.opensaml.profile.context.ProfileRequestContext;
import org.opensaml.profile.context.navigate.InboundMessageContextLookup;
import org.opensaml.profile.context.navigate.OutboundMessageContextLookup;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.nimbusds.jwt.JWTClaimsSet;
import com.nimbusds.jwt.PlainJWT;
import com.nimbusds.oauth2.sdk.Scope;
import com.nimbusds.oauth2.sdk.id.ClientID;
import com.nimbusds.openid.connect.sdk.AuthenticationRequest;
import com.nimbusds.openid.connect.sdk.claims.ClaimsSet;

import net.minidev.json.JSONArray;
import net.shibboleth.idp.authn.context.SubjectContext;
import net.shibboleth.idp.plugin.oidc.op.messaging.context.AccessTokenContext;
import net.shibboleth.idp.plugin.oidc.op.messaging.context.OIDCAuthenticationResponseConsentContext;
import net.shibboleth.idp.plugin.oidc.op.messaging.context.OIDCAuthenticationResponseContext;
import net.shibboleth.idp.plugin.oidc.op.messaging.context.OIDCAuthenticationResponseTokenClaimsContext;
import net.shibboleth.idp.plugin.oidc.op.profile.context.navigate.OIDCAuthenticationResponseContextLookupFunction;
import net.shibboleth.idp.plugin.oidc.op.profile.context.navigate.TokenRequestClientIDLookupFunction;
import net.shibboleth.idp.plugin.oidc.op.profile.impl.AbstractOIDCResponseAction;
import net.shibboleth.idp.plugin.oidc.op.token.support.AccessTokenClaimsSet;
import net.shibboleth.idp.plugin.oidc.op.token.support.AuthorizeCodeClaimsSet;
import net.shibboleth.idp.plugin.oidc.op.token.support.RefreshTokenClaimsSet;
import net.shibboleth.idp.plugin.oidc.op.token.support.TokenClaimsSet;
import net.shibboleth.idp.plugin.oidc.op.token.support.AccessTokenClaimsSet.Builder;
import net.shibboleth.idp.profile.IdPEventIds;
import net.shibboleth.idp.profile.context.navigate.ResponderIdLookupFunction;
import net.shibboleth.oidc.profile.config.logic.AttributeConsentFlowEnabledPredicate;
import net.shibboleth.oidc.profile.config.navigate.AccessTokenClaimsSetManipulationStrategyLookupFunction;
import net.shibboleth.oidc.profile.config.navigate.AccessTokenLifetimeLookupFunction;
import net.shibboleth.oidc.profile.config.navigate.AccessTokenTypeLookupFunction;

import org.opensaml.messaging.context.navigate.ChildContextLookup;
import org.opensaml.profile.action.ActionSupport;

import net.shibboleth.utilities.java.support.annotation.constraint.NonnullAfterInit;
import net.shibboleth.utilities.java.support.component.ComponentInitializationException;
import net.shibboleth.utilities.java.support.component.ComponentSupport;
import net.shibboleth.utilities.java.support.logic.Constraint;
import net.shibboleth.utilities.java.support.logic.FunctionSupport;
import net.shibboleth.utilities.java.support.primitive.StringSupport;
import net.shibboleth.utilities.java.support.security.DataSealer;
import net.shibboleth.utilities.java.support.security.DataSealerException;
import net.shibboleth.utilities.java.support.security.IdentifierGenerationStrategy;
import net.shibboleth.utilities.java.support.security.impl.SecureRandomIdentifierGenerationStrategy;

/**
 * Action that creates an Access Token, and stores it to an {@link AccessTokenContext}.
 *
 * <p>There are various cases handled across different grant types and orders of operation.
 * The token may be produced solely for a third-party service to consume, or may also or instead
 * be usable with the OP's UserInfo endpoint.</p>
 * 
 * <p>The action supports either opaque access tokens sealed under the IdP's secret key, or the
 * RFC 9068 standard for JWT-based tokens.</p>
 * 
 * @event {@link EventIds#PROCEED_EVENT_ID}
 * @event {@link EventIds#MESSAGE_PROC_ERROR}
 * @event {@link EventIds#INVALID_MSG_CTX}
 * @event {@link EventIds#INVALID_PROFILE_CTX}
 * @event {@link EventIds#MESSAGE_PROC_ERROR}
 * @event {@link IdPEventIds#INVALID_ATTRIBUTE_CTX}
 * @event {@link IdPEventIds#INVALID_PROFILE_CONFIG}
 * @event {@link IdPEventIds#INVALID_SUBJECT_CTX}
 * 
 * @since 3.1.0
 */
public class BuildAccessToken extends AbstractOIDCResponseAction {

    /** Class logger. */
    @Nonnull private Logger log = LoggerFactory.getLogger(BuildAccessToken.class);

    /** Sealer to use for opaque tokens. */
    @NonnullAfterInit private DataSealer dataSealer;
    
    /** Strategy used to obtain the response issuer value. */
    @Nonnull private Function<ProfileRequestContext,String> issuerLookupStrategy;
    
    /** Strategy used to obtain the original client ID. */
    @Nonnull private Function<ProfileRequestContext,ClientID> clientIDLookupStrategy;

    /** Strategy used to obtain the access token type to issue. */
    @Nonnull private Function<ProfileRequestContext,String> accessTokenTypeLookupStrategy;    
    
    /** Strategy used to obtain the access token lifetime. */
    @Nonnull private Function<ProfileRequestContext,Duration> accessTokenLifetimeLookupStrategy;

    /** Strategy used to locate the {@link IdentifierGenerationStrategy} to use. */
    @Nonnull private Function<ProfileRequestContext,IdentifierGenerationStrategy> idGeneratorLookupStrategy;

    /** Strategy used to locate the {@link OIDCAuthenticationResponseTokenClaimsContext}. */
    @Nonnull private Function<ProfileRequestContext,OIDCAuthenticationResponseTokenClaimsContext>
        tokenClaimsContextLookupStrategy;
    
    /** Strategy used to locate the {@link OIDCAuthenticationResponseConsentContext}. */
    @Nonnull private Function<ProfileRequestContext, OIDCAuthenticationResponseConsentContext>
        consentContextLookupStrategy;
    
    /** Predicate used to check if consent is enabled with a given {@link ProfileRequestContext}. */
    @Nonnull private Predicate<ProfileRequestContext> consentEnabledPredicate;
    
    /** Strategy used to create the subcontext to hold the token. */
    @Nonnull private Function<ProfileRequestContext,AccessTokenContext> accessTokenContextCreationStrategy;

    /** Lookup function to supply strategy bi-function for manipulating token claims set. */ 
    @Nonnull
    private Function<ProfileRequestContext,BiFunction<ProfileRequestContext,Map<String,Object>,Map<String,Object>>>
        tokenClaimsSetManipulationStrategyLookupStrategy;

    /** The strategy used for manipulating the token claims set. */
    @Nullable private BiFunction<ProfileRequestContext,Map<String,Object>,Map<String,Object>> manipulationStrategy;

    /** Authorize Code / Refresh Token the access token is based on, if any. */
    @Nullable private TokenClaimsSet tokenClaimsSet;

    /** Authentication request in the case of such. */
    @Nullable private AuthenticationRequest authenticationRequest;
    
    /** Subject context. */
    @Nullable private SubjectContext subjectCtx;
    
    /** Use a JWT for the token. */
    private boolean jwtTokenType;
    
    /** The generator to use. */
    @Nullable private IdentifierGenerationStrategy idGenerator;
    
    /** Access token context. */
    @Nullable private AccessTokenContext accessTokenCtx;

    /** Constructor. */
    public BuildAccessToken() {
        accessTokenTypeLookupStrategy = new AccessTokenTypeLookupFunction();
        accessTokenLifetimeLookupStrategy = new AccessTokenLifetimeLookupFunction();
        issuerLookupStrategy = new ResponderIdLookupFunction();
        
        clientIDLookupStrategy = FunctionSupport.compose(new TokenRequestClientIDLookupFunction(),
                new InboundMessageContextLookup());
        
        idGeneratorLookupStrategy = FunctionSupport.constant(new SecureRandomIdentifierGenerationStrategy());
        
        tokenClaimsContextLookupStrategy =
                new ChildContextLookup<>(OIDCAuthenticationResponseTokenClaimsContext.class).compose(
                        new OIDCAuthenticationResponseContextLookupFunction());
        consentContextLookupStrategy =
                new ChildContextLookup<>(OIDCAuthenticationResponseConsentContext.class).compose(
                        new OIDCAuthenticationResponseContextLookupFunction());

        consentEnabledPredicate = new AttributeConsentFlowEnabledPredicate();
        
        // PRC -> inbound message context -> OIDC response context -> ATC
        accessTokenContextCreationStrategy = new ChildContextLookup<>(AccessTokenContext.class, true).compose(
                new ChildContextLookup<>(OIDCAuthenticationResponseContext.class).compose(
                        new OutboundMessageContextLookup()));
        tokenClaimsSetManipulationStrategyLookupStrategy =
                new AccessTokenClaimsSetManipulationStrategyLookupFunction();
    }
    
    /**
     * Set {@link DataSealer} to use for opaque tokens.
     * 
     * @param sealer sealer to use for opaque tokens
     */
    public void setDataSealer(@Nullable final DataSealer sealer) {
        ComponentSupport.ifInitializedThrowUnmodifiabledComponentException(this);
        
        dataSealer = sealer;
    }
    
   /**
    * Set the strategy used to obtain the access token type.
    * 
    * @param strategy lookup strategy
    */
    public void setAccessTokenTypeLookupStrategy(@Nonnull final Function<ProfileRequestContext,String> strategy) {
        ComponentSupport.ifInitializedThrowUnmodifiabledComponentException(this);
       
        accessTokenTypeLookupStrategy =
                Constraint.isNotNull(strategy, "Access token type lookup strategy cannot be null");
    }
    
    /**
     * Set the strategy used to obtain the access token lifetime.
     * 
     * @param strategy lookup strategy
     */
    public void setAccessTokenLifetimeLookupStrategy(@Nonnull final Function<ProfileRequestContext,Duration> strategy) {
        ComponentSupport.ifInitializedThrowUnmodifiabledComponentException(this);
        
        accessTokenLifetimeLookupStrategy =
                Constraint.isNotNull(strategy, "Access token lifetime lookup strategy cannot be null");
    }

    /**
     * Set the strategy used to locate the {@link IdentifierGenerationStrategy} to use.
     * 
     * @param strategy lookup strategy
     */
    public void setIdentifierGeneratorLookupStrategy(
            @Nonnull final Function<ProfileRequestContext,IdentifierGenerationStrategy> strategy) {
        ComponentSupport.ifInitializedThrowUnmodifiabledComponentException(this);

        idGeneratorLookupStrategy =
                Constraint.isNotNull(strategy, "Identifier generation strategy cannot be null");
    }
    
    /**
     * Set the strategy used to locate the issuer value to use.
     * 
     * @param strategy lookup strategy
     */
    public void setIssuerLookupStrategy(@Nonnull final Function<ProfileRequestContext,String> strategy) {
        ComponentSupport.ifInitializedThrowUnmodifiabledComponentException(this);
        
        issuerLookupStrategy = Constraint.isNotNull(strategy, "Issuer lookup strategy cannot be null");
    }
    
    /**
     * Set the strategy used to locate the original {@link ClientID} from the request.
     * 
     * @param strategy lookup strategy
     */
    public void setClientIDLookupStrategy(@Nonnull final Function<ProfileRequestContext,ClientID> strategy) {
        ComponentSupport.ifInitializedThrowUnmodifiabledComponentException(this);
        
        clientIDLookupStrategy = Constraint.isNotNull(strategy, "ClientID lookup strategy cannot be null");
    }
    
    /**
     * Set the strategy used to locate the {@link OIDCAuthenticationResponseTokenClaimsContext} associated with a given
     * {@link ProfileRequestContext}.
     * 
     * @param strategy lookup strategy
     */
    public void setOIDCAuthenticationResponseTokenClaimsContextLookupStrategy(
            @Nonnull final Function<ProfileRequestContext,OIDCAuthenticationResponseTokenClaimsContext> strategy) {
        ComponentSupport.ifInitializedThrowUnmodifiabledComponentException(this);
        tokenClaimsContextLookupStrategy = Constraint.isNotNull(strategy,
                "OIDCAuthenticationResponseTokenClaimsContextt lookup strategy cannot be null");
    }

    /**
     * Set the strategy used to locate the {@link OIDCAuthenticationResponseConsentContext} associated with a given
     * {@link ProfileRequestContext}.
     * 
     * @param strategy lookup strategy
     */
    public void setOIDCAuthenticationResponseConsentContextLookupStrategy(
            @Nonnull final Function<ProfileRequestContext,OIDCAuthenticationResponseConsentContext> strategy) {
        ComponentSupport.ifInitializedThrowUnmodifiabledComponentException(this);
        consentContextLookupStrategy = Constraint.isNotNull(strategy,
                "OIDCAuthenticationResponseConsentContext lookup strategy cannot be null");
    }

    /**
     * Set the predicate used to check if consent is enabled with a given {@link ProfileRequestContext}.
     * 
     * @param predicate predicate used to check if consent is enabled with a given {@link ProfileRequestContext}.
     */
    public void setConsentEnabledPredicate(@Nonnull final Predicate<ProfileRequestContext> predicate) {
        ComponentSupport.ifInitializedThrowUnmodifiabledComponentException(this);

        consentEnabledPredicate =
                Constraint.isNotNull(predicate, "predicate used to check if consent is enabled cannot be null");
    }
    
    /**
     * Set the strategy used to create the {@link AccessTokenContext} to use.
     * 
     * @param strategy creation strategy
     */
    public void setAccessTokenContextCreationStrategy(
            @Nonnull final Function<ProfileRequestContext,AccessTokenContext> strategy) {
        ComponentSupport.ifInitializedThrowUnmodifiabledComponentException(this);
        
        accessTokenContextCreationStrategy =
                Constraint.isNotNull(strategy, "AccessTokenContext creation strategy cannot be null");
    }
    
    /**
     * Set the lookup function to supply strategy bi-function for manipulating token claims set.
     * 
     * @param strategy What to set
     */
    public void setTokenClaimsSetManipulationStrategyLookupStrategy(@Nonnull final
            Function<ProfileRequestContext,BiFunction<ProfileRequestContext,Map<String,Object>,Map<String,Object>>>
            strategy) {
        ComponentSupport.ifInitializedThrowUnmodifiabledComponentException(this);

        tokenClaimsSetManipulationStrategyLookupStrategy =
                Constraint.isNotNull(strategy, "Manipulation strategy lookup strategy cannot be null");
    }

    /** {@inheritDoc} */
    @Override
    protected void doInitialize() throws ComponentInitializationException {
        super.doInitialize();
        
        if (dataSealer == null) {
            throw new ComponentInitializationException("DataSealer cannot be null");
        }
    }

 // Checkstyle: CyclomaticComplexity|MethodLength OFF
    /** {@inheritDoc} */
    @Override
    protected boolean doPreExecute(@Nonnull final ProfileRequestContext profileRequestContext) {
        if (!super.doPreExecute(profileRequestContext)) {
            return false;
        }
        
        final String tokenType = accessTokenTypeLookupStrategy.apply(profileRequestContext);
        jwtTokenType = tokenType != null && "JWT".equals(tokenType);
        
        if (!jwtTokenType && dataSealer == null) {
            log.error("{} DataSealer required for opaque access tokens", getLogPrefix());
            ActionSupport.buildEvent(profileRequestContext, EventIds.UNABLE_TO_ENCRYPT);
            return false;
        }
        
        tokenClaimsSet = getOidcResponseContext().getAuthorizationGrantClaimsSet();
        if (tokenClaimsSet != null && !(tokenClaimsSet instanceof RefreshTokenClaimsSet)
                && !(tokenClaimsSet instanceof AuthorizeCodeClaimsSet)) {
            log.error("{} Authorization grant is of unknown type: {}", getLogPrefix(),
                    tokenClaimsSet.getClass().getName());
            ActionSupport.buildEvent(profileRequestContext, EventIds.INVALID_PROFILE_CTX);
            return false;
        }

        idGenerator = idGeneratorLookupStrategy.apply(profileRequestContext);
        if (idGenerator == null) {
            log.error("{} No identifier generation strategy", getLogPrefix());
            ActionSupport.buildEvent(profileRequestContext, EventIds.INVALID_PROFILE_CTX);
            return false;
        }

        if (tokenClaimsSet == null) {
            /*
             * Typically this path applies when the client_credentials grant is used.
             * 
             * Alternatively the access token may be provided by the authz endpoint without a code.
             * This is the case only with the "token id_token" response type.
             */
            subjectCtx = profileRequestContext.getSubcontext(SubjectContext.class);
            if (subjectCtx == null) {
                log.error("{} No subject context", getLogPrefix());
                ActionSupport.buildEvent(profileRequestContext, EventIds.INVALID_PROFILE_CTX);
                return false;
            }
            
            if (profileRequestContext.getInboundMessageContext() != null
                    && profileRequestContext.getInboundMessageContext().getMessage() instanceof AuthenticationRequest) {
                authenticationRequest =
                        (AuthenticationRequest) profileRequestContext.getInboundMessageContext().getMessage();
            }
        }
        
        accessTokenCtx = accessTokenContextCreationStrategy.apply(profileRequestContext);
        if (accessTokenCtx == null) {
            log.error("{} Unable to create AccessTokenContext", getLogPrefix());
            ActionSupport.buildEvent(profileRequestContext, EventIds.INVALID_PROFILE_CTX);
            return false;
        }
        
        final Duration lifetime = accessTokenLifetimeLookupStrategy.apply(profileRequestContext);
        if (lifetime == null) {
            log.error("{} No lifetime supplied for access token", getLogPrefix());
            ActionSupport.buildEvent(profileRequestContext, IdPEventIds.INVALID_PROFILE_CONFIG);
            return false;
        }
        accessTokenCtx.setLifetime(lifetime);

        manipulationStrategy = tokenClaimsSetManipulationStrategyLookupStrategy.apply(profileRequestContext);

        return true;
    }

    /** {@inheritDoc} */
    @Override
    protected void doExecute(@Nonnull final ProfileRequestContext profileRequestContext) {

        final String issuer = issuerLookupStrategy.apply(profileRequestContext);
        final ClientID clientID = clientIDLookupStrategy.apply(profileRequestContext);
        if (issuer == null || clientID == null) {
            log.error("{} Unable to determine issuer or clientID, failing request", getLogPrefix());
            ActionSupport.buildEvent(profileRequestContext, EventIds.MESSAGE_PROC_ERROR);
            return;
        }
        
        ClaimsSet claims = null;
        ClaimsSet claimsUI = null;
        final OIDCAuthenticationResponseTokenClaimsContext tokenClaimsCtx =
                tokenClaimsContextLookupStrategy.apply(profileRequestContext);
        if (tokenClaimsCtx != null) {
            claims = tokenClaimsCtx.getClaims();
            claimsUI = tokenClaimsCtx.getUserinfoClaims();
        }
        
        final OIDCAuthenticationResponseContext responseCtx = getOidcResponseContext();

        final Scope scope = responseCtx.getScope() != null ? responseCtx.getScope() : new Scope();
        log.debug("{} Building access token with scope: {}", getLogPrefix(), scope);
        
        final boolean oidc = scope.contains("openid");
        
        if (oidc) {
            responseCtx.getAudience().add(issuer);
        }
        log.debug("{} Building access token with audience: {}", getLogPrefix(), responseCtx.getAudience());

        final Instant now = Instant.now();
        final Instant dateExp = now.plus(accessTokenCtx.getLifetime());
        
        final AccessTokenClaimsSet.Builder builder;
        
        if (tokenClaimsSet != null) {
            // We may not use original claims as input for scope / delivery claims as they may have been reduced.
            builder = new AccessTokenClaimsSet.Builder(
                    tokenClaimsSet,
                    scope,
                    oidc ? claims : null,
                    oidc ? claimsUI : null,
                    Instant.now(),
                    dateExp);
            // Add additional bits.
            builder.setAudience(responseCtx.getAudience());
            builder.setJWTID(idGenerator);
            // Set root token identifier to contain jit from the claims set used for building the new token
            if (StringSupport.trimOrNull(tokenClaimsSet.getRootTokenIdentifier()) == null) {
                builder.setRootTokenIdentifier(tokenClaimsSet.getID());
            }
        } else {
            final OIDCAuthenticationResponseConsentContext consentCtx =
                    consentContextLookupStrategy.apply(profileRequestContext);
            final JSONArray consented = consentCtx != null ? consentCtx.getConsentedAttributes() : null;
            
            builder = (Builder) new AccessTokenClaimsSet.Builder()
                    .setJWTID(idGenerator)
                    .setClientID(clientID)
                    .setIssuer(issuer)
                    .setPrincipal(subjectCtx.getPrincipalName())
                    .setSubject(responseCtx.getSubject())
                    .setIssuedAt(now)
                    .setExpiresAt(dateExp)
                    .setACR(responseCtx.getAcr())
                    .setAuthenticationTime(responseCtx.getAuthTime())
                    .setScope(scope)
                    .setAudience(responseCtx.getAudience())
                    .setDlClaims(claims)
                    .setDlClaimsUI(claimsUI)
                    .setConsentedClaims(consented)
                    .setConsentEnabled(consentEnabledPredicate.test(profileRequestContext));

            if (authenticationRequest != null) {
                builder
                    .setNonce(authenticationRequest.getNonce())
                    .setClaimsRequest(authenticationRequest.getOIDCClaims());
            }
        }
        
        if (jwtTokenType && responseCtx.getAccessTokenClaimSet() != null) {
            builder.setCustomClaims(responseCtx.getAccessTokenClaimSet().toJSONObject());
        }
        
        final AccessTokenClaimsSet claimsSet = builder.build();

        if (manipulationStrategy != null) {
            log.debug("{} Manipulation strategy has been set, applying it to the claims set {}", getLogPrefix(),
                    claimsSet.serialize());
            final Map<String, Object> result = manipulationStrategy.apply(profileRequestContext,
                    claimsSet.getClaimsSet().toJSONObject());
            if (result == null) {
                log.debug("{} Manipulation strategy returned null, leaving token claims set untouched.",
                        getLogPrefix());
            } else {
                log.debug("{} Applying the manipulated claims into the token claims set", getLogPrefix());
                try {
                    claimsSet.setClaimsSet(JWTClaimsSet.parse(result));
                } catch (final ParseException e) {
                    log.error("{} The resulted claims set could not be transformed into ", getLogPrefix(), e);
                    ActionSupport.buildEvent(profileRequestContext, IdPEventIds.INVALID_PROFILE_CONFIG);
                    return;
                }
            }
        } else {
            log.debug("{} No manipulation strategy configured", getLogPrefix());
        }

        try {
            if (jwtTokenType) {
                accessTokenCtx.setJWT(new PlainJWT(sealClaims(claimsSet.getClaimsSet())));
                log.debug("{} Claims stored to JWT access token: {}", getLogPrefix(), claimsSet.serialize());
            } else { 
                accessTokenCtx.setOpaque(claimsSet.serialize(dataSealer));
                log.debug("{} Claims converted to opaque access token: {}", getLogPrefix(), claimsSet.serialize());
            }
        } catch (final DataSealerException | ParseException e) {
            log.error("{} Access Token wrapping failed: {}", getLogPrefix(), e);
            ActionSupport.buildEvent(profileRequestContext, EventIds.MESSAGE_PROC_ERROR);
        }
    }
 // Checkstyle: CyclomaticComplexity|MethodLength ON

    /**
     * Rewrites a plaintext claimsset to hide custom claims used solely by the OP.
     * 
     * @param claims the input claims
     * 
     * @return a rewritten claims set to use for the access token
     * 
     * @throws ParseException if unable to parse a claims set
     * @throws DataSealerException if unable to seal the custom claims
     */
    @Nonnull private JWTClaimsSet sealClaims(@Nonnull final JWTClaimsSet claims)
            throws DataSealerException, ParseException {
        
        // Rewrite as a mutable map.
        final Map<String,Object> map = claims.toJSONObject();

        // Put the claims to hide here.
        final Map<String,Object> toSeal = new HashMap<>();

        if (map.containsKey(TokenClaimsSet.KEY_TYPE)) {
            toSeal.put(TokenClaimsSet.KEY_TYPE, map.remove(TokenClaimsSet.KEY_TYPE));
        }
        
        if (map.containsKey(TokenClaimsSet.KEY_USER_PRINCIPAL)) {
            toSeal.put(TokenClaimsSet.KEY_USER_PRINCIPAL, map.remove(TokenClaimsSet.KEY_USER_PRINCIPAL));
        }
        
        if (map.containsKey(TokenClaimsSet.KEY_DELIVERY_CLAIMS)) {
            toSeal.put(TokenClaimsSet.KEY_DELIVERY_CLAIMS, map.remove(TokenClaimsSet.KEY_DELIVERY_CLAIMS));
        }

        if (map.containsKey(TokenClaimsSet.KEY_DELIVERY_CLAIMS_USERINFO)) {
            toSeal.put(TokenClaimsSet.KEY_DELIVERY_CLAIMS_USERINFO,
                    map.remove(TokenClaimsSet.KEY_DELIVERY_CLAIMS_USERINFO));
        }

        if (map.containsKey(TokenClaimsSet.KEY_CONSENTED_CLAIMS)) {
            toSeal.put(TokenClaimsSet.KEY_CONSENTED_CLAIMS, map.remove(TokenClaimsSet.KEY_CONSENTED_CLAIMS));
        }

        if (map.containsKey(TokenClaimsSet.KEY_CONSENT_ENABLED)) {
            toSeal.put(TokenClaimsSet.KEY_CONSENT_ENABLED, map.remove(TokenClaimsSet.KEY_CONSENT_ENABLED));
        }

        if (map.containsKey(TokenClaimsSet.KEY_CODE_CHALLENGE)) {
            toSeal.put(TokenClaimsSet.KEY_CODE_CHALLENGE, map.remove(TokenClaimsSet.KEY_CODE_CHALLENGE));
        }

        if (map.containsKey(TokenClaimsSet.KEY_NONCE)) {
            toSeal.put(TokenClaimsSet.KEY_NONCE, map.remove(TokenClaimsSet.KEY_NONCE));
        }

        if (toSeal.isEmpty()) {
            // Nothing to do.
            return claims;
        }
        
        // Wrap the sealed claims and re-embed back in original claims set.
        final String sealed = dataSealer.wrap(JWTClaimsSet.parse(toSeal).toString());
        map.put(TokenClaimsSet.KEY_SEALED_FOR_OP, sealed);
        
        // Re-parse the claims.
        return JWTClaimsSet.parse(map);
    }
    
}