/*
 * Licensed to the University Corporation for Advanced Internet Development,
 * Inc. (UCAID) under one or more contributor license agreements.  See the
 * NOTICE file distributed with this work for additional information regarding
 * copyright ownership. The UCAID licenses this file to You under the Apache
 * License, Version 2.0 (the "License"); you may not use this file except in
 * compliance with the License.  You may obtain a copy of the License at
 *
 *    http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package net.shibboleth.idp.plugin.oidc.op.encoding.impl;

import java.util.function.Function;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;

import org.opensaml.messaging.context.MessageContext;
import org.opensaml.messaging.encoder.MessageEncoder;
import org.opensaml.profile.context.ProfileRequestContext;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.nimbusds.oauth2.sdk.Response;

import net.shibboleth.utilities.java.support.annotation.constraint.NonnullAfterInit;
import net.shibboleth.utilities.java.support.component.AbstractInitializableComponent;
import net.shibboleth.utilities.java.support.component.ComponentInitializationException;
import net.shibboleth.utilities.java.support.logic.Constraint;

/**
 * A source of encoders that first verifies a message being an instance of Nimbus
 * {@link Response} and then returns the attached {@link MessageEncoder}.
 */
public class OIDCResponseEncoderFactory extends AbstractInitializableComponent
        implements Function<ProfileRequestContext,MessageEncoder> {

    /** Class logger. */
    @Nonnull private final Logger log = LoggerFactory.getLogger(OIDCResponseEncoderFactory.class);
    
    /** The message encoder to be returned by this factory. */
    @NonnullAfterInit private MessageEncoder messageEncoder;
    
    /**
     * Set the message encoder to be returned by this factory.
     * @param encoder What to set.
     */
    public void setMessageEncoder(@Nonnull final MessageEncoder encoder) {
        messageEncoder = Constraint.isNotNull(encoder, "The message encoder cannot be null");
    }
    
    /** {@inheritDoc} */
    @Override
    protected void doInitialize() throws ComponentInitializationException {
        super.doInitialize();
        
        if (messageEncoder == null) {
            throw new ComponentInitializationException("The message encoder cannot be null");
        }
    }

    /** {@inheritDoc} */
    @Nullable public MessageEncoder apply(@Nullable final ProfileRequestContext input) {
        final MessageContext messageContext = input != null ? input.getOutboundMessageContext() : null;
        if (messageContext == null) {
            log.error("No outbound message context available");
            return null;
        }
        final Object message = messageContext.getMessage();
        if (message == null || !(message instanceof Response)) {
            log.error("Unexpected message in the outbound message context: {}", message);
        }
        return messageEncoder;
    }

}