/*
 * Decompiled with CFR 0.152.
 */
package io.trino.spooling.filesystem;

import com.google.common.collect.ImmutableMap;
import com.google.inject.Inject;
import io.airlift.slice.BasicSliceInput;
import io.airlift.slice.DynamicSliceOutput;
import io.airlift.slice.Slice;
import io.airlift.units.Duration;
import io.trino.filesystem.Location;
import io.trino.filesystem.TrinoFileSystem;
import io.trino.filesystem.TrinoFileSystemFactory;
import io.trino.filesystem.TrinoInputFile;
import io.trino.filesystem.TrinoOutputFile;
import io.trino.filesystem.encryption.EncryptionKey;
import io.trino.spi.QueryId;
import io.trino.spi.protocol.SpooledLocation;
import io.trino.spi.protocol.SpooledSegmentHandle;
import io.trino.spi.protocol.SpoolingContext;
import io.trino.spi.protocol.SpoolingManager;
import io.trino.spi.security.ConnectorIdentity;
import io.trino.spooling.filesystem.FileSystemSpooledSegmentHandle;
import io.trino.spooling.filesystem.FileSystemSpoolingConfig;
import io.trino.spooling.filesystem.encryption.EncryptionHeadersTranslator;
import io.trino.spooling.filesystem.encryption.ExceptionMappingInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.nio.charset.StandardCharsets;
import java.time.Instant;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.Random;
import java.util.concurrent.ThreadLocalRandom;
import java.util.concurrent.TimeUnit;

public class FileSystemSpoolingManager
implements SpoolingManager {
    private final Location location;
    private final EncryptionHeadersTranslator encryptionHeadersTranslator;
    private final TrinoFileSystem fileSystem;
    private final Duration ttl;
    private final boolean encryptionEnabled;
    private final Random random = ThreadLocalRandom.current();

    @Inject
    public FileSystemSpoolingManager(FileSystemSpoolingConfig config, TrinoFileSystemFactory fileSystemFactory) {
        Objects.requireNonNull(config, "config is null");
        this.location = Location.of((String)config.getLocation());
        this.fileSystem = Objects.requireNonNull(fileSystemFactory, "fileSystemFactory is null").create(ConnectorIdentity.ofUser((String)"ignored"));
        this.encryptionHeadersTranslator = EncryptionHeadersTranslator.encryptionHeadersTranslator(this.location);
        this.ttl = config.getTtl();
        this.encryptionEnabled = config.isEncryptionEnabled();
    }

    public OutputStream createOutputStream(SpooledSegmentHandle handle) throws IOException {
        FileSystemSpooledSegmentHandle fileHandle = (FileSystemSpooledSegmentHandle)handle;
        Location storageLocation = this.location(fileHandle);
        Optional<EncryptionKey> encryption = fileHandle.encryptionKey();
        TrinoOutputFile outputFile = this.encryptionEnabled ? this.fileSystem.newEncryptedOutputFile(storageLocation, encryption.orElseThrow()) : this.fileSystem.newOutputFile(storageLocation);
        return outputFile.create();
    }

    public FileSystemSpooledSegmentHandle create(SpoolingContext context) {
        Instant expireAt = Instant.now().plusMillis(this.ttl.toMillis());
        if (this.encryptionEnabled) {
            return FileSystemSpooledSegmentHandle.random(this.random, context, expireAt, Optional.of(EncryptionKey.randomAes256()));
        }
        return FileSystemSpooledSegmentHandle.random(this.random, context, expireAt);
    }

    public InputStream openInputStream(SpooledSegmentHandle handle) throws IOException {
        FileSystemSpooledSegmentHandle fileHandle = (FileSystemSpooledSegmentHandle)handle;
        this.checkExpiration(fileHandle);
        Optional<EncryptionKey> encryption = fileHandle.encryptionKey();
        Location storageLocation = this.location(fileHandle);
        TrinoInputFile inputFile = this.encryptionEnabled ? this.fileSystem.newEncryptedInputFile(storageLocation, encryption.orElseThrow()) : this.fileSystem.newInputFile(storageLocation);
        FileSystemSpoolingManager.checkFileExists(inputFile);
        return new ExceptionMappingInputStream((InputStream)inputFile.newStream());
    }

    public void acknowledge(SpooledSegmentHandle handle) throws IOException {
        this.fileSystem.deleteFile(this.location((FileSystemSpooledSegmentHandle)handle));
    }

    public Optional<SpooledLocation.DirectLocation> directLocation(SpooledSegmentHandle handle) throws IOException {
        FileSystemSpooledSegmentHandle fileHandle = (FileSystemSpooledSegmentHandle)handle;
        Location storageLocation = this.location(fileHandle);
        Duration ttl = this.remainingTtl(fileHandle.expirationTime());
        Optional<EncryptionKey> key = fileHandle.encryptionKey();
        Optional<SpooledLocation.DirectLocation> directLocation = this.encryptionEnabled ? this.fileSystem.encryptedPreSignedUri(storageLocation, ttl, key.orElseThrow()).map(uri -> new SpooledLocation.DirectLocation(uri.uri(), uri.headers())) : this.fileSystem.preSignedUri(storageLocation, ttl).map(uri -> new SpooledLocation.DirectLocation(uri.uri(), uri.headers()));
        if (directLocation.isEmpty()) {
            throw new IOException("Failed to generate pre-signed URI for query %s and segment %s".formatted(fileHandle.queryId(), fileHandle.identifier()));
        }
        return directLocation;
    }

    public SpooledLocation location(SpooledSegmentHandle handle) {
        FileSystemSpooledSegmentHandle fileHandle = (FileSystemSpooledSegmentHandle)handle;
        DynamicSliceOutput output = new DynamicSliceOutput(64);
        output.writeBytes(fileHandle.uuid());
        output.writeShort(fileHandle.queryId().toString().length());
        output.writeShort(fileHandle.encoding().length());
        output.writeBytes(fileHandle.queryId().toString().getBytes(StandardCharsets.UTF_8));
        output.writeBytes(fileHandle.encoding().getBytes(StandardCharsets.UTF_8));
        output.writeBoolean(fileHandle.encryptionKey().isPresent());
        return SpooledLocation.coordinatorLocation((Slice)output.slice(), this.headers(fileHandle));
    }

    private Map<String, List<String>> headers(FileSystemSpooledSegmentHandle fileHandle) {
        return fileHandle.encryptionKey().map(this.encryptionHeadersTranslator::createHeaders).orElse((Map)ImmutableMap.of());
    }

    public SpooledSegmentHandle handle(SpooledLocation location) {
        if (!(location instanceof SpooledLocation.CoordinatorLocation)) {
            throw new IllegalArgumentException("Cannot convert direct location to handle");
        }
        SpooledLocation.CoordinatorLocation coordinatorLocation = (SpooledLocation.CoordinatorLocation)location;
        BasicSliceInput input = coordinatorLocation.identifier().getInput();
        byte[] uuid = new byte[16];
        input.readBytes(uuid);
        short queryLength = input.readShort();
        short encodingLength = input.readShort();
        QueryId queryId = QueryId.valueOf((String)input.readSlice((int)queryLength).toStringUtf8());
        String encoding = input.readSlice((int)encodingLength).toStringUtf8();
        if (!input.readBoolean()) {
            return new FileSystemSpooledSegmentHandle(encoding, queryId, uuid, Optional.empty());
        }
        return new FileSystemSpooledSegmentHandle(encoding, queryId, uuid, Optional.of(this.encryptionHeadersTranslator.extractKey(location.headers())));
    }

    private Location location(FileSystemSpooledSegmentHandle handle) throws IOException {
        this.checkExpiration(handle);
        return this.location.appendPath(handle.storageObjectName());
    }

    private Duration remainingTtl(Instant expiresAt) {
        return new Duration((double)java.time.Duration.between(Instant.now(), expiresAt).toMillis(), TimeUnit.MILLISECONDS);
    }

    private void checkExpiration(FileSystemSpooledSegmentHandle handle) throws IOException {
        if (handle.expirationTime().isBefore(Instant.now())) {
            throw new IOException("Segment not found or expired");
        }
    }

    private static void checkFileExists(TrinoInputFile inputFile) throws IOException {
        if (!inputFile.exists()) {
            throw new IOException("Segment not found or expired");
        }
    }
}

