/*
 * Decompiled with CFR 0.152.
 */
package io.trino.spooling.filesystem;

import com.google.common.annotations.VisibleForTesting;
import com.google.inject.Inject;
import io.airlift.log.Logger;
import io.airlift.units.Duration;
import io.trino.filesystem.FileEntry;
import io.trino.filesystem.FileIterator;
import io.trino.filesystem.Location;
import io.trino.filesystem.TrinoFileSystem;
import io.trino.filesystem.TrinoFileSystemFactory;
import io.trino.spi.security.ConnectorIdentity;
import io.trino.spooling.filesystem.FileSystemSpooledSegmentHandle;
import io.trino.spooling.filesystem.FileSystemSpoolingConfig;
import io.trino.spooling.filesystem.ForSegmentPruner;
import jakarta.annotation.PostConstruct;
import jakarta.annotation.PreDestroy;
import java.io.IOException;
import java.time.Instant;
import java.time.temporal.ChronoUnit;
import java.util.ArrayList;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.TimeUnit;

public class FileSystemSegmentPruner {
    private final Logger log = Logger.get(FileSystemSegmentPruner.class);
    private final TrinoFileSystem fileSystem;
    private final ScheduledExecutorService executor;
    private final boolean enabled;
    private final Duration interval;
    private final Location location;
    private final long batchSize;
    private boolean closed;
    private boolean filesAreOrdered = true;

    @Inject
    public FileSystemSegmentPruner(FileSystemSpoolingConfig config, TrinoFileSystemFactory fileSystemFactory, @ForSegmentPruner ScheduledExecutorService executor) {
        this.fileSystem = Objects.requireNonNull(fileSystemFactory, "fileSystemFactory is null").create(ConnectorIdentity.ofUser((String)"ignored"));
        this.executor = Objects.requireNonNull(executor, "executor is null");
        this.enabled = config.isPruningEnabled();
        this.interval = config.getPruningInterval();
        this.batchSize = config.getPruningBatchSize();
        this.location = Location.of((String)config.getLocation());
    }

    @PostConstruct
    public void start() {
        if (!this.enabled) {
            return;
        }
        this.log.info("Started expired segment pruning with interval %s and batch size %d", new Object[]{this.interval, this.batchSize});
        this.executor.scheduleAtFixedRate(this::prune, 0L, this.interval.toMillis(), TimeUnit.MILLISECONDS);
    }

    @PreDestroy
    public void shutdown() {
        if (!this.closed) {
            this.closed = true;
            this.executor.shutdownNow();
        }
    }

    private void prune() {
        this.pruneExpiredBefore(Instant.now().truncatedTo(ChronoUnit.SECONDS));
    }

    @VisibleForTesting
    long pruneExpiredBefore(Instant expiredBefore) {
        if (this.closed) {
            return 0L;
        }
        long pruned = 0L;
        try {
            ArrayList<Location> expiredSegments = new ArrayList<Location>();
            FileIterator iterator = this.orderDetectingIterator(this.fileSystem.listFiles(this.location));
            while (iterator.hasNext()) {
                FileEntry file = iterator.next();
                Optional<Instant> handle = FileSystemSpooledSegmentHandle.getExpirationFromLocation(file.location());
                if (handle.isEmpty()) continue;
                if (handle.get().isBefore(expiredBefore)) {
                    expiredSegments.add(file.location());
                    if ((long)expiredSegments.size() < this.batchSize) continue;
                    pruned += (long)expiredSegments.size();
                    this.pruneExpiredSegments(expiredBefore, expiredSegments);
                    expiredSegments.clear();
                    continue;
                }
                if (!this.filesAreOrdered) continue;
                this.pruneExpiredSegments(expiredBefore, expiredSegments);
                return pruned + (long)expiredSegments.size();
            }
            this.pruneExpiredSegments(expiredBefore, expiredSegments);
            return pruned + (long)expiredSegments.size();
        }
        catch (IOException e) {
            this.log.error((Throwable)e, "Failed to prune segments");
            return pruned;
        }
    }

    private void pruneExpiredSegments(Instant expiredBefore, List<Location> expiredSegments) {
        if (expiredSegments.isEmpty()) {
            return;
        }
        try {
            int batchSize = expiredSegments.size();
            Instant oldest = FileSystemSpooledSegmentHandle.getExpirationFromLocation(expiredSegments.getFirst()).orElseThrow(() -> new IllegalStateException("No expiration time found for " + String.valueOf(expiredSegments.getFirst())));
            Instant newest = FileSystemSpooledSegmentHandle.getExpirationFromLocation(expiredSegments.getLast()).orElseThrow(() -> new IllegalStateException("No expiration time found for " + String.valueOf(expiredSegments.getLast())));
            this.fileSystem.deleteFiles(expiredSegments);
            this.log.info("Pruned %d segments expired before %s [oldest: %s, newest: %s]", new Object[]{batchSize, expiredBefore, oldest, newest});
        }
        catch (IOException e) {
            this.log.warn((Throwable)e, "Failed to delete %d expired segments", new Object[]{expiredSegments.size()});
        }
        catch (Exception e) {
            this.log.error((Throwable)e, "Unexpected error while pruning expired segments");
        }
    }

    private FileIterator orderDetectingIterator(final FileIterator delegate) {
        return new FileIterator(){
            private FileEntry last;
            final /* synthetic */ FileSystemSegmentPruner this$0;
            {
                this.this$0 = this$0;
            }

            public boolean hasNext() throws IOException {
                return delegate.hasNext();
            }

            public FileEntry next() throws IOException {
                FileEntry next = delegate.next();
                if (this.this$0.filesAreOrdered && this.last != null && this.last.location().fileName().compareTo(next.location().fileName()) > 0) {
                    this.this$0.filesAreOrdered = false;
                    this.last = next;
                }
                return next;
            }
        };
    }
}

