/*
 * Decompiled with CFR 0.152.
 */
package io.trino.plugin.password.ldap;

import com.google.common.annotations.VisibleForTesting;
import com.google.common.base.CharMatcher;
import com.google.common.base.Function;
import com.google.common.base.Preconditions;
import com.google.common.base.Throwables;
import com.google.common.cache.CacheBuilder;
import com.google.common.cache.CacheLoader;
import com.google.common.collect.Iterables;
import com.google.common.util.concurrent.UncheckedExecutionException;
import io.airlift.log.Logger;
import io.trino.collect.cache.NonKeyEvictableLoadingCache;
import io.trino.collect.cache.SafeCaches;
import io.trino.plugin.password.Credential;
import io.trino.plugin.password.ldap.LdapAuthenticatorClient;
import io.trino.plugin.password.ldap.LdapConfig;
import io.trino.spi.classloader.ThreadContextClassLoader;
import io.trino.spi.security.AccessDeniedException;
import io.trino.spi.security.BasicPrincipal;
import io.trino.spi.security.PasswordAuthenticator;
import java.security.Principal;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.Set;
import java.util.concurrent.TimeUnit;
import javax.inject.Inject;
import javax.naming.NamingException;

public class LdapAuthenticator
implements PasswordAuthenticator {
    private static final Logger log = Logger.get(LdapAuthenticator.class);
    private static final CharMatcher SPECIAL_CHARACTERS = CharMatcher.anyOf((CharSequence)",=+<>#;*()\"\\\u0000");
    private static final CharMatcher WHITESPACE = CharMatcher.anyOf((CharSequence)" \r");
    private final LdapAuthenticatorClient client;
    private final List<String> userBindSearchPatterns;
    private final Optional<String> groupAuthorizationSearchPattern;
    private final Optional<String> userBaseDistinguishedName;
    private final Optional<String> bindDistinguishedName;
    private final Optional<String> bindPassword;
    private final NonKeyEvictableLoadingCache<Credential, Principal> authenticationCache;

    @Inject
    public LdapAuthenticator(LdapAuthenticatorClient client, LdapConfig ldapConfig) {
        this.client = Objects.requireNonNull(client, "client is null");
        this.userBindSearchPatterns = ldapConfig.getUserBindSearchPatterns();
        this.groupAuthorizationSearchPattern = Optional.ofNullable(ldapConfig.getGroupAuthorizationSearchPattern());
        this.userBaseDistinguishedName = Optional.ofNullable(ldapConfig.getUserBaseDistinguishedName());
        this.bindDistinguishedName = Optional.ofNullable(ldapConfig.getBindDistingushedName());
        this.bindPassword = Optional.ofNullable(ldapConfig.getBindPassword());
        Preconditions.checkArgument((this.groupAuthorizationSearchPattern.isEmpty() || this.userBaseDistinguishedName.isPresent() ? 1 : 0) != 0, (Object)"Base distinguished name (DN) for user must be provided");
        Preconditions.checkArgument((this.bindDistinguishedName.isPresent() == this.bindPassword.isPresent() ? 1 : 0) != 0, (Object)"Both bind distinguished name and bind password must be provided together");
        Preconditions.checkArgument((this.bindDistinguishedName.isEmpty() || this.groupAuthorizationSearchPattern.isPresent() ? 1 : 0) != 0, (Object)"Group authorization search pattern must be provided when bind distinguished name is used");
        Preconditions.checkArgument((this.bindDistinguishedName.isPresent() || !this.userBindSearchPatterns.isEmpty() ? 1 : 0) != 0, (Object)"Either user bind search pattern or bind distinguished name must be provided");
        this.authenticationCache = SafeCaches.buildNonEvictableCacheWithWeakInvalidateAll((CacheBuilder)CacheBuilder.newBuilder().expireAfterWrite(ldapConfig.getLdapCacheTtl().toMillis(), TimeUnit.MILLISECONDS), (CacheLoader)CacheLoader.from((Function)(this.bindDistinguishedName.isPresent() ? this::authenticateWithBindDistinguishedName : this::authenticateWithUserBind)));
    }

    @VisibleForTesting
    void invalidateCache() {
        this.authenticationCache.invalidateAll();
    }

    public Principal createAuthenticatedPrincipal(String user, String password) {
        Principal principal;
        ThreadContextClassLoader ignored = new ThreadContextClassLoader(this.getClass().getClassLoader());
        try {
            principal = (Principal)this.authenticationCache.getUnchecked((Object)new Credential(user, password));
        }
        catch (Throwable throwable) {
            try {
                try {
                    ignored.close();
                }
                catch (Throwable throwable2) {
                    throwable.addSuppressed(throwable2);
                }
                throw throwable;
            }
            catch (UncheckedExecutionException e) {
                Throwables.throwIfInstanceOf((Throwable)e.getCause(), AccessDeniedException.class);
                throw e;
            }
        }
        ignored.close();
        return principal;
    }

    private Principal authenticateWithUserBind(Credential credential) {
        String user = credential.getUser();
        if (LdapAuthenticator.containsSpecialCharacters(user)) {
            throw new AccessDeniedException("Username contains a special LDAP character");
        }
        Throwable lastException = new RuntimeException();
        for (String userBindSearchPattern : this.userBindSearchPatterns) {
            try {
                String userDistinguishedName = LdapAuthenticator.replaceUser(userBindSearchPattern, user);
                if (this.groupAuthorizationSearchPattern.isPresent()) {
                    String groupSearch;
                    String searchBase = this.userBaseDistinguishedName.orElseThrow();
                    if (!this.client.isGroupMember(searchBase, groupSearch = LdapAuthenticator.replaceUser(this.groupAuthorizationSearchPattern.get(), user), userDistinguishedName, credential.getPassword())) {
                        String message = String.format("User [%s] not a member of an authorized group", user);
                        log.debug("%s", new Object[]{message});
                        throw new AccessDeniedException(message);
                    }
                } else {
                    this.client.validatePassword(userDistinguishedName, credential.getPassword());
                }
                log.debug("Authentication successful for user [%s]", new Object[]{user});
                return new BasicPrincipal(user);
            }
            catch (AccessDeniedException | NamingException e) {
                lastException = e;
            }
        }
        log.debug(lastException, "Authentication failed for user [%s], %s", new Object[]{user, lastException.getMessage()});
        if (lastException instanceof AccessDeniedException) {
            throw (AccessDeniedException)lastException;
        }
        throw new RuntimeException("Authentication error");
    }

    private Principal authenticateWithBindDistinguishedName(Credential credential) {
        String user = credential.getUser();
        if (LdapAuthenticator.containsSpecialCharacters(user)) {
            throw new AccessDeniedException("Username contains a special LDAP character");
        }
        try {
            String userDistinguishedName = this.lookupUserDistinguishedName(user);
            this.client.validatePassword(userDistinguishedName, credential.getPassword());
            log.debug("Authentication successful for user [%s]", new Object[]{user});
        }
        catch (NamingException e) {
            log.debug((Throwable)e, "Authentication failed for user [%s], %s", new Object[]{user, e.getMessage()});
            throw new RuntimeException("Authentication error");
        }
        return new BasicPrincipal(credential.getUser());
    }

    @VisibleForTesting
    static boolean containsSpecialCharacters(String user) {
        if (WHITESPACE.indexIn((CharSequence)user) == 0 || WHITESPACE.lastIndexIn((CharSequence)user) == user.length() - 1) {
            return true;
        }
        return SPECIAL_CHARACTERS.matchesAnyOf((CharSequence)user);
    }

    private String lookupUserDistinguishedName(String user) throws NamingException {
        String searchFilter;
        String searchBase = this.userBaseDistinguishedName.orElseThrow();
        Set<String> userDistinguishedNames = this.client.lookupUserDistinguishedNames(searchBase, searchFilter = LdapAuthenticator.replaceUser(this.groupAuthorizationSearchPattern.orElseThrow(), user), this.bindDistinguishedName.orElseThrow(), this.bindPassword.orElseThrow());
        if (userDistinguishedNames.isEmpty()) {
            String message = String.format("User [%s] not a member of an authorized group", user);
            log.debug("%s", new Object[]{message});
            throw new AccessDeniedException(message);
        }
        if (userDistinguishedNames.size() > 1) {
            String message = String.format("Multiple group membership results for user [%s]: %s", user, userDistinguishedNames);
            log.debug("%s", new Object[]{message});
            throw new AccessDeniedException(message);
        }
        return (String)Iterables.getOnlyElement(userDistinguishedNames);
    }

    private static String replaceUser(String pattern, String user) {
        return pattern.replace("${USER}", user);
    }
}

