/*
 * Decompiled with CFR 0.152.
 */
package io.trino.server.security.jwt;

import com.fasterxml.jackson.annotation.JsonAnySetter;
import com.fasterxml.jackson.annotation.JsonCreator;
import com.fasterxml.jackson.annotation.JsonProperty;
import com.google.common.base.Strings;
import com.google.common.collect.ImmutableList;
import com.google.common.collect.ImmutableMap;
import io.airlift.json.JsonCodec;
import io.airlift.log.Logger;
import io.trino.server.security.jwt.EcCurve;
import java.math.BigInteger;
import java.security.PublicKey;
import java.security.interfaces.ECPublicKey;
import java.security.interfaces.RSAPublicKey;
import java.security.spec.ECParameterSpec;
import java.security.spec.ECPoint;
import java.util.Base64;
import java.util.Collection;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.Function;

public final class JwkDecoder {
    private static final Logger log = Logger.get(JwkDecoder.class);
    private static final JsonCodec<Keys> KEYS_CODEC = JsonCodec.jsonCodec(Keys.class);

    private JwkDecoder() {
    }

    public static Map<String, PublicKey> decodeKeys(String jwkJson) {
        Keys keys = (Keys)KEYS_CODEC.fromJson(jwkJson);
        return (Map)keys.getKeys().stream().map(JwkDecoder::tryDecodeJwkKey).filter(Optional::isPresent).map(Optional::get).collect(ImmutableMap.toImmutableMap(JwkPublicKey::getKeyId, Function.identity()));
    }

    public static Optional<? extends JwkPublicKey> tryDecodeJwkKey(Key key) {
        if (key.getKid().isEmpty() || key.getKid().get().isEmpty()) {
            return Optional.empty();
        }
        String keyId = key.getKid().get();
        return switch (key.getKty()) {
            case "RSA" -> JwkDecoder.tryDecodeRsaKey(keyId, key);
            case "EC" -> JwkDecoder.tryDecodeEcKey(keyId, key);
            default -> Optional.empty();
        };
    }

    public static Optional<JwkRsaPublicKey> tryDecodeRsaKey(String keyId, Key key) {
        Optional modulus = key.getStringProperty("n").flatMap(encodedModulus -> JwkDecoder.decodeBigint(keyId, "modulus", encodedModulus));
        if (modulus.isEmpty()) {
            return Optional.empty();
        }
        Optional exponent = key.getStringProperty("e").flatMap(encodedExponent -> JwkDecoder.decodeBigint(keyId, "exponent", encodedExponent));
        if (exponent.isEmpty()) {
            return Optional.empty();
        }
        return Optional.of(new JwkRsaPublicKey(keyId, (BigInteger)exponent.get(), (BigInteger)modulus.get()));
    }

    public static Optional<JwkEcPublicKey> tryDecodeEcKey(String keyId, Key key) {
        Optional<String> curveName = key.getStringProperty("crv");
        Optional curve = curveName.flatMap(EcCurve::tryGet);
        if (curve.isEmpty()) {
            log.error("JWK EC %s curve '%s' is not supported", new Object[]{keyId, curveName});
            return Optional.empty();
        }
        Optional x = key.getStringProperty("x").flatMap(encodedX -> JwkDecoder.decodeBigint(keyId, "x", encodedX));
        if (x.isEmpty()) {
            return Optional.empty();
        }
        Optional y = key.getStringProperty("y").flatMap(encodedY -> JwkDecoder.decodeBigint(keyId, "y", encodedY));
        if (y.isEmpty()) {
            return Optional.empty();
        }
        ECPoint w = new ECPoint((BigInteger)x.get(), (BigInteger)y.get());
        return Optional.of(new JwkEcPublicKey(keyId, (ECParameterSpec)curve.get(), w));
    }

    private static Optional<BigInteger> decodeBigint(String keyId, String fieldName, String encodedNumber) {
        try {
            return Optional.of(new BigInteger(1, Base64.getUrlDecoder().decode(encodedNumber)));
        }
        catch (IllegalArgumentException e) {
            log.error((Throwable)e, "JWK %s %s is not a valid number", new Object[]{keyId, fieldName});
            return Optional.empty();
        }
    }

    public static class Keys {
        private final List<Key> keys;

        @JsonCreator
        public Keys(@JsonProperty(value="keys") List<Key> keys) {
            this.keys = ImmutableList.copyOf((Collection)Objects.requireNonNull(keys, "keys is null"));
        }

        public List<Key> getKeys() {
            return this.keys;
        }
    }

    public static class Key {
        private final String kty;
        private final Optional<String> kid;
        private final Map<String, Object> other = new HashMap<String, Object>();

        @JsonCreator
        public Key(@JsonProperty(value="kty") String kty, @JsonProperty(value="kid") Optional<String> kid) {
            this.kty = Objects.requireNonNull(kty, "kty is null");
            this.kid = Objects.requireNonNull(kid, "kid is null");
        }

        public String getKty() {
            return this.kty;
        }

        public Optional<String> getKid() {
            return this.kid;
        }

        public Optional<String> getStringProperty(String name) {
            Object value = this.other.get(name);
            if (value instanceof String && !Strings.isNullOrEmpty((String)((String)value))) {
                return Optional.of((String)value);
            }
            return Optional.empty();
        }

        @JsonAnySetter
        public void set(String name, Object value) {
            this.other.put(name, value);
        }
    }

    public static class JwkRsaPublicKey
    implements JwkPublicKey,
    RSAPublicKey {
        private final String keyId;
        private final BigInteger modulus;
        private final BigInteger exponent;

        public JwkRsaPublicKey(String keyId, BigInteger exponent, BigInteger modulus) {
            this.keyId = Objects.requireNonNull(keyId, "keyId is null");
            this.exponent = Objects.requireNonNull(exponent, "exponent is null");
            this.modulus = Objects.requireNonNull(modulus, "modulus is null");
        }

        @Override
        public String getKeyId() {
            return this.keyId;
        }

        @Override
        public BigInteger getModulus() {
            return this.modulus;
        }

        @Override
        public BigInteger getPublicExponent() {
            return this.exponent;
        }

        @Override
        public String getAlgorithm() {
            return "RSA";
        }

        @Override
        public String getFormat() {
            return "JWK";
        }

        @Override
        public byte[] getEncoded() {
            throw new UnsupportedOperationException();
        }
    }

    public static class JwkEcPublicKey
    implements JwkPublicKey,
    ECPublicKey {
        private final String keyId;
        private final ECParameterSpec parameterSpec;
        private final ECPoint w;

        public JwkEcPublicKey(String keyId, ECParameterSpec parameterSpec, ECPoint w) {
            this.keyId = Objects.requireNonNull(keyId, "keyId is null");
            this.parameterSpec = Objects.requireNonNull(parameterSpec, "parameterSpec is null");
            this.w = Objects.requireNonNull(w, "w is null");
        }

        @Override
        public String getKeyId() {
            return this.keyId;
        }

        @Override
        public ECParameterSpec getParams() {
            return this.parameterSpec;
        }

        @Override
        public ECPoint getW() {
            return this.w;
        }

        @Override
        public String getAlgorithm() {
            return "EC";
        }

        @Override
        public String getFormat() {
            return "JWK";
        }

        @Override
        public byte[] getEncoded() {
            throw new UnsupportedOperationException();
        }
    }

    public static interface JwkPublicKey
    extends PublicKey {
        public String getKeyId();
    }
}

