/*
 * Decompiled with CFR 0.152.
 */
package io.trino.operator.scalar;

import com.google.common.annotations.VisibleForTesting;
import com.google.common.base.Verify;
import io.trino.spi.ErrorCodeSupplier;
import io.trino.spi.StandardErrorCode;
import io.trino.spi.TrinoException;
import io.trino.spi.block.ArrayBlock;
import io.trino.spi.block.Block;
import io.trino.spi.block.DictionaryBlock;
import io.trino.spi.block.PageBuilderStatus;
import io.trino.spi.function.Description;
import io.trino.spi.function.ScalarFunction;
import io.trino.spi.function.SqlType;
import io.trino.spi.function.TypeParameter;
import io.trino.spi.type.ArrayType;
import io.trino.spi.type.Type;
import io.trino.util.Failures;
import java.util.Arrays;
import java.util.Optional;

@ScalarFunction(value="combinations")
@Description(value="Return n-element subsets from array")
public final class ArrayCombinationsFunction {
    private static final int MAX_COMBINATION_LENGTH = 5;
    private static final int MAX_RESULT_ELEMENTS = 100000;

    private ArrayCombinationsFunction() {
    }

    @TypeParameter(value="T")
    @SqlType(value="array(array(T))")
    public static Block combinations(@TypeParameter(value="T") Type elementType, @SqlType(value="array(T)") Block array, @SqlType(value="integer") long n) {
        int arrayLength = array.getPositionCount();
        int combinationLength = StrictMath.toIntExact(n);
        Failures.checkCondition(combinationLength >= 0, (ErrorCodeSupplier)StandardErrorCode.INVALID_FUNCTION_ARGUMENT, "combination size must not be negative: %s", combinationLength);
        Failures.checkCondition(combinationLength <= 5, (ErrorCodeSupplier)StandardErrorCode.INVALID_FUNCTION_ARGUMENT, "combination size must not exceed %s: %s", 5, combinationLength);
        ArrayType arrayType = new ArrayType(elementType);
        if (combinationLength > arrayLength) {
            return arrayType.createBlockBuilder(new PageBuilderStatus().createBlockBuilderStatus(), 0).build();
        }
        int combinationCount = ArrayCombinationsFunction.combinationCount(arrayLength, combinationLength);
        Failures.checkCondition((long)combinationCount * (long)combinationLength <= 100000L, (ErrorCodeSupplier)StandardErrorCode.INVALID_FUNCTION_ARGUMENT, "combinations exceed max size", new Object[0]);
        int[] ids = new int[combinationCount * combinationLength];
        int idsPosition = 0;
        int[] combination = ArrayCombinationsFunction.firstCombination(arrayLength, combinationLength);
        do {
            System.arraycopy(combination, 0, ids, idsPosition, combinationLength);
            idsPosition += combinationLength;
        } while (ArrayCombinationsFunction.nextCombination(combination, combinationLength));
        Verify.verify((idsPosition == ids.length ? 1 : 0) != 0, (String)"idsPosition != ids.length, %s and %s respectively", (int)idsPosition, (int)ids.length);
        int[] offsets = new int[combinationCount + 1];
        Arrays.setAll(offsets, i -> i * combinationLength);
        return ArrayBlock.fromElementBlock((int)combinationCount, Optional.empty(), (int[])offsets, (Block)DictionaryBlock.create((int)ids.length, (Block)array, (int[])ids));
    }

    @VisibleForTesting
    static int combinationCount(int arrayLength, int combinationLength) {
        try {
            int combinations = 1;
            for (int i = 1; i <= combinationLength; ++i) {
                combinations = Math.multiplyExact(combinations, arrayLength - combinationLength + i) / i;
            }
            return combinations;
        }
        catch (ArithmeticException e) {
            throw new TrinoException((ErrorCodeSupplier)StandardErrorCode.INVALID_FUNCTION_ARGUMENT, String.format("Number of combinations too large for array of size %s and combination length %s", arrayLength, combinationLength));
        }
    }

    private static int[] firstCombination(int arrayLength, int combinationLength) {
        int[] combination = new int[combinationLength + 1];
        Arrays.setAll(combination, i -> i);
        combination[combinationLength] = arrayLength;
        return combination;
    }

    private static boolean nextCombination(int[] combination, int combinationLength) {
        for (int i = 0; i < combinationLength; ++i) {
            if (combination[i] + 1 >= combination[i + 1]) continue;
            int n = i;
            combination[n] = combination[n] + 1;
            ArrayCombinationsFunction.resetCombination(combination, i);
            return true;
        }
        return false;
    }

    private static void resetCombination(int[] combination, int to) {
        for (int i = 0; i < to; ++i) {
            combination[i] = i;
        }
    }
}

