/*
 * Decompiled with CFR 0.152.
 */
package io.trino.plugin.geospatial;

import com.esri.core.geometry.Envelope;
import com.esri.core.geometry.Geometry;
import com.esri.core.geometry.GeometryCursor;
import com.esri.core.geometry.GeometryEngine;
import com.esri.core.geometry.ListeningGeometryCursor;
import com.esri.core.geometry.MultiPath;
import com.esri.core.geometry.MultiPoint;
import com.esri.core.geometry.MultiVertexGeometry;
import com.esri.core.geometry.NonSimpleResult;
import com.esri.core.geometry.OperatorSimplifyOGC;
import com.esri.core.geometry.OperatorUnion;
import com.esri.core.geometry.Point;
import com.esri.core.geometry.Polygon;
import com.esri.core.geometry.Polyline;
import com.esri.core.geometry.SpatialReference;
import com.esri.core.geometry.ogc.OGCConcreteGeometryCollection;
import com.esri.core.geometry.ogc.OGCGeometry;
import com.esri.core.geometry.ogc.OGCGeometryCollection;
import com.esri.core.geometry.ogc.OGCLineString;
import com.esri.core.geometry.ogc.OGCPoint;
import com.esri.core.geometry.ogc.OGCPolygon;
import com.google.common.base.Joiner;
import com.google.common.base.Preconditions;
import com.google.common.base.VerifyException;
import com.google.common.collect.ImmutableList;
import com.google.common.collect.ImmutableMap;
import io.airlift.slice.Slice;
import io.airlift.slice.Slices;
import io.trino.geospatial.GeometryUtils;
import io.trino.geospatial.KdbTree;
import io.trino.geospatial.Rectangle;
import io.trino.geospatial.serde.GeometrySerde;
import io.trino.geospatial.serde.GeometrySerializationType;
import io.trino.geospatial.serde.JtsGeometrySerde;
import io.trino.plugin.geospatial.GeometryType;
import io.trino.spi.ErrorCodeSupplier;
import io.trino.spi.StandardErrorCode;
import io.trino.spi.TrinoException;
import io.trino.spi.block.Block;
import io.trino.spi.block.BlockBuilder;
import io.trino.spi.block.RowValueBuilder;
import io.trino.spi.block.SqlRow;
import io.trino.spi.block.VariableWidthBlockBuilder;
import io.trino.spi.function.Description;
import io.trino.spi.function.ScalarFunction;
import io.trino.spi.function.SqlNullable;
import io.trino.spi.function.SqlType;
import io.trino.spi.type.IntegerType;
import io.trino.spi.type.RowType;
import java.nio.ByteBuffer;
import java.nio.ByteOrder;
import java.util.ArrayDeque;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.Deque;
import java.util.EnumSet;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.NoSuchElementException;
import java.util.Objects;
import java.util.Set;
import org.locationtech.jts.geom.Coordinate;
import org.locationtech.jts.geom.GeometryCollection;
import org.locationtech.jts.geom.GeometryFactory;
import org.locationtech.jts.linearref.LengthIndexedLine;
import org.locationtech.jts.operation.distance.DistanceOp;
import org.locationtech.jts.simplify.TopologyPreservingSimplifier;

public final class GeoFunctions {
    private static final Joiner OR_JOINER = Joiner.on((String)" or ");
    private static final Slice EMPTY_POLYGON = GeometrySerde.serialize((OGCGeometry)new OGCPolygon(new Polygon(), null));
    private static final Slice EMPTY_MULTIPOINT = GeometrySerde.serialize((OGCGeometry)OGCGeometry.createFromEsriGeometry((Geometry)new MultiPoint(), null, (boolean)true));
    private static final double EARTH_RADIUS_KM = 6371.01;
    private static final double EARTH_RADIUS_M = 6371010.0;
    private static final Map<NonSimpleResult.Reason, String> NON_SIMPLE_REASONS = ImmutableMap.builder().put((Object)NonSimpleResult.Reason.DegenerateSegments, (Object)"Degenerate segments").put((Object)NonSimpleResult.Reason.Clustering, (Object)"Repeated points").put((Object)NonSimpleResult.Reason.Cracking, (Object)"Intersecting or overlapping segments").put((Object)NonSimpleResult.Reason.CrossOver, (Object)"Self-intersection").put((Object)NonSimpleResult.Reason.OGCPolylineSelfTangency, (Object)"Self-tangency").put((Object)NonSimpleResult.Reason.OGCPolygonSelfTangency, (Object)"Self-tangency").put((Object)NonSimpleResult.Reason.OGCDisconnectedInterior, (Object)"Disconnected interior").buildOrThrow();
    private static final int NUMBER_OF_DIMENSIONS = 3;
    private static final Block EMPTY_ARRAY_OF_INTS = IntegerType.INTEGER.createFixedSizeBlockBuilder(0).build();
    private static final float MIN_LATITUDE = -90.0f;
    private static final float MAX_LATITUDE = 90.0f;
    private static final float MIN_LONGITUDE = -180.0f;
    private static final float MAX_LONGITUDE = 180.0f;
    private static final int HADOOP_SHAPE_SIZE_WKID = 4;
    private static final int HADOOP_SHAPE_SIZE_TYPE = 1;
    private static final int[] HADOOP_SHAPE_TYPES = new int[]{0, 1, 4, 16, 2, 8, 32};
    private static final EnumSet<Geometry.Type> GEOMETRY_TYPES_FOR_SPHERICAL_GEOGRAPHY = EnumSet.of(Geometry.Type.Point, Geometry.Type.Polyline, Geometry.Type.Polygon, Geometry.Type.MultiPoint);
    private static final EnumSet<io.trino.geospatial.GeometryType> VALID_TYPES_FOR_ST_POINTS = EnumSet.of(io.trino.geospatial.GeometryType.LINE_STRING, new io.trino.geospatial.GeometryType[]{io.trino.geospatial.GeometryType.POLYGON, io.trino.geospatial.GeometryType.POINT, io.trino.geospatial.GeometryType.MULTI_POINT, io.trino.geospatial.GeometryType.MULTI_LINE_STRING, io.trino.geospatial.GeometryType.MULTI_POLYGON, io.trino.geospatial.GeometryType.GEOMETRY_COLLECTION});

    private GeoFunctions() {
    }

    @Description(value="Returns a Geometry type LineString object from Well-Known Text representation (WKT)")
    @ScalarFunction(value="ST_LineFromText")
    @SqlType(value="Geometry")
    public static Slice parseLine(@SqlType(value="varchar") Slice input) {
        OGCGeometry geometry = GeoFunctions.geometryFromText(input);
        GeoFunctions.validateType("ST_LineFromText", geometry, EnumSet.of(io.trino.geospatial.GeometryType.LINE_STRING));
        return GeometrySerde.serialize((OGCGeometry)geometry);
    }

    @Description(value="Returns a LineString from an array of points")
    @ScalarFunction(value="ST_LineString")
    @SqlType(value="Geometry")
    public static Slice stLineString(@SqlType(value="array(Geometry)") Block input) {
        Polyline multipath = new Polyline();
        OGCPoint previousPoint = null;
        for (int i = 0; i < input.getPositionCount(); ++i) {
            Slice slice = GeometryType.GEOMETRY.getSlice(input, i);
            if (slice.getInput().available() == 0) {
                throw new TrinoException((ErrorCodeSupplier)StandardErrorCode.INVALID_FUNCTION_ARGUMENT, String.format("Invalid input to ST_LineString: null point at index %s", i + 1));
            }
            OGCGeometry geometry = GeometrySerde.deserialize((Slice)slice);
            if (!(geometry instanceof OGCPoint)) {
                throw new TrinoException((ErrorCodeSupplier)StandardErrorCode.INVALID_FUNCTION_ARGUMENT, String.format("ST_LineString takes only an array of valid points, %s was passed", geometry.geometryType()));
            }
            OGCPoint point = (OGCPoint)geometry;
            if (point.isEmpty()) {
                throw new TrinoException((ErrorCodeSupplier)StandardErrorCode.INVALID_FUNCTION_ARGUMENT, String.format("Invalid input to ST_LineString: empty point at index %s", i + 1));
            }
            if (previousPoint == null) {
                multipath.startPath(point.X(), point.Y());
            } else {
                if (point.Equals(previousPoint)) {
                    throw new TrinoException((ErrorCodeSupplier)StandardErrorCode.INVALID_FUNCTION_ARGUMENT, String.format("Invalid input to ST_LineString: consecutive duplicate points at index %s", i + 1));
                }
                multipath.lineTo(point.X(), point.Y());
            }
            previousPoint = point;
        }
        OGCLineString linestring = new OGCLineString((MultiPath)multipath, 0, null);
        return GeometrySerde.serialize((OGCGeometry)linestring);
    }

    @Description(value="Returns a Geometry type Point object with the given coordinate values")
    @ScalarFunction(value="ST_Point")
    @SqlType(value="Geometry")
    public static Slice stPoint(@SqlType(value="double") double x, @SqlType(value="double") double y) {
        OGCGeometry geometry = OGCGeometry.createFromEsriGeometry((Geometry)new Point(x, y), null);
        return GeometrySerde.serialize((OGCGeometry)geometry);
    }

    @SqlNullable
    @Description(value="Returns a multi-point geometry formed from input points")
    @ScalarFunction(value="ST_MultiPoint")
    @SqlType(value="Geometry")
    public static Slice stMultiPoint(@SqlType(value="array(Geometry)") Block input) {
        MultiPoint multipoint = new MultiPoint();
        for (int i = 0; i < input.getPositionCount(); ++i) {
            if (input.isNull(i)) {
                throw new TrinoException((ErrorCodeSupplier)StandardErrorCode.INVALID_FUNCTION_ARGUMENT, String.format("Invalid input to ST_MultiPoint: null at index %s", i + 1));
            }
            Slice slice = GeometryType.GEOMETRY.getSlice(input, i);
            OGCGeometry geometry = GeometrySerde.deserialize((Slice)slice);
            if (!(geometry instanceof OGCPoint)) {
                throw new TrinoException((ErrorCodeSupplier)StandardErrorCode.INVALID_FUNCTION_ARGUMENT, String.format("Invalid input to ST_MultiPoint: geometry is not a point: %s at index %s", geometry.geometryType(), i + 1));
            }
            OGCPoint point = (OGCPoint)geometry;
            if (point.isEmpty()) {
                throw new TrinoException((ErrorCodeSupplier)StandardErrorCode.INVALID_FUNCTION_ARGUMENT, String.format("Invalid input to ST_MultiPoint: empty point at index %s", i + 1));
            }
            multipoint.add(point.X(), point.Y());
        }
        if (multipoint.getPointCount() == 0) {
            return null;
        }
        return GeometrySerde.serialize((OGCGeometry)OGCGeometry.createFromEsriGeometry((Geometry)multipoint, null, (boolean)true));
    }

    @Description(value="Returns a Geometry type Polygon object from Well-Known Text representation (WKT)")
    @ScalarFunction(value="ST_Polygon")
    @SqlType(value="Geometry")
    public static Slice stPolygon(@SqlType(value="varchar") Slice input) {
        OGCGeometry geometry = GeoFunctions.geometryFromText(input);
        GeoFunctions.validateType("ST_Polygon", geometry, EnumSet.of(io.trino.geospatial.GeometryType.POLYGON));
        return GeometrySerde.serialize((OGCGeometry)geometry);
    }

    @Description(value="Returns the 2D Euclidean area of a geometry")
    @ScalarFunction(value="ST_Area")
    @SqlType(value="double")
    public static double stArea(@SqlType(value="Geometry") Slice input) {
        OGCGeometry geometry = GeometrySerde.deserialize((Slice)input);
        io.trino.geospatial.GeometryType type = io.trino.geospatial.GeometryType.getForEsriGeometryType((String)geometry.geometryType());
        if (type == io.trino.geospatial.GeometryType.GEOMETRY_COLLECTION) {
            double area = 0.0;
            GeometryCursor cursor = geometry.getEsriGeometryCursor();
            while (true) {
                Geometry esriGeometry;
                if ((esriGeometry = cursor.next()) == null) {
                    return area;
                }
                area += esriGeometry.calculateArea2D();
            }
        }
        return geometry.getEsriGeometry().calculateArea2D();
    }

    @Description(value="Returns a Geometry type object from Well-Known Text representation (WKT)")
    @ScalarFunction(value="ST_GeometryFromText")
    @SqlType(value="Geometry")
    public static Slice stGeometryFromText(@SqlType(value="varchar") Slice input) {
        return GeometrySerde.serialize((OGCGeometry)GeoFunctions.geometryFromText(input));
    }

    @Description(value="Returns a Geometry type object from Well-Known Binary representation (WKB)")
    @ScalarFunction(value="ST_GeomFromBinary")
    @SqlType(value="Geometry")
    public static Slice stGeomFromBinary(@SqlType(value="varbinary") Slice input) {
        return GeometrySerde.serialize((OGCGeometry)GeoFunctions.geomFromBinary(input));
    }

    @Description(value="Returns a Geometry type object from Spatial Framework for Hadoop representation")
    @ScalarFunction(value="geometry_from_hadoop_shape")
    @SqlType(value="Geometry")
    public static Slice geometryFromHadoopShape(@SqlType(value="varbinary") Slice input) {
        Objects.requireNonNull(input, "input is null");
        try {
            OGCGeometry geometry = OGCGeometry.fromEsriShape((ByteBuffer)GeoFunctions.getShapeByteBuffer(input));
            String wkt = GeometryEngine.geometryToWkt((Geometry)geometry.getEsriGeometry(), (int)GeoFunctions.getWktExportFlags(input));
            return GeometrySerde.serialize((OGCGeometry)OGCGeometry.fromText((String)wkt));
        }
        catch (IllegalArgumentException | IndexOutOfBoundsException | UnsupportedOperationException e) {
            throw new TrinoException((ErrorCodeSupplier)StandardErrorCode.INVALID_FUNCTION_ARGUMENT, "Invalid Hadoop shape", (Throwable)e);
        }
    }

    @Description(value="Converts a Geometry object to a SphericalGeography object")
    @ScalarFunction(value="to_spherical_geography")
    @SqlType(value="SphericalGeography")
    public static Slice toSphericalGeography(@SqlType(value="Geometry") Slice input) {
        Geometry subGeometry;
        OGCGeometry geometry;
        Envelope envelope = GeometrySerde.deserializeEnvelope((Slice)input);
        if (!envelope.isEmpty()) {
            GeoFunctions.checkLatitude(envelope.getYMin());
            GeoFunctions.checkLatitude(envelope.getYMax());
            GeoFunctions.checkLongitude(envelope.getXMin());
            GeoFunctions.checkLongitude(envelope.getXMax());
        }
        if ((geometry = GeometrySerde.deserialize((Slice)input)).is3D()) {
            throw new TrinoException((ErrorCodeSupplier)StandardErrorCode.INVALID_FUNCTION_ARGUMENT, "Cannot convert 3D geometry to a spherical geography");
        }
        GeometryCursor cursor = geometry.getEsriGeometryCursor();
        while ((subGeometry = cursor.next()) != null) {
            if (GEOMETRY_TYPES_FOR_SPHERICAL_GEOGRAPHY.contains(subGeometry.getType())) continue;
            throw new TrinoException((ErrorCodeSupplier)StandardErrorCode.INVALID_FUNCTION_ARGUMENT, "Cannot convert geometry of this type to spherical geography: " + String.valueOf(subGeometry.getType()));
        }
        return input;
    }

    @Description(value="Converts a SphericalGeography object to a Geometry object.")
    @ScalarFunction(value="to_geometry")
    @SqlType(value="Geometry")
    public static Slice toGeometry(@SqlType(value="SphericalGeography") Slice input) {
        return input;
    }

    @Description(value="Returns the Well-Known Text (WKT) representation of the geometry")
    @ScalarFunction(value="ST_AsText")
    @SqlType(value="varchar")
    public static Slice stAsText(@SqlType(value="Geometry") Slice input) {
        return Slices.utf8Slice((String)GeometrySerde.deserialize((Slice)input).asText());
    }

    @Description(value="Returns the Well-Known Binary (WKB) representation of the geometry")
    @ScalarFunction(value="ST_AsBinary")
    @SqlType(value="varbinary")
    public static Slice stAsBinary(@SqlType(value="Geometry") Slice input) {
        return Slices.wrappedHeapBuffer((ByteBuffer)GeometrySerde.deserialize((Slice)input).asBinary());
    }

    @SqlNullable
    @Description(value="Returns the geometry that represents all points whose distance from the specified geometry is less than or equal to the specified distance")
    @ScalarFunction(value="ST_Buffer")
    @SqlType(value="Geometry")
    public static Slice stBuffer(@SqlType(value="Geometry") Slice input, @SqlType(value="double") double distance) {
        if (Double.isNaN(distance)) {
            throw new TrinoException((ErrorCodeSupplier)StandardErrorCode.INVALID_FUNCTION_ARGUMENT, "distance is NaN");
        }
        if (distance < 0.0) {
            throw new TrinoException((ErrorCodeSupplier)StandardErrorCode.INVALID_FUNCTION_ARGUMENT, "distance is negative");
        }
        if (distance == 0.0) {
            return input;
        }
        OGCGeometry geometry = GeometrySerde.deserialize((Slice)input);
        if (geometry.isEmpty()) {
            return null;
        }
        return GeometrySerde.serialize((OGCGeometry)geometry.buffer(distance));
    }

    @Description(value="Returns the Point value that is the mathematical centroid of a Geometry")
    @ScalarFunction(value="ST_Centroid")
    @SqlType(value="Geometry")
    public static Slice stCentroid(@SqlType(value="Geometry") Slice input) {
        OGCGeometry geometry = GeometrySerde.deserialize((Slice)input);
        GeoFunctions.validateType("ST_Centroid", geometry, EnumSet.of(io.trino.geospatial.GeometryType.POINT, new io.trino.geospatial.GeometryType[]{io.trino.geospatial.GeometryType.MULTI_POINT, io.trino.geospatial.GeometryType.LINE_STRING, io.trino.geospatial.GeometryType.MULTI_LINE_STRING, io.trino.geospatial.GeometryType.POLYGON, io.trino.geospatial.GeometryType.MULTI_POLYGON}));
        io.trino.geospatial.GeometryType geometryType = io.trino.geospatial.GeometryType.getForEsriGeometryType((String)geometry.geometryType());
        if (geometryType == io.trino.geospatial.GeometryType.POINT) {
            return input;
        }
        int pointCount = ((MultiVertexGeometry)geometry.getEsriGeometry()).getPointCount();
        if (pointCount == 0) {
            return GeometrySerde.serialize((OGCGeometry)OGCGeometry.createFromEsriGeometry((Geometry)new Point(), (SpatialReference)geometry.getEsriSpatialReference()));
        }
        return GeometrySerde.serialize((OGCGeometry)geometry.centroid());
    }

    @Description(value="Returns the minimum convex geometry that encloses all input geometries")
    @ScalarFunction(value="ST_ConvexHull")
    @SqlType(value="Geometry")
    public static Slice stConvexHull(@SqlType(value="Geometry") Slice input) {
        OGCGeometry geometry = GeometrySerde.deserialize((Slice)input);
        if (geometry.isEmpty()) {
            return input;
        }
        if (io.trino.geospatial.GeometryType.getForEsriGeometryType((String)geometry.geometryType()) == io.trino.geospatial.GeometryType.POINT) {
            return input;
        }
        return GeometrySerde.serialize((OGCGeometry)geometry.convexHull());
    }

    @Description(value="Return the coordinate dimension of the Geometry")
    @ScalarFunction(value="ST_CoordDim")
    @SqlType(value="tinyint")
    public static long stCoordinateDimension(@SqlType(value="Geometry") Slice input) {
        return GeometrySerde.deserialize((Slice)input).coordinateDimension();
    }

    @Description(value="Returns the inherent dimension of this Geometry object, which must be less than or equal to the coordinate dimension")
    @ScalarFunction(value="ST_Dimension")
    @SqlType(value="tinyint")
    public static long stDimension(@SqlType(value="Geometry") Slice input) {
        return GeometrySerde.deserialize((Slice)input).dimension();
    }

    @SqlNullable
    @Description(value="Returns TRUE if the LineString or Multi-LineString's start and end points are coincident")
    @ScalarFunction(value="ST_IsClosed")
    @SqlType(value="boolean")
    public static Boolean stIsClosed(@SqlType(value="Geometry") Slice input) {
        OGCGeometry geometry = GeometrySerde.deserialize((Slice)input);
        GeoFunctions.validateType("ST_IsClosed", geometry, EnumSet.of(io.trino.geospatial.GeometryType.LINE_STRING, io.trino.geospatial.GeometryType.MULTI_LINE_STRING));
        MultiPath lines = (MultiPath)geometry.getEsriGeometry();
        int pathCount = lines.getPathCount();
        for (int i = 0; i < pathCount; ++i) {
            Point start = lines.getPoint(lines.getPathStart(i));
            Point end = lines.getPoint(lines.getPathEnd(i) - 1);
            if (end.equals((Object)start)) continue;
            return false;
        }
        return true;
    }

    @SqlNullable
    @Description(value="Returns TRUE if this Geometry is an empty geometrycollection, polygon, point etc")
    @ScalarFunction(value="ST_IsEmpty")
    @SqlType(value="boolean")
    public static Boolean stIsEmpty(@SqlType(value="Geometry") Slice input) {
        return GeometrySerde.deserializeEnvelope((Slice)input).isEmpty();
    }

    @Description(value="Returns TRUE if this Geometry has no anomalous geometric points, such as self intersection or self tangency")
    @ScalarFunction(value="ST_IsSimple")
    @SqlType(value="boolean")
    public static boolean stIsSimple(@SqlType(value="Geometry") Slice input) {
        OGCGeometry geometry = GeometrySerde.deserialize((Slice)input);
        return geometry.isEmpty() || geometry.isSimple();
    }

    @Description(value="Returns true if the input geometry is well formed")
    @ScalarFunction(value="ST_IsValid")
    @SqlType(value="boolean")
    public static boolean stIsValid(@SqlType(value="Geometry") Slice input) {
        Geometry geometry;
        GeometryCursor cursor = GeometrySerde.deserialize((Slice)input).getEsriGeometryCursor();
        do {
            if ((geometry = cursor.next()) != null) continue;
            return true;
        } while (OperatorSimplifyOGC.local().isSimpleOGC(geometry, null, true, null, null));
        return false;
    }

    @Description(value="Returns the reason for why the input geometry is not valid. Returns null if the input is valid.")
    @ScalarFunction(value="geometry_invalid_reason")
    @SqlType(value="varchar")
    @SqlNullable
    public static Slice invalidReason(@SqlType(value="Geometry") Slice input) {
        Geometry geometry;
        GeometryCursor cursor = GeometrySerde.deserialize((Slice)input).getEsriGeometryCursor();
        NonSimpleResult result = new NonSimpleResult();
        do {
            if ((geometry = cursor.next()) != null) continue;
            return null;
        } while (OperatorSimplifyOGC.local().isSimpleOGC(geometry, null, true, result, null));
        String reasonText = NON_SIMPLE_REASONS.getOrDefault(result.m_reason, result.m_reason.name());
        if (!(geometry instanceof MultiVertexGeometry)) {
            return Slices.utf8Slice((String)reasonText);
        }
        MultiVertexGeometry multiVertexGeometry = (MultiVertexGeometry)geometry;
        if (result.m_vertexIndex1 >= 0 && result.m_vertexIndex2 >= 0) {
            Point point1 = multiVertexGeometry.getPoint(result.m_vertexIndex1);
            Point point2 = multiVertexGeometry.getPoint(result.m_vertexIndex2);
            return Slices.utf8Slice((String)String.format("%s at or near (%s %s) and (%s %s)", reasonText, point1.getX(), point1.getY(), point2.getX(), point2.getY()));
        }
        if (result.m_vertexIndex1 >= 0) {
            Point point = multiVertexGeometry.getPoint(result.m_vertexIndex1);
            return Slices.utf8Slice((String)String.format("%s at or near (%s %s)", reasonText, point.getX(), point.getY()));
        }
        return Slices.utf8Slice((String)reasonText);
    }

    @Description(value="Returns the length of a LineString or Multi-LineString using Euclidean measurement on a 2D plane (based on spatial ref) in projected units")
    @ScalarFunction(value="ST_Length")
    @SqlType(value="double")
    public static double stLength(@SqlType(value="Geometry") Slice input) {
        OGCGeometry geometry = GeometrySerde.deserialize((Slice)input);
        GeoFunctions.validateType("ST_Length", geometry, EnumSet.of(io.trino.geospatial.GeometryType.LINE_STRING, io.trino.geospatial.GeometryType.MULTI_LINE_STRING));
        return geometry.getEsriGeometry().calculateLength2D();
    }

    @SqlNullable
    @Description(value="Returns the great-circle length in meters of a linestring or multi-linestring on Earth's surface")
    @ScalarFunction(value="ST_Length")
    @SqlType(value="double")
    public static Double stSphericalLength(@SqlType(value="SphericalGeography") Slice input) {
        OGCGeometry geometry = GeometrySerde.deserialize((Slice)input);
        if (geometry.isEmpty()) {
            return null;
        }
        GeoFunctions.validateSphericalType("ST_Length", geometry, EnumSet.of(io.trino.geospatial.GeometryType.LINE_STRING, io.trino.geospatial.GeometryType.MULTI_LINE_STRING));
        MultiPath lineString = (MultiPath)geometry.getEsriGeometry();
        double sum = 0.0;
        for (int path = 0; path < lineString.getPathCount(); ++path) {
            if (lineString.getPathSize(path) < 2) continue;
            int pathStart = lineString.getPathStart(path);
            Point previous = lineString.getPoint(pathStart);
            for (int i = pathStart + 1; i < lineString.getPathEnd(path); ++i) {
                Point next = lineString.getPoint(i);
                sum += GeoFunctions.greatCircleDistance(previous.getY(), previous.getX(), next.getY(), next.getX());
                previous = next;
            }
        }
        return sum * 1000.0;
    }

    @SqlNullable
    @Description(value="Returns a float between 0 and 1 representing the location of the closest point on the LineString to the given Point, as a fraction of total 2d line length.")
    @ScalarFunction(value="line_locate_point")
    @SqlType(value="double")
    public static Double lineLocatePoint(@SqlType(value="Geometry") Slice lineSlice, @SqlType(value="Geometry") Slice pointSlice) {
        org.locationtech.jts.geom.Geometry line = JtsGeometrySerde.deserialize((Slice)lineSlice);
        org.locationtech.jts.geom.Geometry point = JtsGeometrySerde.deserialize((Slice)pointSlice);
        if (line.isEmpty() || point.isEmpty()) {
            return null;
        }
        io.trino.geospatial.GeometryType lineType = io.trino.geospatial.GeometryType.getForJtsGeometryType((String)line.getGeometryType());
        if (lineType != io.trino.geospatial.GeometryType.LINE_STRING && lineType != io.trino.geospatial.GeometryType.MULTI_LINE_STRING) {
            throw new TrinoException((ErrorCodeSupplier)StandardErrorCode.INVALID_FUNCTION_ARGUMENT, String.format("First argument to line_locate_point must be a LineString or a MultiLineString. Got: %s", line.getGeometryType()));
        }
        io.trino.geospatial.GeometryType pointType = io.trino.geospatial.GeometryType.getForJtsGeometryType((String)point.getGeometryType());
        if (pointType != io.trino.geospatial.GeometryType.POINT) {
            throw new TrinoException((ErrorCodeSupplier)StandardErrorCode.INVALID_FUNCTION_ARGUMENT, String.format("Second argument to line_locate_point must be a Point. Got: %s", point.getGeometryType()));
        }
        return new LengthIndexedLine(line).indexOf(point.getCoordinate()) / line.getLength();
    }

    @SqlNullable
    @Description(value="Returns a Point interpolated along a LineString at the fraction given.")
    @ScalarFunction(value="line_interpolate_point")
    @SqlType(value="Geometry")
    public static Slice lineInterpolatePoint(@SqlType(value="Geometry") Slice input, @SqlType(value="double") double distanceFraction) {
        OGCGeometry geometry = GeometrySerde.deserialize((Slice)input);
        if (geometry.isEmpty()) {
            return null;
        }
        List<Point> interpolatedPoints = GeoFunctions.interpolatePoints(geometry, distanceFraction, false);
        return GeometrySerde.serialize((OGCGeometry)OGCGeometry.createFromEsriGeometry((Geometry)((Geometry)interpolatedPoints.get(0)), null));
    }

    @SqlNullable
    @Description(value="Returns an array of Points interpolated along a LineString.")
    @ScalarFunction(value="line_interpolate_points")
    @SqlType(value="array(Geometry)")
    public static Block lineInterpolatePoints(@SqlType(value="Geometry") Slice input, @SqlType(value="double") double fractionStep) {
        OGCGeometry geometry = GeometrySerde.deserialize((Slice)input);
        if (geometry.isEmpty()) {
            return null;
        }
        List<Point> interpolatedPoints = GeoFunctions.interpolatePoints(geometry, fractionStep, true);
        VariableWidthBlockBuilder blockBuilder = GeometryType.GEOMETRY.createBlockBuilder(null, interpolatedPoints.size());
        for (Point point : interpolatedPoints) {
            GeometryType.GEOMETRY.writeSlice((BlockBuilder)blockBuilder, GeometrySerde.serialize((OGCGeometry)OGCGeometry.createFromEsriGeometry((Geometry)point, null)));
        }
        return blockBuilder.build();
    }

    private static List<Point> interpolatePoints(OGCGeometry geometry, double fractionStep, boolean repeated) {
        GeoFunctions.validateType("line_interpolate_point", geometry, EnumSet.of(io.trino.geospatial.GeometryType.LINE_STRING));
        if (fractionStep < 0.0 || fractionStep > 1.0) {
            throw new TrinoException((ErrorCodeSupplier)StandardErrorCode.INVALID_FUNCTION_ARGUMENT, "fraction must be between 0 and 1");
        }
        MultiPath path = (MultiPath)geometry.getEsriGeometry();
        if (fractionStep == 0.0) {
            return Collections.singletonList(path.getPoint(0));
        }
        if (fractionStep == 1.0) {
            return Collections.singletonList(path.getPoint(path.getPointCount() - 1));
        }
        int pointCount = repeated ? (int)Math.floor(1.0 / fractionStep) : 1;
        ArrayList<Point> interpolatedPoints = new ArrayList<Point>(pointCount);
        double lineStringLength = path.calculateLength2D();
        Point previous = path.getPoint(0);
        double fractionConsumed = 0.0;
        double fractionIncrement = fractionStep;
        for (int i = 1; i < path.getPointCount() && interpolatedPoints.size() < pointCount; ++i) {
            Point current = path.getPoint(i);
            double segmentLengthFraction = GeometryEngine.distance((Geometry)previous, (Geometry)current, null) / lineStringLength;
            while (fractionStep < fractionConsumed + segmentLengthFraction && interpolatedPoints.size() < pointCount) {
                double segmentFraction = (fractionStep - fractionConsumed) / segmentLengthFraction;
                Point point = new Point();
                point.setX(previous.getX() + (current.getX() - previous.getX()) * segmentFraction);
                point.setY(previous.getY() + (current.getY() - previous.getY()) * segmentFraction);
                interpolatedPoints.add(point);
                fractionStep += fractionIncrement;
            }
            fractionConsumed += segmentLengthFraction;
            previous = current;
        }
        if (interpolatedPoints.size() < pointCount) {
            interpolatedPoints.add(path.getPoint(path.getPointCount() - 1));
        }
        return interpolatedPoints;
    }

    @SqlNullable
    @Description(value="Returns X maxima of a bounding box of a Geometry")
    @ScalarFunction(value="ST_XMax")
    @SqlType(value="double")
    public static Double stXMax(@SqlType(value="Geometry") Slice input) {
        Envelope envelope = GeometrySerde.deserializeEnvelope((Slice)input);
        if (envelope.isEmpty()) {
            return null;
        }
        return envelope.getXMax();
    }

    @SqlNullable
    @Description(value="Returns Y maxima of a bounding box of a Geometry")
    @ScalarFunction(value="ST_YMax")
    @SqlType(value="double")
    public static Double stYMax(@SqlType(value="Geometry") Slice input) {
        Envelope envelope = GeometrySerde.deserializeEnvelope((Slice)input);
        if (envelope.isEmpty()) {
            return null;
        }
        return envelope.getYMax();
    }

    @SqlNullable
    @Description(value="Returns X minima of a bounding box of a Geometry")
    @ScalarFunction(value="ST_XMin")
    @SqlType(value="double")
    public static Double stXMin(@SqlType(value="Geometry") Slice input) {
        Envelope envelope = GeometrySerde.deserializeEnvelope((Slice)input);
        if (envelope.isEmpty()) {
            return null;
        }
        return envelope.getXMin();
    }

    @SqlNullable
    @Description(value="Returns Y minima of a bounding box of a Geometry")
    @ScalarFunction(value="ST_YMin")
    @SqlType(value="double")
    public static Double stYMin(@SqlType(value="Geometry") Slice input) {
        Envelope envelope = GeometrySerde.deserializeEnvelope((Slice)input);
        if (envelope.isEmpty()) {
            return null;
        }
        return envelope.getYMin();
    }

    @SqlNullable
    @Description(value="Returns the cardinality of the collection of interior rings of a polygon")
    @ScalarFunction(value="ST_NumInteriorRing")
    @SqlType(value="bigint")
    public static Long stNumInteriorRings(@SqlType(value="Geometry") Slice input) {
        OGCGeometry geometry = GeometrySerde.deserialize((Slice)input);
        GeoFunctions.validateType("ST_NumInteriorRing", geometry, EnumSet.of(io.trino.geospatial.GeometryType.POLYGON));
        if (geometry.isEmpty()) {
            return null;
        }
        return ((OGCPolygon)geometry).numInteriorRing();
    }

    @SqlNullable
    @Description(value="Returns an array of interior rings of a polygon")
    @ScalarFunction(value="ST_InteriorRings")
    @SqlType(value="array(Geometry)")
    public static Block stInteriorRings(@SqlType(value="Geometry") Slice input) {
        OGCGeometry geometry = GeometrySerde.deserialize((Slice)input);
        GeoFunctions.validateType("ST_InteriorRings", geometry, EnumSet.of(io.trino.geospatial.GeometryType.POLYGON));
        if (geometry.isEmpty()) {
            return null;
        }
        OGCPolygon polygon = (OGCPolygon)geometry;
        VariableWidthBlockBuilder blockBuilder = GeometryType.GEOMETRY.createBlockBuilder(null, polygon.numInteriorRing());
        for (int i = 0; i < polygon.numInteriorRing(); ++i) {
            GeometryType.GEOMETRY.writeSlice((BlockBuilder)blockBuilder, GeometrySerde.serialize((OGCGeometry)polygon.interiorRingN(i)));
        }
        return blockBuilder.build();
    }

    @Description(value="Returns the cardinality of the geometry collection")
    @ScalarFunction(value="ST_NumGeometries")
    @SqlType(value="integer")
    public static long stNumGeometries(@SqlType(value="Geometry") Slice input) {
        OGCGeometry geometry = GeometrySerde.deserialize((Slice)input);
        if (geometry.isEmpty()) {
            return 0L;
        }
        io.trino.geospatial.GeometryType type = io.trino.geospatial.GeometryType.getForEsriGeometryType((String)geometry.geometryType());
        if (!type.isMultitype()) {
            return 1L;
        }
        return ((OGCGeometryCollection)geometry).numGeometries();
    }

    @Description(value="Returns a geometry that represents the point set union of the input geometries.")
    @ScalarFunction(value="ST_Union")
    @SqlType(value="Geometry")
    public static Slice stUnion(@SqlType(value="Geometry") Slice left, @SqlType(value="Geometry") Slice right) {
        return GeoFunctions.stUnion((Iterable<Slice>)ImmutableList.of((Object)left, (Object)right));
    }

    @Description(value="Returns a geometry that represents the point set union of the input geometries.")
    @ScalarFunction(value="geometry_union")
    @SqlType(value="Geometry")
    public static Slice geometryUnion(@SqlType(value="array(Geometry)") Block input) {
        return GeoFunctions.stUnion(GeoFunctions.getGeometrySlicesFromBlock(input));
    }

    private static Slice stUnion(Iterable<Slice> slices) {
        ListeningGeometryCursor[] cursorsByDimension = new ListeningGeometryCursor[3];
        GeometryCursor[] operatorsByDimension = new GeometryCursor[3];
        Arrays.setAll(cursorsByDimension, i -> new ListeningGeometryCursor());
        Arrays.setAll(operatorsByDimension, i -> OperatorUnion.local().execute((GeometryCursor)cursorsByDimension[i], null, null));
        Iterator<Slice> slicesIterator = slices.iterator();
        if (!slicesIterator.hasNext()) {
            return null;
        }
        while (slicesIterator.hasNext()) {
            Slice slice = slicesIterator.next();
            if (slice.getInput().available() == 0) continue;
            for (OGCGeometry geometry : GeoFunctions.flattenCollection(GeometrySerde.deserialize((Slice)slice))) {
                int dimension = geometry.dimension();
                cursorsByDimension[dimension].tick(geometry.getEsriGeometry());
                operatorsByDimension[dimension].tock();
            }
        }
        ArrayList<OGCGeometry> outputs = new ArrayList<OGCGeometry>();
        for (GeometryCursor operator : operatorsByDimension) {
            OGCGeometry unionedGeometry = OGCGeometry.createFromEsriGeometry((Geometry)operator.next(), null);
            if (unionedGeometry == null) continue;
            outputs.add(unionedGeometry);
        }
        if (outputs.size() == 1) {
            return GeometrySerde.serialize((OGCGeometry)((OGCGeometry)outputs.get(0)));
        }
        return GeometrySerde.serialize((OGCGeometry)new OGCConcreteGeometryCollection(outputs, null).flattenAndRemoveOverlaps().reduceFromMulti());
    }

    @SqlNullable
    @Description(value="Returns the geometry element at the specified index (indices started with 1)")
    @ScalarFunction(value="ST_GeometryN")
    @SqlType(value="Geometry")
    public static Slice stGeometryN(@SqlType(value="Geometry") Slice input, @SqlType(value="integer") long index) {
        OGCGeometry geometry = GeometrySerde.deserialize((Slice)input);
        if (geometry.isEmpty()) {
            return null;
        }
        io.trino.geospatial.GeometryType type = io.trino.geospatial.GeometryType.getForEsriGeometryType((String)geometry.geometryType());
        if (!type.isMultitype()) {
            if (index == 1L) {
                return input;
            }
            return null;
        }
        OGCGeometryCollection geometryCollection = (OGCGeometryCollection)geometry;
        if (index < 1L || index > (long)geometryCollection.numGeometries()) {
            return null;
        }
        OGCGeometry ogcGeometry = geometryCollection.geometryN((int)index - 1);
        return GeometrySerde.serialize((OGCGeometry)ogcGeometry);
    }

    @SqlNullable
    @Description(value="Returns the vertex of a linestring at the specified index (indices started with 1) ")
    @ScalarFunction(value="ST_PointN")
    @SqlType(value="Geometry")
    public static Slice stPointN(@SqlType(value="Geometry") Slice input, @SqlType(value="integer") long index) {
        OGCGeometry geometry = GeometrySerde.deserialize((Slice)input);
        GeoFunctions.validateType("ST_PointN", geometry, EnumSet.of(io.trino.geospatial.GeometryType.LINE_STRING));
        OGCLineString linestring = (OGCLineString)geometry;
        if (index < 1L || index > (long)linestring.numPoints()) {
            return null;
        }
        return GeometrySerde.serialize((OGCGeometry)linestring.pointN(Math.toIntExact(index) - 1));
    }

    @SqlNullable
    @Description(value="Returns an array of geometries in the specified collection")
    @ScalarFunction(value="ST_Geometries")
    @SqlType(value="array(Geometry)")
    public static Block stGeometries(@SqlType(value="Geometry") Slice input) {
        OGCGeometry geometry = GeometrySerde.deserialize((Slice)input);
        if (geometry.isEmpty()) {
            return null;
        }
        io.trino.geospatial.GeometryType type = io.trino.geospatial.GeometryType.getForEsriGeometryType((String)geometry.geometryType());
        if (!type.isMultitype()) {
            VariableWidthBlockBuilder blockBuilder = GeometryType.GEOMETRY.createBlockBuilder(null, 1);
            GeometryType.GEOMETRY.writeSlice((BlockBuilder)blockBuilder, GeometrySerde.serialize((OGCGeometry)geometry));
            return blockBuilder.build();
        }
        OGCGeometryCollection collection = (OGCGeometryCollection)geometry;
        VariableWidthBlockBuilder blockBuilder = GeometryType.GEOMETRY.createBlockBuilder(null, collection.numGeometries());
        for (int i = 0; i < collection.numGeometries(); ++i) {
            GeometryType.GEOMETRY.writeSlice((BlockBuilder)blockBuilder, GeometrySerde.serialize((OGCGeometry)collection.geometryN(i)));
        }
        return blockBuilder.build();
    }

    @SqlNullable
    @Description(value="Returns the interior ring element at the specified index (indices start at 1)")
    @ScalarFunction(value="ST_InteriorRingN")
    @SqlType(value="Geometry")
    public static Slice stInteriorRingN(@SqlType(value="Geometry") Slice input, @SqlType(value="integer") long index) {
        OGCGeometry geometry = GeometrySerde.deserialize((Slice)input);
        GeoFunctions.validateType("ST_InteriorRingN", geometry, EnumSet.of(io.trino.geospatial.GeometryType.POLYGON));
        OGCPolygon polygon = (OGCPolygon)geometry;
        if (index < 1L || index > (long)polygon.numInteriorRing()) {
            return null;
        }
        OGCLineString interiorRing = polygon.interiorRingN(Math.toIntExact(index) - 1);
        return GeometrySerde.serialize((OGCGeometry)interiorRing);
    }

    @Description(value="Returns the number of points in a Geometry")
    @ScalarFunction(value="ST_NumPoints")
    @SqlType(value="bigint")
    public static long stNumPoints(@SqlType(value="Geometry") Slice input) {
        return GeometryUtils.getPointCount((OGCGeometry)GeometrySerde.deserialize((Slice)input));
    }

    @SqlNullable
    @Description(value="Returns TRUE if and only if the line is closed and simple")
    @ScalarFunction(value="ST_IsRing")
    @SqlType(value="boolean")
    public static Boolean stIsRing(@SqlType(value="Geometry") Slice input) {
        OGCGeometry geometry = GeometrySerde.deserialize((Slice)input);
        GeoFunctions.validateType("ST_IsRing", geometry, EnumSet.of(io.trino.geospatial.GeometryType.LINE_STRING));
        OGCLineString line = (OGCLineString)geometry;
        return line.isClosed() && line.isSimple();
    }

    @SqlNullable
    @Description(value="Returns the first point of a LINESTRING geometry as a Point")
    @ScalarFunction(value="ST_StartPoint")
    @SqlType(value="Geometry")
    public static Slice stStartPoint(@SqlType(value="Geometry") Slice input) {
        OGCGeometry geometry = GeometrySerde.deserialize((Slice)input);
        GeoFunctions.validateType("ST_StartPoint", geometry, EnumSet.of(io.trino.geospatial.GeometryType.LINE_STRING));
        if (geometry.isEmpty()) {
            return null;
        }
        MultiPath lines = (MultiPath)geometry.getEsriGeometry();
        SpatialReference reference = geometry.getEsriSpatialReference();
        return GeometrySerde.serialize((OGCGeometry)OGCGeometry.createFromEsriGeometry((Geometry)lines.getPoint(0), (SpatialReference)reference));
    }

    @Description(value="Returns a \"simplified\" version of the given geometry")
    @ScalarFunction(value="simplify_geometry")
    @SqlType(value="Geometry")
    public static Slice simplifyGeometry(@SqlType(value="Geometry") Slice input, @SqlType(value="double") double distanceTolerance) {
        if (Double.isNaN(distanceTolerance)) {
            throw new TrinoException((ErrorCodeSupplier)StandardErrorCode.INVALID_FUNCTION_ARGUMENT, "distanceTolerance is NaN");
        }
        if (distanceTolerance < 0.0) {
            throw new TrinoException((ErrorCodeSupplier)StandardErrorCode.INVALID_FUNCTION_ARGUMENT, "distanceTolerance is negative");
        }
        if (distanceTolerance == 0.0) {
            return input;
        }
        return JtsGeometrySerde.serialize((org.locationtech.jts.geom.Geometry)TopologyPreservingSimplifier.simplify((org.locationtech.jts.geom.Geometry)JtsGeometrySerde.deserialize((Slice)input), (double)distanceTolerance));
    }

    @SqlNullable
    @Description(value="Returns the last point of a LINESTRING geometry as a Point")
    @ScalarFunction(value="ST_EndPoint")
    @SqlType(value="Geometry")
    public static Slice stEndPoint(@SqlType(value="Geometry") Slice input) {
        OGCGeometry geometry = GeometrySerde.deserialize((Slice)input);
        GeoFunctions.validateType("ST_EndPoint", geometry, EnumSet.of(io.trino.geospatial.GeometryType.LINE_STRING));
        if (geometry.isEmpty()) {
            return null;
        }
        MultiPath lines = (MultiPath)geometry.getEsriGeometry();
        SpatialReference reference = geometry.getEsriSpatialReference();
        return GeometrySerde.serialize((OGCGeometry)OGCGeometry.createFromEsriGeometry((Geometry)lines.getPoint(lines.getPointCount() - 1), (SpatialReference)reference));
    }

    @SqlNullable
    @Description(value="Returns an array of points in a geometry")
    @ScalarFunction(value="ST_Points")
    @SqlType(value="array(Geometry)")
    public static Block stPoints(@SqlType(value="Geometry") Slice input) {
        org.locationtech.jts.geom.Geometry geometry = JtsGeometrySerde.deserialize((Slice)input);
        GeoFunctions.validateType("ST_Points", geometry, VALID_TYPES_FOR_ST_POINTS);
        if (geometry.isEmpty()) {
            return null;
        }
        int pointCount = geometry.getNumPoints();
        VariableWidthBlockBuilder blockBuilder = GeometryType.GEOMETRY.createBlockBuilder(null, pointCount);
        GeoFunctions.buildPointsBlock(geometry, (BlockBuilder)blockBuilder);
        return blockBuilder.build();
    }

    private static void buildPointsBlock(org.locationtech.jts.geom.Geometry geometry, BlockBuilder blockBuilder) {
        io.trino.geospatial.GeometryType type = io.trino.geospatial.GeometryType.getForJtsGeometryType((String)geometry.getGeometryType());
        if (type == io.trino.geospatial.GeometryType.POINT) {
            GeometryType.GEOMETRY.writeSlice(blockBuilder, JtsGeometrySerde.serialize((org.locationtech.jts.geom.Geometry)geometry));
        } else if (type == io.trino.geospatial.GeometryType.GEOMETRY_COLLECTION) {
            GeometryCollection collection = (GeometryCollection)geometry;
            for (int i = 0; i < collection.getNumGeometries(); ++i) {
                org.locationtech.jts.geom.Geometry entry = collection.getGeometryN(i);
                GeoFunctions.buildPointsBlock(entry, blockBuilder);
            }
        } else {
            Coordinate[] vertices;
            GeometryFactory geometryFactory = geometry.getFactory();
            for (Coordinate coordinate : vertices = geometry.getCoordinates()) {
                GeometryType.GEOMETRY.writeSlice(blockBuilder, JtsGeometrySerde.serialize((org.locationtech.jts.geom.Geometry)geometryFactory.createPoint(coordinate)));
            }
        }
    }

    @SqlNullable
    @Description(value="Return the X coordinate of the point")
    @ScalarFunction(value="ST_X")
    @SqlType(value="double")
    public static Double stX(@SqlType(value="Geometry") Slice input) {
        OGCGeometry geometry = GeometrySerde.deserialize((Slice)input);
        GeoFunctions.validateType("ST_X", geometry, EnumSet.of(io.trino.geospatial.GeometryType.POINT));
        if (geometry.isEmpty()) {
            return null;
        }
        return ((OGCPoint)geometry).X();
    }

    @SqlNullable
    @Description(value="Return the Y coordinate of the point")
    @ScalarFunction(value="ST_Y")
    @SqlType(value="double")
    public static Double stY(@SqlType(value="Geometry") Slice input) {
        OGCGeometry geometry = GeometrySerde.deserialize((Slice)input);
        GeoFunctions.validateType("ST_Y", geometry, EnumSet.of(io.trino.geospatial.GeometryType.POINT));
        if (geometry.isEmpty()) {
            return null;
        }
        return ((OGCPoint)geometry).Y();
    }

    @Description(value="Returns the closure of the combinatorial boundary of this Geometry")
    @ScalarFunction(value="ST_Boundary")
    @SqlType(value="Geometry")
    public static Slice stBoundary(@SqlType(value="Geometry") Slice input) {
        OGCGeometry geometry = GeometrySerde.deserialize((Slice)input);
        if (geometry.isEmpty() && io.trino.geospatial.GeometryType.getForEsriGeometryType((String)geometry.geometryType()) == io.trino.geospatial.GeometryType.LINE_STRING) {
            return EMPTY_MULTIPOINT;
        }
        return GeometrySerde.serialize((OGCGeometry)geometry.boundary());
    }

    @Description(value="Returns the bounding rectangular polygon of a Geometry")
    @ScalarFunction(value="ST_Envelope")
    @SqlType(value="Geometry")
    public static Slice stEnvelope(@SqlType(value="Geometry") Slice input) {
        Envelope envelope = GeometrySerde.deserializeEnvelope((Slice)input);
        if (envelope.isEmpty()) {
            return EMPTY_POLYGON;
        }
        return GeometrySerde.serialize((Envelope)envelope);
    }

    @SqlNullable
    @Description(value="Returns the lower left and upper right corners of bounding rectangular polygon of a Geometry")
    @ScalarFunction(value="ST_EnvelopeAsPts")
    @SqlType(value="array(Geometry)")
    public static Block stEnvelopeAsPts(@SqlType(value="Geometry") Slice input) {
        Envelope envelope = GeometrySerde.deserializeEnvelope((Slice)input);
        if (envelope.isEmpty()) {
            return null;
        }
        VariableWidthBlockBuilder blockBuilder = GeometryType.GEOMETRY.createBlockBuilder(null, 2);
        Point lowerLeftCorner = new Point(envelope.getXMin(), envelope.getYMin());
        Point upperRightCorner = new Point(envelope.getXMax(), envelope.getYMax());
        GeometryType.GEOMETRY.writeSlice((BlockBuilder)blockBuilder, GeometrySerde.serialize((OGCGeometry)OGCGeometry.createFromEsriGeometry((Geometry)lowerLeftCorner, null, (boolean)false)));
        GeometryType.GEOMETRY.writeSlice((BlockBuilder)blockBuilder, GeometrySerde.serialize((OGCGeometry)OGCGeometry.createFromEsriGeometry((Geometry)upperRightCorner, null, (boolean)false)));
        return blockBuilder.build();
    }

    @Description(value="Returns the Geometry value that represents the point set difference of two geometries")
    @ScalarFunction(value="ST_Difference")
    @SqlType(value="Geometry")
    public static Slice stDifference(@SqlType(value="Geometry") Slice left, @SqlType(value="Geometry") Slice right) {
        OGCGeometry leftGeometry = GeometrySerde.deserialize((Slice)left);
        OGCGeometry rightGeometry = GeometrySerde.deserialize((Slice)right);
        GeoFunctions.verifySameSpatialReference(leftGeometry, rightGeometry);
        return GeometrySerde.serialize((OGCGeometry)leftGeometry.difference(rightGeometry));
    }

    @SqlNullable
    @Description(value="Returns the 2-dimensional cartesian minimum distance (based on spatial ref) between two geometries in projected units")
    @ScalarFunction(value="ST_Distance")
    @SqlType(value="double")
    public static Double stDistance(@SqlType(value="Geometry") Slice left, @SqlType(value="Geometry") Slice right) {
        OGCGeometry leftGeometry = GeometrySerde.deserialize((Slice)left);
        OGCGeometry rightGeometry = GeometrySerde.deserialize((Slice)right);
        GeoFunctions.verifySameSpatialReference(leftGeometry, rightGeometry);
        return leftGeometry.isEmpty() || rightGeometry.isEmpty() ? null : Double.valueOf(leftGeometry.distance(rightGeometry));
    }

    @SqlNullable
    @Description(value="Return the closest points on the two geometries")
    @ScalarFunction(value="geometry_nearest_points")
    @SqlType(value="row(Geometry,Geometry)")
    public static SqlRow geometryNearestPoints(@SqlType(value="Geometry") Slice left, @SqlType(value="Geometry") Slice right) {
        org.locationtech.jts.geom.Geometry leftGeometry = JtsGeometrySerde.deserialize((Slice)left);
        org.locationtech.jts.geom.Geometry rightGeometry = JtsGeometrySerde.deserialize((Slice)right);
        if (leftGeometry.isEmpty() || rightGeometry.isEmpty()) {
            return null;
        }
        RowType rowType = RowType.anonymous((List)ImmutableList.of((Object)((Object)GeometryType.GEOMETRY), (Object)((Object)GeometryType.GEOMETRY)));
        GeometryFactory geometryFactory = leftGeometry.getFactory();
        Coordinate[] nearestCoordinates = DistanceOp.nearestPoints((org.locationtech.jts.geom.Geometry)leftGeometry, (org.locationtech.jts.geom.Geometry)rightGeometry);
        return RowValueBuilder.buildRowValue((RowType)rowType, fieldBuilders -> {
            GeometryType.GEOMETRY.writeSlice((BlockBuilder)fieldBuilders.get(0), JtsGeometrySerde.serialize((org.locationtech.jts.geom.Geometry)geometryFactory.createPoint(nearestCoordinates[0])));
            GeometryType.GEOMETRY.writeSlice((BlockBuilder)fieldBuilders.get(1), JtsGeometrySerde.serialize((org.locationtech.jts.geom.Geometry)geometryFactory.createPoint(nearestCoordinates[1])));
        });
    }

    @SqlNullable
    @Description(value="Returns a line string representing the exterior ring of the POLYGON")
    @ScalarFunction(value="ST_ExteriorRing")
    @SqlType(value="Geometry")
    public static Slice stExteriorRing(@SqlType(value="Geometry") Slice input) {
        OGCGeometry geometry = GeometrySerde.deserialize((Slice)input);
        GeoFunctions.validateType("ST_ExteriorRing", geometry, EnumSet.of(io.trino.geospatial.GeometryType.POLYGON));
        if (geometry.isEmpty()) {
            return null;
        }
        return GeometrySerde.serialize((OGCGeometry)((OGCPolygon)geometry).exteriorRing());
    }

    @Description(value="Returns the Geometry value that represents the point set intersection of two Geometries")
    @ScalarFunction(value="ST_Intersection")
    @SqlType(value="Geometry")
    public static Slice stIntersection(@SqlType(value="Geometry") Slice left, @SqlType(value="Geometry") Slice right) {
        if (GeometrySerde.deserializeType((Slice)left) == GeometrySerializationType.ENVELOPE && GeometrySerde.deserializeType((Slice)right) == GeometrySerializationType.ENVELOPE) {
            Envelope rightEnvelope;
            Envelope leftEnvelope = GeometrySerde.deserializeEnvelope((Slice)left);
            if (!leftEnvelope.intersect(rightEnvelope = GeometrySerde.deserializeEnvelope((Slice)right))) {
                return EMPTY_POLYGON;
            }
            Envelope intersection = leftEnvelope;
            if (intersection.getXMin() == intersection.getXMax()) {
                if (intersection.getYMin() == intersection.getYMax()) {
                    return GeometrySerde.serialize((OGCGeometry)OGCGeometry.createFromEsriGeometry((Geometry)new Point(intersection.getXMin(), intersection.getXMax()), null));
                }
                return GeometrySerde.serialize((OGCGeometry)OGCGeometry.createFromEsriGeometry((Geometry)new Polyline(new Point(intersection.getXMin(), intersection.getYMin()), new Point(intersection.getXMin(), intersection.getYMax())), null));
            }
            if (intersection.getYMin() == intersection.getYMax()) {
                return GeometrySerde.serialize((OGCGeometry)OGCGeometry.createFromEsriGeometry((Geometry)new Polyline(new Point(intersection.getXMin(), intersection.getYMin()), new Point(intersection.getXMax(), intersection.getYMin())), null));
            }
            return GeometrySerde.serialize((Envelope)intersection);
        }
        OGCGeometry leftGeometry = GeometrySerde.deserialize((Slice)left);
        OGCGeometry rightGeometry = GeometrySerde.deserialize((Slice)right);
        GeoFunctions.verifySameSpatialReference(leftGeometry, rightGeometry);
        return GeometrySerde.serialize((OGCGeometry)leftGeometry.intersection(rightGeometry));
    }

    @Description(value="Returns the Geometry value that represents the point set symmetric difference of two Geometries")
    @ScalarFunction(value="ST_SymDifference")
    @SqlType(value="Geometry")
    public static Slice stSymmetricDifference(@SqlType(value="Geometry") Slice left, @SqlType(value="Geometry") Slice right) {
        OGCGeometry leftGeometry = GeometrySerde.deserialize((Slice)left);
        OGCGeometry rightGeometry = GeometrySerde.deserialize((Slice)right);
        GeoFunctions.verifySameSpatialReference(leftGeometry, rightGeometry);
        return GeometrySerde.serialize((OGCGeometry)leftGeometry.symDifference(rightGeometry));
    }

    @SqlNullable
    @Description(value="Returns TRUE if and only if no points of right lie in the exterior of left, and at least one point of the interior of left lies in the interior of right")
    @ScalarFunction(value="ST_Contains")
    @SqlType(value="boolean")
    public static Boolean stContains(@SqlType(value="Geometry") Slice left, @SqlType(value="Geometry") Slice right) {
        if (!GeoFunctions.envelopes(left, right, Envelope::contains)) {
            return false;
        }
        OGCGeometry leftGeometry = GeometrySerde.deserialize((Slice)left);
        OGCGeometry rightGeometry = GeometrySerde.deserialize((Slice)right);
        GeoFunctions.verifySameSpatialReference(leftGeometry, rightGeometry);
        return leftGeometry.contains(rightGeometry);
    }

    @SqlNullable
    @Description(value="Returns TRUE if the supplied geometries have some, but not all, interior points in common")
    @ScalarFunction(value="ST_Crosses")
    @SqlType(value="boolean")
    public static Boolean stCrosses(@SqlType(value="Geometry") Slice left, @SqlType(value="Geometry") Slice right) {
        if (!GeoFunctions.envelopes(left, right, Envelope::intersect)) {
            return false;
        }
        OGCGeometry leftGeometry = GeometrySerde.deserialize((Slice)left);
        OGCGeometry rightGeometry = GeometrySerde.deserialize((Slice)right);
        GeoFunctions.verifySameSpatialReference(leftGeometry, rightGeometry);
        return leftGeometry.crosses(rightGeometry);
    }

    @SqlNullable
    @Description(value="Returns TRUE if the Geometries do not spatially intersect - if they do not share any space together")
    @ScalarFunction(value="ST_Disjoint")
    @SqlType(value="boolean")
    public static Boolean stDisjoint(@SqlType(value="Geometry") Slice left, @SqlType(value="Geometry") Slice right) {
        if (!GeoFunctions.envelopes(left, right, Envelope::intersect)) {
            return true;
        }
        OGCGeometry leftGeometry = GeometrySerde.deserialize((Slice)left);
        OGCGeometry rightGeometry = GeometrySerde.deserialize((Slice)right);
        GeoFunctions.verifySameSpatialReference(leftGeometry, rightGeometry);
        return leftGeometry.disjoint(rightGeometry);
    }

    @SqlNullable
    @Description(value="Returns TRUE if the given geometries represent the same geometry")
    @ScalarFunction(value="ST_Equals")
    @SqlType(value="boolean")
    public static Boolean stEquals(@SqlType(value="Geometry") Slice left, @SqlType(value="Geometry") Slice right) {
        OGCGeometry leftGeometry = GeometrySerde.deserialize((Slice)left);
        OGCGeometry rightGeometry = GeometrySerde.deserialize((Slice)right);
        GeoFunctions.verifySameSpatialReference(leftGeometry, rightGeometry);
        return leftGeometry.Equals(rightGeometry);
    }

    @SqlNullable
    @Description(value="Returns TRUE if the Geometries spatially intersect in 2D - (share any portion of space) and FALSE if they don't (they are Disjoint)")
    @ScalarFunction(value="ST_Intersects")
    @SqlType(value="boolean")
    public static Boolean stIntersects(@SqlType(value="Geometry") Slice left, @SqlType(value="Geometry") Slice right) {
        if (!GeoFunctions.envelopes(left, right, Envelope::intersect)) {
            return false;
        }
        OGCGeometry leftGeometry = GeometrySerde.deserialize((Slice)left);
        OGCGeometry rightGeometry = GeometrySerde.deserialize((Slice)right);
        GeoFunctions.verifySameSpatialReference(leftGeometry, rightGeometry);
        return leftGeometry.intersects(rightGeometry);
    }

    @SqlNullable
    @Description(value="Returns TRUE if the Geometries share space, are of the same dimension, but are not completely contained by each other")
    @ScalarFunction(value="ST_Overlaps")
    @SqlType(value="boolean")
    public static Boolean stOverlaps(@SqlType(value="Geometry") Slice left, @SqlType(value="Geometry") Slice right) {
        if (!GeoFunctions.envelopes(left, right, Envelope::intersect)) {
            return false;
        }
        OGCGeometry leftGeometry = GeometrySerde.deserialize((Slice)left);
        OGCGeometry rightGeometry = GeometrySerde.deserialize((Slice)right);
        GeoFunctions.verifySameSpatialReference(leftGeometry, rightGeometry);
        return leftGeometry.overlaps(rightGeometry);
    }

    @SqlNullable
    @Description(value="Returns TRUE if this Geometry is spatially related to another Geometry")
    @ScalarFunction(value="ST_Relate")
    @SqlType(value="boolean")
    public static Boolean stRelate(@SqlType(value="Geometry") Slice left, @SqlType(value="Geometry") Slice right, @SqlType(value="varchar") Slice relation) {
        OGCGeometry leftGeometry = GeometrySerde.deserialize((Slice)left);
        OGCGeometry rightGeometry = GeometrySerde.deserialize((Slice)right);
        GeoFunctions.verifySameSpatialReference(leftGeometry, rightGeometry);
        return leftGeometry.relate(rightGeometry, relation.toStringUtf8());
    }

    @SqlNullable
    @Description(value="Returns TRUE if the geometries have at least one point in common, but their interiors do not intersect")
    @ScalarFunction(value="ST_Touches")
    @SqlType(value="boolean")
    public static Boolean stTouches(@SqlType(value="Geometry") Slice left, @SqlType(value="Geometry") Slice right) {
        if (!GeoFunctions.envelopes(left, right, Envelope::intersect)) {
            return false;
        }
        OGCGeometry leftGeometry = GeometrySerde.deserialize((Slice)left);
        OGCGeometry rightGeometry = GeometrySerde.deserialize((Slice)right);
        GeoFunctions.verifySameSpatialReference(leftGeometry, rightGeometry);
        return leftGeometry.touches(rightGeometry);
    }

    @SqlNullable
    @Description(value="Returns TRUE if the geometry A is completely inside geometry B")
    @ScalarFunction(value="ST_Within")
    @SqlType(value="boolean")
    public static Boolean stWithin(@SqlType(value="Geometry") Slice left, @SqlType(value="Geometry") Slice right) {
        if (!GeoFunctions.envelopes(right, left, Envelope::contains)) {
            return false;
        }
        OGCGeometry leftGeometry = GeometrySerde.deserialize((Slice)left);
        OGCGeometry rightGeometry = GeometrySerde.deserialize((Slice)right);
        GeoFunctions.verifySameSpatialReference(leftGeometry, rightGeometry);
        return leftGeometry.within(rightGeometry);
    }

    @Description(value="Returns the type of the geometry")
    @ScalarFunction(value="ST_GeometryType")
    @SqlType(value="varchar")
    public static Slice stGeometryType(@SqlType(value="Geometry") Slice input) {
        return GeometrySerde.getGeometryType((Slice)input).standardName();
    }

    @ScalarFunction
    @SqlNullable
    @Description(value="Returns an array of spatial partition IDs for a given geometry")
    @SqlType(value="array(integer)")
    public static Block spatialPartitions(@SqlType(value="KdbTree") Object kdbTree, @SqlType(value="Geometry") Slice geometry) {
        Envelope envelope = GeometrySerde.deserializeEnvelope((Slice)geometry);
        if (envelope.isEmpty()) {
            return null;
        }
        return GeoFunctions.spatialPartitions((KdbTree)kdbTree, new Rectangle(envelope.getXMin(), envelope.getYMin(), envelope.getXMax(), envelope.getYMax()));
    }

    @ScalarFunction(value="from_geojson_geometry")
    @Description(value="Returns a spherical geography from a GeoJSON string")
    @SqlType(value="SphericalGeography")
    public static Slice fromGeoJsonGeometry(@SqlType(value="varchar") Slice input) {
        return JtsGeometrySerde.serialize((org.locationtech.jts.geom.Geometry)GeometryUtils.jtsGeometryFromJson((String)input.toStringUtf8()));
    }

    @SqlNullable
    @ScalarFunction(value="to_geojson_geometry")
    @Description(value="Returns GeoJSON string based on the input spherical geography")
    @SqlType(value="varchar")
    public static Slice toGeoJsonGeometry(@SqlType(value="SphericalGeography") Slice input) {
        return Slices.utf8Slice((String)GeometryUtils.jsonFromJtsGeometry((org.locationtech.jts.geom.Geometry)JtsGeometrySerde.deserialize((Slice)input)));
    }

    @ScalarFunction
    @SqlNullable
    @Description(value="Returns an array of spatial partition IDs for a geometry representing a set of points within specified distance from the input geometry")
    @SqlType(value="array(integer)")
    public static Block spatialPartitions(@SqlType(value="KdbTree") Object kdbTree, @SqlType(value="Geometry") Slice geometry, @SqlType(value="double") double distance) {
        if (Double.isNaN(distance)) {
            throw new TrinoException((ErrorCodeSupplier)StandardErrorCode.INVALID_FUNCTION_ARGUMENT, "distance is NaN");
        }
        if (Double.isInfinite(distance)) {
            throw new TrinoException((ErrorCodeSupplier)StandardErrorCode.INVALID_FUNCTION_ARGUMENT, "distance is infinite");
        }
        if (distance < 0.0) {
            throw new TrinoException((ErrorCodeSupplier)StandardErrorCode.INVALID_FUNCTION_ARGUMENT, "distance is negative");
        }
        Envelope envelope = GeometrySerde.deserializeEnvelope((Slice)geometry);
        if (envelope.isEmpty()) {
            return null;
        }
        Rectangle expandedEnvelope2D = new Rectangle(envelope.getXMin() - distance, envelope.getYMin() - distance, envelope.getXMax() + distance, envelope.getYMax() + distance);
        return GeoFunctions.spatialPartitions((KdbTree)kdbTree, expandedEnvelope2D);
    }

    private static Block spatialPartitions(KdbTree kdbTree, Rectangle envelope) {
        boolean point;
        Map partitions = kdbTree.findIntersectingLeaves(envelope);
        if (partitions.isEmpty()) {
            return EMPTY_ARRAY_OF_INTS;
        }
        boolean bl = point = envelope.getWidth() == 0.0 && envelope.getHeight() == 0.0;
        if (point) {
            for (Map.Entry partition : partitions.entrySet()) {
                if (!(envelope.getXMin() < ((Rectangle)partition.getValue()).getXMax()) || !(envelope.getYMin() < ((Rectangle)partition.getValue()).getYMax())) continue;
                BlockBuilder blockBuilder = IntegerType.INTEGER.createFixedSizeBlockBuilder(1);
                IntegerType.INTEGER.writeInt(blockBuilder, ((Integer)partition.getKey()).intValue());
                return blockBuilder.build();
            }
            throw new VerifyException(String.format("Cannot find half-open partition extent for a point: (%s, %s)", envelope.getXMin(), envelope.getYMin()));
        }
        BlockBuilder blockBuilder = IntegerType.INTEGER.createFixedSizeBlockBuilder(partitions.size());
        Iterator iterator = partitions.keySet().iterator();
        while (iterator.hasNext()) {
            int id = (Integer)iterator.next();
            IntegerType.INTEGER.writeInt(blockBuilder, id);
        }
        return blockBuilder.build();
    }

    @ScalarFunction
    @Description(value="Calculates the great-circle distance between two points on the Earth's surface in kilometers")
    @SqlType(value="double")
    public static double greatCircleDistance(@SqlType(value="double") double latitude1, @SqlType(value="double") double longitude1, @SqlType(value="double") double latitude2, @SqlType(value="double") double longitude2) {
        GeoFunctions.checkLatitude(latitude1);
        GeoFunctions.checkLongitude(longitude1);
        GeoFunctions.checkLatitude(latitude2);
        GeoFunctions.checkLongitude(longitude2);
        double radianLatitude1 = Math.toRadians(latitude1);
        double radianLatitude2 = Math.toRadians(latitude2);
        double sin1 = Math.sin(radianLatitude1);
        double cos1 = Math.cos(radianLatitude1);
        double sin2 = Math.sin(radianLatitude2);
        double cos2 = Math.cos(radianLatitude2);
        double deltaLongitude = Math.toRadians(longitude1) - Math.toRadians(longitude2);
        double cosDeltaLongitude = Math.cos(deltaLongitude);
        double t1 = cos2 * Math.sin(deltaLongitude);
        double t2 = cos1 * sin2 - sin1 * cos2 * cosDeltaLongitude;
        double t3 = sin1 * sin2 + cos1 * cos2 * cosDeltaLongitude;
        return Math.atan2(Math.sqrt(t1 * t1 + t2 * t2), t3) * 6371.01;
    }

    private static void checkLatitude(double latitude) {
        if (Double.isNaN(latitude) || Double.isInfinite(latitude) || latitude < -90.0 || latitude > 90.0) {
            throw new TrinoException((ErrorCodeSupplier)StandardErrorCode.INVALID_FUNCTION_ARGUMENT, "Latitude must be between -90 and 90");
        }
    }

    private static void checkLongitude(double longitude) {
        if (Double.isNaN(longitude) || Double.isInfinite(longitude) || longitude < -180.0 || longitude > 180.0) {
            throw new TrinoException((ErrorCodeSupplier)StandardErrorCode.INVALID_FUNCTION_ARGUMENT, "Longitude must be between -180 and 180");
        }
    }

    private static OGCGeometry geometryFromText(Slice input) {
        OGCGeometry geometry;
        try {
            geometry = OGCGeometry.fromText((String)input.toStringUtf8());
        }
        catch (IllegalArgumentException e) {
            throw new TrinoException((ErrorCodeSupplier)StandardErrorCode.INVALID_FUNCTION_ARGUMENT, "Invalid WKT: " + input.toStringUtf8(), (Throwable)e);
        }
        geometry.setSpatialReference(null);
        return geometry;
    }

    private static OGCGeometry geomFromBinary(Slice input) {
        OGCGeometry geometry;
        Objects.requireNonNull(input, "input is null");
        try {
            geometry = OGCGeometry.fromBinary((ByteBuffer)input.toByteBuffer().slice());
        }
        catch (IllegalArgumentException | IndexOutOfBoundsException e) {
            throw new TrinoException((ErrorCodeSupplier)StandardErrorCode.INVALID_FUNCTION_ARGUMENT, "Invalid WKB", (Throwable)e);
        }
        geometry.setSpatialReference(null);
        return geometry;
    }

    private static ByteBuffer getShapeByteBuffer(Slice input) {
        int offset = 5;
        if (input.length() <= offset) {
            throw new TrinoException((ErrorCodeSupplier)StandardErrorCode.INVALID_FUNCTION_ARGUMENT, "Hadoop shape input is too short");
        }
        return input.toByteBuffer(offset, input.length() - offset).slice().order(ByteOrder.LITTLE_ENDIAN);
    }

    private static int getWktExportFlags(Slice input) {
        byte hadoopShapeType = input.getByte(4);
        if (hadoopShapeType < 0 || hadoopShapeType >= HADOOP_SHAPE_TYPES.length) {
            throw new TrinoException((ErrorCodeSupplier)StandardErrorCode.INVALID_FUNCTION_ARGUMENT, "Invalid Hadoop shape type: " + hadoopShapeType);
        }
        return HADOOP_SHAPE_TYPES[hadoopShapeType];
    }

    private static void validateType(String function, OGCGeometry geometry, Set<io.trino.geospatial.GeometryType> validTypes) {
        io.trino.geospatial.GeometryType type = io.trino.geospatial.GeometryType.getForEsriGeometryType((String)geometry.geometryType());
        if (!validTypes.contains(type)) {
            throw new TrinoException((ErrorCodeSupplier)StandardErrorCode.INVALID_FUNCTION_ARGUMENT, String.format("%s only applies to %s. Input type is: %s", function, OR_JOINER.join(validTypes), type));
        }
    }

    private static void validateType(String function, org.locationtech.jts.geom.Geometry geometry, Set<io.trino.geospatial.GeometryType> validTypes) {
        io.trino.geospatial.GeometryType type = io.trino.geospatial.GeometryType.getForJtsGeometryType((String)geometry.getGeometryType());
        if (!validTypes.contains(type)) {
            throw new TrinoException((ErrorCodeSupplier)StandardErrorCode.INVALID_FUNCTION_ARGUMENT, String.format("%s only applies to %s. Input type is: %s", function, OR_JOINER.join(validTypes), type));
        }
    }

    private static void verifySameSpatialReference(OGCGeometry leftGeometry, OGCGeometry rightGeometry) {
        Preconditions.checkArgument((boolean)Objects.equals(leftGeometry.getEsriSpatialReference(), rightGeometry.getEsriSpatialReference()), (Object)"Input geometries must have the same spatial reference");
    }

    private static boolean envelopes(Slice left, Slice right, EnvelopesPredicate predicate) {
        Envelope leftEnvelope = GeometrySerde.deserializeEnvelope((Slice)left);
        Envelope rightEnvelope = GeometrySerde.deserializeEnvelope((Slice)right);
        if (leftEnvelope.isEmpty() || rightEnvelope.isEmpty()) {
            return false;
        }
        return predicate.apply(leftEnvelope, rightEnvelope);
    }

    @SqlNullable
    @Description(value="Returns the great-circle distance in meters between two SphericalGeography points.")
    @ScalarFunction(value="ST_Distance")
    @SqlType(value="double")
    public static Double stSphericalDistance(@SqlType(value="SphericalGeography") Slice left, @SqlType(value="SphericalGeography") Slice right) {
        OGCGeometry leftGeometry = GeometrySerde.deserialize((Slice)left);
        OGCGeometry rightGeometry = GeometrySerde.deserialize((Slice)right);
        if (leftGeometry.isEmpty() || rightGeometry.isEmpty()) {
            return null;
        }
        GeoFunctions.validateSphericalType("ST_Distance", leftGeometry, EnumSet.of(io.trino.geospatial.GeometryType.POINT));
        GeoFunctions.validateSphericalType("ST_Distance", rightGeometry, EnumSet.of(io.trino.geospatial.GeometryType.POINT));
        Point leftPoint = (Point)leftGeometry.getEsriGeometry();
        Point rightPoint = (Point)rightGeometry.getEsriGeometry();
        return GeoFunctions.greatCircleDistance(leftPoint.getY(), leftPoint.getX(), rightPoint.getY(), rightPoint.getX()) * 1000.0;
    }

    private static void validateSphericalType(String function, OGCGeometry geometry, Set<io.trino.geospatial.GeometryType> validTypes) {
        io.trino.geospatial.GeometryType type = io.trino.geospatial.GeometryType.getForEsriGeometryType((String)geometry.geometryType());
        if (!validTypes.contains(type)) {
            throw new TrinoException((ErrorCodeSupplier)StandardErrorCode.INVALID_FUNCTION_ARGUMENT, String.format("When applied to SphericalGeography inputs, %s only supports %s. Input type is: %s", function, OR_JOINER.join(validTypes), type));
        }
    }

    @SqlNullable
    @Description(value="Returns the area of a geometry on the Earth's surface using spherical model")
    @ScalarFunction(value="ST_Area")
    @SqlType(value="double")
    public static Double stSphericalArea(@SqlType(value="SphericalGeography") Slice input) {
        OGCGeometry geometry = GeometrySerde.deserialize((Slice)input);
        if (geometry.isEmpty()) {
            return null;
        }
        GeoFunctions.validateSphericalType("ST_Area", geometry, EnumSet.of(io.trino.geospatial.GeometryType.POLYGON, io.trino.geospatial.GeometryType.MULTI_POLYGON));
        Polygon polygon = (Polygon)geometry.getEsriGeometry();
        double sphericalExcess = 0.0;
        int numPaths = polygon.getPathCount();
        for (int i = 0; i < numPaths; ++i) {
            double sign = polygon.isExteriorRing(i) ? 1.0 : -1.0;
            sphericalExcess += sign * Math.abs(GeoFunctions.computeSphericalExcess(polygon, polygon.getPathStart(i), polygon.getPathEnd(i)));
        }
        return Math.abs(sphericalExcess * 6371010.0 * 6371010.0);
    }

    private static double computeSphericalExcess(Polygon polygon, int start, int end) {
        if (polygon.getPoint(end - 1).equals((Object)polygon.getPoint(start))) {
            --end;
        }
        if (end - start < 3) {
            throw new TrinoException((ErrorCodeSupplier)StandardErrorCode.INVALID_FUNCTION_ARGUMENT, "Polygon is not valid: a loop contains less then 3 vertices.");
        }
        Point point = new Point();
        polygon.getPoint(end - 1, point);
        double sphericalExcess = 0.0;
        double courseDelta = 0.0;
        boolean firstPoint = true;
        double firstInitialBearing = 0.0;
        double previousFinalBearing = 0.0;
        double previousPhi = Math.toRadians(point.getY());
        double previousCos = Math.cos(previousPhi);
        double previousSin = Math.sin(previousPhi);
        double previousTan = Math.tan(previousPhi / 2.0);
        double previousLongitude = Math.toRadians(point.getX());
        for (int i = start; i < end; ++i) {
            polygon.getPoint(i, point);
            double phi = Math.toRadians(point.getY());
            double tan = Math.tan(phi / 2.0);
            double longitude = Math.toRadians(point.getX());
            if (longitude == previousLongitude && phi == previousPhi) {
                throw new TrinoException((ErrorCodeSupplier)StandardErrorCode.INVALID_FUNCTION_ARGUMENT, "Polygon is not valid: it has two identical consecutive vertices");
            }
            double deltaLongitude = longitude - previousLongitude;
            sphericalExcess += 2.0 * Math.atan2(Math.tan(deltaLongitude / 2.0) * (previousTan + tan), 1.0 + previousTan * tan);
            double cos = Math.cos(phi);
            double sin = Math.sin(phi);
            double sinOfDeltaLongitude = Math.sin(deltaLongitude);
            double cosOfDeltaLongitude = Math.cos(deltaLongitude);
            double y = sinOfDeltaLongitude * cos;
            double x = previousCos * sin - previousSin * cos * cosOfDeltaLongitude;
            double initialBearing = (Math.atan2(y, x) + Math.PI * 2) % (Math.PI * 2);
            double finalY = -sinOfDeltaLongitude * previousCos;
            double finalX = previousSin * cos - previousCos * sin * cosOfDeltaLongitude;
            double finalBearing = (Math.atan2(finalY, finalX) + Math.PI) % (Math.PI * 2);
            if (firstPoint) {
                firstInitialBearing = initialBearing;
                firstPoint = false;
            } else {
                courseDelta += (initialBearing - previousFinalBearing + Math.PI * 3) % (Math.PI * 2) - Math.PI;
            }
            courseDelta += (finalBearing - initialBearing + Math.PI * 3) % (Math.PI * 2) - Math.PI;
            previousFinalBearing = finalBearing;
            previousCos = cos;
            previousSin = sin;
            previousPhi = phi;
            previousTan = tan;
            previousLongitude = longitude;
        }
        if (Math.abs(courseDelta += (firstInitialBearing - previousFinalBearing + Math.PI * 3) % (Math.PI * 2) - Math.PI) < 0.7853981633974483) {
            sphericalExcess = Math.abs(sphericalExcess) - Math.PI * 2;
        }
        return sphericalExcess;
    }

    private static Iterable<Slice> getGeometrySlicesFromBlock(final Block block) {
        Objects.requireNonNull(block, "block is null");
        return () -> new Iterator<Slice>(){
            private int iteratorPosition;

            @Override
            public boolean hasNext() {
                return this.iteratorPosition != block.getPositionCount();
            }

            @Override
            public Slice next() {
                if (!this.hasNext()) {
                    throw new NoSuchElementException("Slices have been consumed");
                }
                return GeometryType.GEOMETRY.getSlice(block, this.iteratorPosition++);
            }
        };
    }

    private static Iterable<OGCGeometry> flattenCollection(OGCGeometry geometry) {
        if (geometry == null) {
            return ImmutableList.of();
        }
        if (!(geometry instanceof OGCConcreteGeometryCollection)) {
            return ImmutableList.of((Object)geometry);
        }
        if (((OGCConcreteGeometryCollection)geometry).numGeometries() == 0) {
            return ImmutableList.of();
        }
        return () -> new GeometryCollectionIterator(geometry);
    }

    private static interface EnvelopesPredicate {
        public boolean apply(Envelope var1, Envelope var2);
    }

    private static class GeometryCollectionIterator
    implements Iterator<OGCGeometry> {
        private final Deque<OGCGeometry> geometriesDeque = new ArrayDeque<OGCGeometry>();

        GeometryCollectionIterator(OGCGeometry geometries) {
            this.geometriesDeque.push(Objects.requireNonNull(geometries, "geometries is null"));
        }

        @Override
        public boolean hasNext() {
            if (this.geometriesDeque.isEmpty()) {
                return false;
            }
            while (this.geometriesDeque.peek() instanceof OGCConcreteGeometryCollection) {
                OGCGeometryCollection collection = (OGCGeometryCollection)this.geometriesDeque.pop();
                for (int i = 0; i < collection.numGeometries(); ++i) {
                    this.geometriesDeque.push(collection.geometryN(i));
                }
            }
            return !this.geometriesDeque.isEmpty();
        }

        @Override
        public OGCGeometry next() {
            if (!this.hasNext()) {
                throw new NoSuchElementException("Geometries have been consumed");
            }
            return this.geometriesDeque.pop();
        }
    }
}

