/*
 * Copyright 2019 the original author or authors.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *        http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package io.restassured.module.jsv;

import com.github.fge.jsonschema.main.JsonSchemaFactory;

/**
 * Settings for the JsonSchemaValidator
 */
public class JsonSchemaValidatorSettings {
    private JsonSchemaFactory jsonSchemaFactory;
    private boolean checkedValidation;
    private boolean parseUriAndUrlsAsJsonNode;

    private JsonSchemaValidatorSettings(JsonSchemaFactory jsonSchemaFactory, boolean checkedValidation, boolean parseUriAndUrlsAsJsonNode) {
        if (jsonSchemaFactory == null) {
            throw new IllegalArgumentException(JsonSchemaFactory.class.getSimpleName() + " cannot be null");
        }
        this.jsonSchemaFactory = jsonSchemaFactory;
        this.checkedValidation = checkedValidation;
        this.parseUriAndUrlsAsJsonNode = parseUriAndUrlsAsJsonNode;
    }

    /**
     * Initializes JsonSchemaValidatorSettings with a specific {@link JsonSchemaFactory} and checked validation.
     * It will also use checked validation and treating uri's and url's as Strings. To change the latter refer to {@link #parseUriAndUrlsAsJsonNode(boolean)}.
     */
    public JsonSchemaValidatorSettings(JsonSchemaFactory jsonSchemaFactory, boolean checkedValidation) {
        this(jsonSchemaFactory, checkedValidation, false);
    }

    /**
     * Initializes JsonSchemaValidatorSettings with a specific {@link JsonSchemaFactory}. The settings will also use checked validation and treating uri's and url's as Strings.
     */
    public JsonSchemaValidatorSettings(JsonSchemaFactory jsonSchemaFactory) {
        this(jsonSchemaFactory, true);
    }

    /**
     * Initializes JsonSchemaValidatorSettings with a default {@link JsonSchemaFactory} (generated by <code>JsonSchemaFactory.byDefault()</code>)
     * and using checked validation and treating uri's and url's as Strings.
     */
    public JsonSchemaValidatorSettings() {
        this(JsonSchemaFactory.byDefault(), true);
    }

    public JsonSchemaFactory jsonSchemaFactory() {
        return jsonSchemaFactory;
    }

    public boolean shouldParseUriAndUrlsAsJsonNode() {
        return parseUriAndUrlsAsJsonNode;
    }

    public boolean shouldUseCheckedValidation() {
        return checkedValidation;
    }

    /**
     * Instruct the JsonSchemaValidator to use checked validation or not.
     *
     * @param shouldUseCheckedValidation <code>true</code> to use checked validation, <code>false</code> otherwise.
     * @return A new instance of JsonSchemaValidatorSettings
     */
    public JsonSchemaValidatorSettings checkedValidation(boolean shouldUseCheckedValidation) {
        return new JsonSchemaValidatorSettings(jsonSchemaFactory, shouldUseCheckedValidation, parseUriAndUrlsAsJsonNode);
    }

    /**
     * Instruct the JsonSchemaValidator to parse URI's and URL's as JsonNode before they are passed to validation.
     * If <code>true</code> then <code>JsonSchemaValidator.matchesJsonSchema(url)</code> handled like this:
     * <pre>
     * instanceSettings.jsonSchemaFactory().getJsonSchema(JsonLoader.fromURL(url))
     * </pre>
     * If <code>false</code> then <code>JsonSchemaValidator.matchesJsonSchema(url)</code> handled like this:
     * <pre>
     * instanceSettings.jsonSchemaFactory().getJsonSchema(url)
     * </pre>
     * The latter is good for you need to resolve relative ref in a parent schema because the validator does not know what is the base URI
     * of the parent schema if first having parsed it as a JsonNode.
     * <p>
     * Default is <code>false</code>.
     * </p>
     *
     * @param parseUriAndUrlsAsJsonNode <code>true</code> to parse URI's and URL's as {@link com.fasterxml.jackson.databind.JsonNode}'s, <code>false</code> otherwise.
     * @return A new instance of JsonSchemaValidatorSettings
     */
    public JsonSchemaValidatorSettings parseUriAndUrlsAsJsonNode(boolean parseUriAndUrlsAsJsonNode) {
        return new JsonSchemaValidatorSettings(jsonSchemaFactory, checkedValidation, parseUriAndUrlsAsJsonNode);
    }

    /**
     * Set a {@link JsonSchemaFactory} instance that will be used when creating the validator.
     *
     * @param jsonSchemaFactory The {@link JsonSchemaFactory} instance to use.
     * @return A new instance of JsonSchemaValidatorSettings
     */
    public JsonSchemaValidatorSettings jsonSchemaFactory(JsonSchemaFactory jsonSchemaFactory) {
        return new JsonSchemaValidatorSettings(jsonSchemaFactory, checkedValidation, parseUriAndUrlsAsJsonNode);
    }

    /**
     * Syntactic sugar.
     *
     * @return The same settings instance.
     */
    public JsonSchemaValidatorSettings and() {
        return this;
    }

    /**
     * Syntactic sugar.
     *
     * @return The same settings instance.
     */
    public JsonSchemaValidatorSettings with() {
        return this;
    }

    /**
     * Create a new instance of {@link JsonSchemaValidatorSettings}. Same as calling {@link #JsonSchemaValidatorSettings()}.
     *
     * @return A default instance of the {@link JsonSchemaValidatorSettings}.
     */
    public static JsonSchemaValidatorSettings settings() {
        return new JsonSchemaValidatorSettings();
    }
}
