/*
 * Decompiled with CFR 0.152.
 */
package io.realm;

import io.realm.BaseRealm;
import io.realm.DynamicRealm;
import io.realm.Realm;
import io.realm.RealmConfiguration;
import io.realm.RealmMigration;
import io.realm.exceptions.RealmFileException;
import io.realm.internal.ColumnIndices;
import io.realm.internal.ObjectServerFacade;
import io.realm.internal.SharedRealm;
import io.realm.internal.Table;
import io.realm.log.RealmLog;
import java.io.File;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.util.Arrays;
import java.util.EnumMap;
import java.util.HashMap;
import java.util.Map;

final class RealmCache {
    private final EnumMap<RealmCacheType, RefAndCount> refAndCountMap;
    private final RealmConfiguration configuration;
    private static final int MAX_ENTRIES_IN_TYPED_COLUMN_INDICES_ARRAY = 4;
    private final ColumnIndices[] typedColumnIndicesArray = new ColumnIndices[4];
    private static Map<String, RealmCache> cachesMap = new HashMap<String, RealmCache>();
    private static final String DIFFERENT_KEY_MESSAGE = "Wrong key used to decrypt Realm.";
    private static final String WRONG_REALM_CLASS_MESSAGE = "The type of Realm class must be Realm or DynamicRealm.";

    private RealmCache(RealmConfiguration config) {
        this.configuration = config;
        this.refAndCountMap = new EnumMap(RealmCacheType.class);
        for (RealmCacheType type : RealmCacheType.values()) {
            this.refAndCountMap.put(type, new RefAndCount());
        }
    }

    static synchronized <E extends BaseRealm> E createRealmOrGetFromCache(RealmConfiguration configuration, Class<E> realmClass) {
        BaseRealm realm;
        Integer refCount;
        boolean isCacheInMap = true;
        RealmCache cache = cachesMap.get(configuration.getPath());
        if (cache == null) {
            cache = new RealmCache(configuration);
            isCacheInMap = false;
            RealmCache.copyAssetFileIfNeeded(configuration);
        } else {
            cache.validateConfiguration(configuration);
        }
        RefAndCount refAndCount = cache.refAndCountMap.get((Object)RealmCacheType.valueOf(realmClass));
        if (refAndCount.globalCount == 0) {
            SharedRealm sharedRealm = SharedRealm.getInstance(configuration);
            if (Table.primaryKeyTableNeedsMigration(sharedRealm)) {
                sharedRealm.beginTransaction();
                if (Table.migratePrimaryKeyTableIfNeeded(sharedRealm)) {
                    sharedRealm.commitTransaction();
                } else {
                    sharedRealm.cancelTransaction();
                }
            }
            sharedRealm.close();
        }
        if (refAndCount.localRealm.get() == null) {
            BaseRealm realm2;
            if (realmClass == Realm.class) {
                realm2 = Realm.createInstance(configuration, cache.typedColumnIndicesArray);
            } else if (realmClass == DynamicRealm.class) {
                realm2 = DynamicRealm.createInstance(configuration);
            } else {
                throw new IllegalArgumentException(WRONG_REALM_CLASS_MESSAGE);
            }
            if (!isCacheInMap) {
                cachesMap.put(configuration.getPath(), cache);
            }
            refAndCount.localRealm.set(realm2);
            refAndCount.localCount.set(0);
        }
        if ((refCount = (Integer)refAndCount.localCount.get()) == 0) {
            if (realmClass == Realm.class && refAndCount.globalCount == 0) {
                realm = (BaseRealm)refAndCount.localRealm.get();
                RealmCache.storeColumnIndices(cache.typedColumnIndicesArray, realm.schema.columnIndices.clone());
            }
            refAndCount.globalCount++;
        }
        refAndCount.localCount.set(refCount + 1);
        realm = (BaseRealm)refAndCount.localRealm.get();
        if (refAndCount.globalCount == 1) {
            ObjectServerFacade.getFacade(configuration.isSyncConfiguration()).realmOpened(configuration);
        }
        return (E)realm;
    }

    static synchronized void release(BaseRealm realm) {
        String canonicalPath = realm.getPath();
        RealmCache cache = cachesMap.get(canonicalPath);
        Integer refCount = null;
        RefAndCount refAndCount = null;
        if (cache != null) {
            refAndCount = cache.refAndCountMap.get((Object)RealmCacheType.valueOf(realm.getClass()));
            refCount = (Integer)refAndCount.localCount.get();
        }
        if (refCount == null) {
            refCount = 0;
        }
        if (refCount <= 0) {
            RealmLog.warn("%s has been closed already.", canonicalPath);
            return;
        }
        if ((refCount = Integer.valueOf(refCount - 1)) == 0) {
            refAndCount.localCount.set(null);
            refAndCount.localRealm.set(null);
            refAndCount.globalCount--;
            if (refAndCount.globalCount < 0) {
                throw new IllegalStateException("Global reference counter of Realm" + canonicalPath + " got corrupted.");
            }
            if (realm instanceof Realm && refAndCount.globalCount == 0) {
                Arrays.fill(cache.typedColumnIndicesArray, null);
            }
            int totalRefCount = 0;
            for (RealmCacheType type : RealmCacheType.values()) {
                totalRefCount += cache.refAndCountMap.get((Object)type).globalCount;
            }
            realm.doClose();
            if (totalRefCount == 0) {
                cachesMap.remove(canonicalPath);
                ObjectServerFacade.getFacade(realm.getConfiguration().isSyncConfiguration()).realmClosed(realm.getConfiguration());
            }
        } else {
            refAndCount.localCount.set(refCount);
        }
    }

    private void validateConfiguration(RealmConfiguration newConfiguration) {
        if (this.configuration.equals(newConfiguration)) {
            return;
        }
        if (!Arrays.equals(this.configuration.getEncryptionKey(), newConfiguration.getEncryptionKey())) {
            throw new IllegalArgumentException(DIFFERENT_KEY_MESSAGE);
        }
        RealmMigration newMigration = newConfiguration.getMigration();
        RealmMigration oldMigration = this.configuration.getMigration();
        if (oldMigration != null && newMigration != null && oldMigration.getClass().equals(newMigration.getClass()) && !newMigration.equals(oldMigration)) {
            throw new IllegalArgumentException("Configurations cannot be different if used to open the same file. The most likely cause is that equals() and hashCode() are not overridden in the migration class: " + newConfiguration.getMigration().getClass().getCanonicalName());
        }
        throw new IllegalArgumentException("Configurations cannot be different if used to open the same file. \nCached configuration: \n" + this.configuration + "\n\nNew configuration: \n" + newConfiguration);
    }

    static synchronized void invokeWithGlobalRefCount(RealmConfiguration configuration, Callback callback) {
        RealmCache cache = cachesMap.get(configuration.getPath());
        if (cache == null) {
            callback.onResult(0);
            return;
        }
        int totalRefCount = 0;
        for (RealmCacheType type : RealmCacheType.values()) {
            totalRefCount += cache.refAndCountMap.get((Object)type).globalCount;
        }
        callback.onResult(totalRefCount);
    }

    static synchronized void updateSchemaCache(Realm realm) {
        RealmCache cache = cachesMap.get(realm.getPath());
        if (cache == null) {
            return;
        }
        RefAndCount refAndCount = cache.refAndCountMap.get((Object)RealmCacheType.TYPED_REALM);
        if (refAndCount.localRealm.get() == null) {
            return;
        }
        ColumnIndices[] globalCacheArray = cache.typedColumnIndicesArray;
        ColumnIndices createdCacheEntry = realm.updateSchemaCache(globalCacheArray);
        if (createdCacheEntry != null) {
            RealmCache.storeColumnIndices(globalCacheArray, createdCacheEntry);
        }
    }

    static synchronized void invokeWithLock(Callback0 callback) {
        callback.onCall();
    }

    private static void copyAssetFileIfNeeded(RealmConfiguration configuration) {
        IOException exceptionWhenClose = null;
        if (configuration.hasAssetFile()) {
            File realmFile = new File(configuration.getRealmDirectory(), configuration.getRealmFileName());
            if (realmFile.exists()) {
                return;
            }
            InputStream inputStream = null;
            FileOutputStream outputStream = null;
            try {
                int bytesRead;
                inputStream = configuration.getAssetFile();
                if (inputStream == null) {
                    throw new RealmFileException(RealmFileException.Kind.ACCESS_ERROR, "Invalid input stream to asset file.");
                }
                outputStream = new FileOutputStream(realmFile);
                byte[] buf = new byte[4096];
                while ((bytesRead = inputStream.read(buf)) > -1) {
                    outputStream.write(buf, 0, bytesRead);
                }
            }
            catch (IOException e) {
                throw new RealmFileException(RealmFileException.Kind.ACCESS_ERROR, "Could not resolve the path to the Realm asset file.", e);
            }
            finally {
                block21: {
                    if (inputStream != null) {
                        try {
                            inputStream.close();
                        }
                        catch (IOException e) {
                            exceptionWhenClose = e;
                        }
                    }
                    if (outputStream != null) {
                        try {
                            outputStream.close();
                        }
                        catch (IOException e) {
                            if (exceptionWhenClose != null) break block21;
                            exceptionWhenClose = e;
                        }
                    }
                }
            }
            if (exceptionWhenClose != null) {
                throw new RealmFileException(RealmFileException.Kind.ACCESS_ERROR, (Throwable)exceptionWhenClose);
            }
        }
    }

    static int getLocalThreadCount(RealmConfiguration configuration) {
        RealmCache cache = cachesMap.get(configuration.getPath());
        if (cache == null) {
            return 0;
        }
        int totalRefCount = 0;
        for (RealmCacheType type : RealmCacheType.values()) {
            totalRefCount += ((Integer)cache.refAndCountMap.get((Object)type).localCount.get()).intValue();
        }
        return totalRefCount;
    }

    public static ColumnIndices findColumnIndices(ColumnIndices[] array, long schemaVersion) {
        for (int i = array.length - 1; 0 <= i; --i) {
            ColumnIndices candidate = array[i];
            if (candidate == null || candidate.getSchemaVersion() != schemaVersion) continue;
            return candidate;
        }
        return null;
    }

    private static int storeColumnIndices(ColumnIndices[] array, ColumnIndices columnIndices) {
        long oldestSchemaVersion = Long.MAX_VALUE;
        int candidateIndex = -1;
        for (int i = array.length - 1; 0 <= i; --i) {
            if (array[i] == null) {
                array[i] = columnIndices;
                return i;
            }
            ColumnIndices target = array[i];
            if (target.getSchemaVersion() > oldestSchemaVersion) continue;
            oldestSchemaVersion = target.getSchemaVersion();
            candidateIndex = i;
        }
        array[candidateIndex] = columnIndices;
        return candidateIndex;
    }

    private static enum RealmCacheType {
        TYPED_REALM,
        DYNAMIC_REALM;


        static RealmCacheType valueOf(Class<? extends BaseRealm> clazz) {
            if (clazz == Realm.class) {
                return TYPED_REALM;
            }
            if (clazz == DynamicRealm.class) {
                return DYNAMIC_REALM;
            }
            throw new IllegalArgumentException(RealmCache.WRONG_REALM_CLASS_MESSAGE);
        }
    }

    private static class RefAndCount {
        private final ThreadLocal<BaseRealm> localRealm = new ThreadLocal();
        private final ThreadLocal<Integer> localCount = new ThreadLocal();
        private int globalCount = 0;

        private RefAndCount() {
        }
    }

    static interface Callback0 {
        public void onCall();
    }

    static interface Callback {
        public void onResult(int var1);
    }
}

