/*
 * Copyright 2017-2020 original authors
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * https://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package io.micronaut.http.client.loadbalance;

import io.micronaut.core.convert.MutableConversionService;
import io.micronaut.core.convert.TypeConverterRegistrar;
import io.micronaut.http.client.LoadBalancer;

import java.net.URI;
import java.net.URISyntaxException;
import java.net.URL;

/**
 * Converters from URL to {@link LoadBalancer} interface.
 *
 * @author graemerocher
 * @since 1.0
 */
public class LoadBalancerConverters implements TypeConverterRegistrar {

    @Override
    public void register(MutableConversionService conversionService) {
        conversionService.addConverter(URI.class, LoadBalancer.class, LoadBalancer::fixed);
        conversionService.addConverter(URL.class, LoadBalancer.class, LoadBalancer::fixed);
        conversionService.addConverter(String.class, LoadBalancer.class, url -> {
            try {
                return LoadBalancer.fixed(new URI(url));
            } catch (URISyntaxException e) {
                return null;
            }
        });
    }
}
