/*
 * Decompiled with CFR 0.152.
 */
package io.micronaut.transaction.jdbc;

import io.micronaut.core.annotation.NonNull;
import io.micronaut.core.annotation.Nullable;
import io.micronaut.transaction.TransactionDefinition;
import io.micronaut.transaction.jdbc.ConnectionHolder;
import io.micronaut.transaction.jdbc.DelegatingDataSource;
import io.micronaut.transaction.jdbc.exceptions.CannotGetJdbcConnectionException;
import io.micronaut.transaction.support.TransactionSynchronization;
import io.micronaut.transaction.support.TransactionSynchronizationAdapter;
import io.micronaut.transaction.support.TransactionSynchronizationManager;
import java.sql.Connection;
import java.sql.SQLException;
import java.sql.Statement;
import java.util.Objects;
import javax.sql.DataSource;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public abstract class DataSourceUtils {
    public static final int CONNECTION_SYNCHRONIZATION_ORDER = 1000;
    private static final Logger LOGGER = LoggerFactory.getLogger(DataSourceUtils.class);

    public static Connection getConnection(DataSource dataSource) throws CannotGetJdbcConnectionException {
        try {
            return DataSourceUtils.doGetConnection(dataSource, true);
        }
        catch (SQLException ex) {
            throw new CannotGetJdbcConnectionException("Failed to obtain JDBC Connection", ex);
        }
        catch (IllegalStateException ex) {
            throw new CannotGetJdbcConnectionException("Failed to obtain JDBC Connection: " + ex.getMessage());
        }
    }

    public static Connection getConnection(DataSource dataSource, boolean allowCreate) throws CannotGetJdbcConnectionException {
        try {
            return DataSourceUtils.doGetConnection(dataSource, allowCreate);
        }
        catch (SQLException ex) {
            throw new CannotGetJdbcConnectionException("Failed to obtain JDBC Connection", ex);
        }
        catch (IllegalStateException ex) {
            throw new CannotGetJdbcConnectionException("Failed to obtain JDBC Connection: " + ex.getMessage());
        }
    }

    private static Connection doGetConnection(DataSource dataSource, boolean allowCreate) throws SQLException {
        Objects.requireNonNull(dataSource, "No DataSource specified");
        ConnectionHolder conHolder = (ConnectionHolder)TransactionSynchronizationManager.getResource(dataSource);
        if (conHolder != null && (conHolder.hasConnection() || conHolder.isSynchronizedWithTransaction())) {
            conHolder.requested();
            if (!conHolder.hasConnection()) {
                if (LOGGER.isDebugEnabled()) {
                    LOGGER.debug("Fetching resumed JDBC Connection from DataSource");
                }
                if (!allowCreate) {
                    throw new CannotGetJdbcConnectionException("No current JDBC Connection found. Consider wrapping this call in transactional boundaries.");
                }
                conHolder.setConnection(DataSourceUtils.fetchConnection(dataSource));
            }
            return conHolder.getConnection();
        }
        if (LOGGER.isDebugEnabled()) {
            LOGGER.debug("Fetching JDBC Connection from DataSource");
        }
        if (!allowCreate) {
            throw new CannotGetJdbcConnectionException("No current JDBC Connection found. Consider wrapping this call in transactional boundaries.");
        }
        Connection con = DataSourceUtils.fetchConnection(dataSource);
        if (TransactionSynchronizationManager.isSynchronizationActive()) {
            try {
                ConnectionHolder holderToUse = conHolder;
                if (holderToUse == null) {
                    holderToUse = new ConnectionHolder(con);
                } else {
                    holderToUse.setConnection(con);
                }
                holderToUse.requested();
                TransactionSynchronizationManager.registerSynchronization(new ConnectionSynchronization(holderToUse, dataSource));
                holderToUse.setSynchronizedWithTransaction(true);
                if (holderToUse != conHolder) {
                    TransactionSynchronizationManager.bindResource(dataSource, holderToUse);
                }
            }
            catch (RuntimeException ex) {
                DataSourceUtils.releaseConnection(con, dataSource);
                throw ex;
            }
        }
        return con;
    }

    private static Connection fetchConnection(DataSource dataSource) throws SQLException {
        Connection con = dataSource.getConnection();
        if (con == null) {
            throw new IllegalStateException("DataSource returned null from getConnection(): " + dataSource);
        }
        return con;
    }

    @Nullable
    public static TransactionDefinition.Isolation prepareConnectionForTransaction(Connection con, @Nullable TransactionDefinition definition) throws SQLException {
        TransactionDefinition.Isolation isolationLevel;
        Objects.requireNonNull(con, "No Connection specified");
        if (definition != null && definition.isReadOnly()) {
            try {
                if (LOGGER.isDebugEnabled()) {
                    LOGGER.debug("Setting JDBC Connection [" + con + "] read-only");
                }
                con.setReadOnly(true);
            }
            catch (RuntimeException | SQLException ex) {
                for (Throwable exToCheck = ex; exToCheck != null; exToCheck = exToCheck.getCause()) {
                    if (!exToCheck.getClass().getSimpleName().contains("Timeout")) continue;
                    throw ex;
                }
                LOGGER.debug("Could not set JDBC Connection read-only", (Throwable)ex);
            }
        }
        TransactionDefinition.Isolation previousIsolationLevel = null;
        if (definition != null && (isolationLevel = definition.getIsolationLevel()) != TransactionDefinition.Isolation.DEFAULT) {
            int currentIsolation;
            if (LOGGER.isDebugEnabled()) {
                LOGGER.debug("Changing isolation level of JDBC Connection [" + con + "] to " + isolationLevel);
            }
            if ((currentIsolation = con.getTransactionIsolation()) != isolationLevel.getCode()) {
                previousIsolationLevel = TransactionDefinition.Isolation.valueOf(currentIsolation);
                con.setTransactionIsolation(isolationLevel.getCode());
            }
        }
        return previousIsolationLevel;
    }

    public static void resetConnectionAfterTransaction(Connection con, @Nullable TransactionDefinition.Isolation previousIsolationLevel) {
        Objects.requireNonNull(con, "No Connection specified");
        try {
            if (previousIsolationLevel != null) {
                if (LOGGER.isDebugEnabled()) {
                    LOGGER.debug("Resetting isolation level of JDBC Connection [" + con + "] to " + previousIsolationLevel);
                }
                con.setTransactionIsolation(previousIsolationLevel.getCode());
            }
            if (con.isReadOnly()) {
                if (LOGGER.isDebugEnabled()) {
                    LOGGER.debug("Resetting read-only flag of JDBC Connection [" + con + "]");
                }
                con.setReadOnly(false);
            }
        }
        catch (Throwable ex) {
            LOGGER.debug("Could not reset JDBC Connection after transaction", ex);
        }
    }

    public static boolean isConnectionTransactional(Connection con, @Nullable DataSource dataSource) {
        if (dataSource == null) {
            return false;
        }
        ConnectionHolder conHolder = (ConnectionHolder)TransactionSynchronizationManager.getResource(dataSource);
        return conHolder != null && DataSourceUtils.connectionEquals(conHolder, con);
    }

    public static void applyTransactionTimeout(Statement stmt, @Nullable DataSource dataSource) throws SQLException {
        DataSourceUtils.applyTimeout(stmt, dataSource, -1);
    }

    public static void applyTimeout(Statement stmt, @Nullable DataSource dataSource, int timeout) throws SQLException {
        Objects.requireNonNull(stmt, "No Statement specified");
        ConnectionHolder holder = null;
        if (dataSource != null) {
            holder = (ConnectionHolder)TransactionSynchronizationManager.getResource(dataSource);
        }
        if (holder != null && holder.hasTimeout()) {
            stmt.setQueryTimeout(holder.getTimeToLiveInSeconds());
        } else if (timeout >= 0) {
            stmt.setQueryTimeout(timeout);
        }
    }

    public static void releaseConnection(@Nullable Connection con, @Nullable DataSource dataSource) {
        try {
            DataSourceUtils.doReleaseConnection(con, dataSource);
        }
        catch (SQLException ex) {
            LOGGER.debug("Could not close JDBC Connection", (Throwable)ex);
        }
        catch (Throwable ex) {
            LOGGER.debug("Unexpected exception on closing JDBC Connection", ex);
        }
    }

    public static void doReleaseConnection(@Nullable Connection con, @Nullable DataSource dataSource) throws SQLException {
        ConnectionHolder conHolder;
        if (con == null) {
            return;
        }
        if (dataSource != null && (conHolder = (ConnectionHolder)TransactionSynchronizationManager.getResource(dataSource)) != null && DataSourceUtils.connectionEquals(conHolder, con)) {
            conHolder.released();
            return;
        }
        DataSourceUtils.doCloseConnection(con, dataSource);
    }

    public static void doCloseConnection(Connection con, @Nullable DataSource dataSource) throws SQLException {
        con.close();
    }

    private static boolean connectionEquals(ConnectionHolder conHolder, Connection passedInCon) {
        if (!conHolder.hasConnection()) {
            return false;
        }
        Connection heldCon = conHolder.getConnection();
        return heldCon == passedInCon || heldCon.equals(passedInCon) || DataSourceUtils.getTargetConnection(heldCon).equals(passedInCon);
    }

    public static Connection getTargetConnection(Connection con) {
        return con;
    }

    private static int getConnectionSynchronizationOrder(DataSource dataSource) {
        int order = 1000;
        DataSource currDs = dataSource;
        while (currDs instanceof DelegatingDataSource) {
            --order;
            currDs = ((DelegatingDataSource)currDs).getTargetDataSource();
        }
        return order;
    }

    private static class ConnectionSynchronization
    extends TransactionSynchronizationAdapter {
        private final ConnectionHolder connectionHolder;
        private final DataSource dataSource;
        private int order;
        private boolean holderActive = true;

        public ConnectionSynchronization(ConnectionHolder connectionHolder, DataSource dataSource) {
            this.connectionHolder = connectionHolder;
            this.dataSource = dataSource;
            this.order = DataSourceUtils.getConnectionSynchronizationOrder(dataSource);
        }

        @Override
        public int getOrder() {
            return this.order;
        }

        @Override
        public void suspend() {
            if (this.holderActive) {
                TransactionSynchronizationManager.unbindResource(this.dataSource);
                if (this.connectionHolder.hasConnection() && !this.connectionHolder.isOpen()) {
                    DataSourceUtils.releaseConnection(this.connectionHolder.getConnection(), this.dataSource);
                    this.connectionHolder.setConnection(null);
                }
            }
        }

        @Override
        public void resume() {
            if (this.holderActive) {
                TransactionSynchronizationManager.bindResource(this.dataSource, this.connectionHolder);
            }
        }

        @Override
        public void beforeCompletion() {
            if (!this.connectionHolder.isOpen()) {
                TransactionSynchronizationManager.unbindResource(this.dataSource);
                this.holderActive = false;
                if (this.connectionHolder.hasConnection()) {
                    DataSourceUtils.releaseConnection(this.connectionHolder.getConnection(), this.dataSource);
                }
            }
        }

        @Override
        public void afterCompletion(@NonNull TransactionSynchronization.Status status) {
            if (this.holderActive) {
                TransactionSynchronizationManager.unbindResourceIfPossible(this.dataSource);
                this.holderActive = false;
                if (this.connectionHolder.hasConnection()) {
                    DataSourceUtils.releaseConnection(this.connectionHolder.getConnection(), this.dataSource);
                    this.connectionHolder.setConnection(null);
                }
            }
            this.connectionHolder.reset();
        }
    }
}

