/**
 * Copyright 2017 VMware, Inc.
 * <p>
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 * <p>
 * https://www.apache.org/licenses/LICENSE-2.0
 * <p>
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package io.micrometer.spring.autoconfigure;

import org.springframework.boot.autoconfigure.AutoConfigureAfter;
import org.springframework.boot.autoconfigure.EnableAutoConfiguration;
import org.springframework.boot.autoconfigure.condition.ConditionMessage;
import org.springframework.boot.autoconfigure.condition.ConditionOutcome;
import org.springframework.boot.autoconfigure.condition.ConditionalOnBean;
import org.springframework.boot.autoconfigure.condition.ConditionalOnClass;
import org.springframework.boot.autoconfigure.condition.ConditionalOnMissingBean;
import org.springframework.boot.autoconfigure.condition.ConditionalOnProperty;
import org.springframework.boot.autoconfigure.condition.SpringBootCondition;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.ConditionContext;
import org.springframework.context.annotation.Conditional;
import org.springframework.context.annotation.Configuration;
import org.springframework.core.type.AnnotatedTypeMetadata;

import ch.qos.logback.classic.LoggerContext;
import io.micrometer.core.instrument.MeterRegistry;
import io.micrometer.core.instrument.binder.logging.LogbackMetrics;
import io.micrometer.spring.autoconfigure.LogbackMetricsAutoConfiguration.LogbackLoggingCondition;
import org.slf4j.ILoggerFactory;
import org.slf4j.LoggerFactory;

/**
 * {@link EnableAutoConfiguration Auto-configuration} for Logback metrics.
 *
 * @author Jon Schneider
 * @author Stephane Nicoll
 * @since 1.1.0
 */
@Configuration
@AutoConfigureAfter(MetricsAutoConfiguration.class)
@ConditionalOnClass({ MeterRegistry.class, LoggerContext.class, LoggerFactory.class })
@ConditionalOnBean(MeterRegistry.class)
@Conditional(LogbackLoggingCondition.class)
@ConditionalOnProperty(value = "management.metrics.binders.logback.enabled", matchIfMissing = true)
public class LogbackMetricsAutoConfiguration {

    @Bean
    @ConditionalOnMissingBean
    public LogbackMetrics logbackMetrics() {
        return new LogbackMetrics();
    }

    static class LogbackLoggingCondition extends SpringBootCondition {

        @Override
        public ConditionOutcome getMatchOutcome(ConditionContext context,
                AnnotatedTypeMetadata metadata) {
            ILoggerFactory loggerFactory = LoggerFactory.getILoggerFactory();
            ConditionMessage.Builder message = ConditionMessage
                    .forCondition("LogbackLoggingCondition");
            if (loggerFactory instanceof LoggerContext) {
                return ConditionOutcome.match(
                        message.because("ILoggerFactory is a Logback LoggerContext"));
            }
            return ConditionOutcome
                    .noMatch(message.because("ILoggerFactory is an instance of "
                            + loggerFactory.getClass().getCanonicalName()));
        }

    }

}
