/*
 * Decompiled with CFR 0.152.
 */
package io.micrometer.cloudwatch;

import com.amazonaws.AbortedException;
import com.amazonaws.handlers.AsyncHandler;
import com.amazonaws.services.cloudwatch.AmazonCloudWatchAsync;
import com.amazonaws.services.cloudwatch.model.Dimension;
import com.amazonaws.services.cloudwatch.model.MetricDatum;
import com.amazonaws.services.cloudwatch.model.PutMetricDataRequest;
import com.amazonaws.services.cloudwatch.model.PutMetricDataResult;
import com.amazonaws.services.cloudwatch.model.StandardUnit;
import io.micrometer.cloudwatch.CloudWatchConfig;
import io.micrometer.cloudwatch.CloudWatchUtils;
import io.micrometer.cloudwatch.MetricDatumPartition;
import io.micrometer.core.instrument.Clock;
import io.micrometer.core.instrument.Counter;
import io.micrometer.core.instrument.DistributionSummary;
import io.micrometer.core.instrument.FunctionCounter;
import io.micrometer.core.instrument.FunctionTimer;
import io.micrometer.core.instrument.Gauge;
import io.micrometer.core.instrument.LongTaskTimer;
import io.micrometer.core.instrument.Meter;
import io.micrometer.core.instrument.Tag;
import io.micrometer.core.instrument.TimeGauge;
import io.micrometer.core.instrument.Timer;
import io.micrometer.core.instrument.config.NamingConvention;
import io.micrometer.core.instrument.step.StepMeterRegistry;
import io.micrometer.core.instrument.step.StepRegistryConfig;
import io.micrometer.core.instrument.util.NamedThreadFactory;
import io.micrometer.core.instrument.util.TimeUtils;
import io.micrometer.core.lang.Nullable;
import java.time.Duration;
import java.util.Date;
import java.util.List;
import java.util.Objects;
import java.util.concurrent.CountDownLatch;
import java.util.concurrent.ThreadFactory;
import java.util.concurrent.TimeUnit;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import java.util.stream.StreamSupport;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class CloudWatchMeterRegistry
extends StepMeterRegistry {
    private final CloudWatchConfig config;
    private final AmazonCloudWatchAsync amazonCloudWatchAsync;
    private final Logger logger = LoggerFactory.getLogger(CloudWatchMeterRegistry.class);

    public CloudWatchMeterRegistry(CloudWatchConfig config, Clock clock, AmazonCloudWatchAsync amazonCloudWatchAsync) {
        this(config, clock, amazonCloudWatchAsync, (ThreadFactory)new NamedThreadFactory("cloudwatch-metrics-publisher"));
    }

    public CloudWatchMeterRegistry(CloudWatchConfig config, Clock clock, AmazonCloudWatchAsync amazonCloudWatchAsync, ThreadFactory threadFactory) {
        super((StepRegistryConfig)config, clock);
        Objects.requireNonNull(config.namespace());
        this.amazonCloudWatchAsync = amazonCloudWatchAsync;
        this.config = config;
        this.config().namingConvention(NamingConvention.identity);
        this.start(threadFactory);
    }

    public void start(ThreadFactory threadFactory) {
        if (this.config.enabled()) {
            this.logger.info("publishing metrics to cloudwatch every " + TimeUtils.format((Duration)this.config.step()));
        }
        super.start(threadFactory);
    }

    protected void publish() {
        for (List<MetricDatum> batch : MetricDatumPartition.partition(this.metricData(), this.config.batchSize())) {
            this.sendMetricData(batch);
        }
    }

    private void sendMetricData(List<MetricDatum> metricData) {
        PutMetricDataRequest putMetricDataRequest = new PutMetricDataRequest().withNamespace(this.config.namespace()).withMetricData(metricData);
        final CountDownLatch latch = new CountDownLatch(1);
        this.amazonCloudWatchAsync.putMetricDataAsync(putMetricDataRequest, (AsyncHandler)new AsyncHandler<PutMetricDataRequest, PutMetricDataResult>(){

            public void onError(Exception exception) {
                if (exception instanceof AbortedException) {
                    CloudWatchMeterRegistry.this.logger.warn("sending metric data was aborted: {}", (Object)exception.getMessage());
                } else {
                    CloudWatchMeterRegistry.this.logger.error("error sending metric data.", (Throwable)exception);
                }
                latch.countDown();
            }

            public void onSuccess(PutMetricDataRequest request, PutMetricDataResult result) {
                CloudWatchMeterRegistry.this.logger.debug("published metric with namespace:{}", (Object)request.getNamespace());
                latch.countDown();
            }
        });
        try {
            latch.await(this.config.readTimeout().toMillis(), TimeUnit.MILLISECONDS);
        }
        catch (InterruptedException e) {
            this.logger.warn("metrics push to cloudwatch took longer than expected");
        }
    }

    List<MetricDatum> metricData() {
        Batch batch = new Batch();
        return this.getMeters().stream().flatMap(m -> (Stream)m.match(x$0 -> batch.gaugeData(x$0), x$0 -> batch.counterData(x$0), x$0 -> batch.timerData(x$0), x$0 -> batch.summaryData(x$0), x$0 -> batch.longTaskTimerData(x$0), x$0 -> batch.timeGaugeData(x$0), batch::functionCounterData, x$0 -> batch.functionTimerData(x$0), x$0 -> batch.metricData(x$0))).collect(Collectors.toList());
    }

    protected TimeUnit getBaseTimeUnit() {
        return TimeUnit.MILLISECONDS;
    }

    class Batch {
        private long wallTime;

        Batch() {
            this.wallTime = CloudWatchMeterRegistry.this.clock.wallTime();
        }

        private Stream<MetricDatum> gaugeData(Gauge gauge) {
            double value = gauge.value();
            if (!Double.isFinite(value)) {
                return Stream.empty();
            }
            return Stream.of(this.metricDatum(gauge.getId(), "value", value));
        }

        private Stream<MetricDatum> counterData(Counter counter) {
            return Stream.of(this.metricDatum(counter.getId(), "count", counter.count()));
        }

        private Stream<MetricDatum> timerData(Timer timer) {
            Stream.Builder<MetricDatum> metrics = Stream.builder();
            metrics.add(this.metricDatum(timer.getId(), "sum", CloudWatchMeterRegistry.this.getBaseTimeUnit().name(), timer.totalTime(CloudWatchMeterRegistry.this.getBaseTimeUnit())));
            metrics.add(this.metricDatum(timer.getId(), "count", "count", timer.count()));
            metrics.add(this.metricDatum(timer.getId(), "avg", CloudWatchMeterRegistry.this.getBaseTimeUnit().name(), timer.mean(CloudWatchMeterRegistry.this.getBaseTimeUnit())));
            metrics.add(this.metricDatum(timer.getId(), "max", CloudWatchMeterRegistry.this.getBaseTimeUnit().name(), timer.max(CloudWatchMeterRegistry.this.getBaseTimeUnit())));
            return metrics.build();
        }

        private Stream<MetricDatum> summaryData(DistributionSummary summary) {
            Stream.Builder<MetricDatum> metrics = Stream.builder();
            metrics.add(this.metricDatum(summary.getId(), "sum", summary.totalAmount()));
            metrics.add(this.metricDatum(summary.getId(), "count", summary.count()));
            metrics.add(this.metricDatum(summary.getId(), "avg", summary.mean()));
            metrics.add(this.metricDatum(summary.getId(), "max", summary.max()));
            return metrics.build();
        }

        private Stream<MetricDatum> longTaskTimerData(LongTaskTimer longTaskTimer) {
            return Stream.of(this.metricDatum(longTaskTimer.getId(), "activeTasks", longTaskTimer.activeTasks()), this.metricDatum(longTaskTimer.getId(), "duration", longTaskTimer.duration(CloudWatchMeterRegistry.this.getBaseTimeUnit())));
        }

        private Stream<MetricDatum> timeGaugeData(TimeGauge gauge) {
            double value = gauge.value(CloudWatchMeterRegistry.this.getBaseTimeUnit());
            if (!Double.isFinite(value)) {
                return Stream.empty();
            }
            return Stream.of(this.metricDatum(gauge.getId(), "value", value));
        }

        Stream<MetricDatum> functionCounterData(FunctionCounter counter) {
            double count = counter.count();
            if (Double.isFinite(count)) {
                return Stream.of(this.metricDatum(counter.getId(), "count", count));
            }
            return Stream.empty();
        }

        private Stream<MetricDatum> functionTimerData(FunctionTimer timer) {
            return Stream.of(this.metricDatum(timer.getId(), "count", timer.count()), this.metricDatum(timer.getId(), "avg", timer.mean(CloudWatchMeterRegistry.this.getBaseTimeUnit())));
        }

        private Stream<MetricDatum> metricData(Meter m) {
            return StreamSupport.stream(m.measure().spliterator(), false).map(ms -> this.metricDatum(m.getId().withTag(ms.getStatistic()), ms.getValue())).filter(Objects::nonNull);
        }

        @Nullable
        private MetricDatum metricDatum(Meter.Id id, double value) {
            return this.metricDatum(id, null, null, value);
        }

        @Nullable
        private MetricDatum metricDatum(Meter.Id id, @Nullable String suffix, double value) {
            return this.metricDatum(id, suffix, null, value);
        }

        @Nullable
        private MetricDatum metricDatum(Meter.Id id, @Nullable String suffix, @Nullable String unit, double value) {
            if (Double.isNaN(value)) {
                return null;
            }
            List tags = id.getConventionTags(CloudWatchMeterRegistry.this.config().namingConvention());
            return new MetricDatum().withMetricName(this.getMetricName(id, suffix)).withDimensions(this.toDimensions(tags)).withTimestamp(new Date(this.wallTime)).withValue(Double.valueOf(CloudWatchUtils.clampMetricValue(value))).withUnit(this.toStandardUnit(unit));
        }

        String getMetricName(Meter.Id id, @Nullable String suffix) {
            String name = suffix != null ? id.getName() + "." + suffix : id.getName();
            return CloudWatchMeterRegistry.this.config().namingConvention().name(name, id.getType(), id.getBaseUnit());
        }

        private StandardUnit toStandardUnit(@Nullable String unit) {
            if (unit == null) {
                return StandardUnit.None;
            }
            switch (unit.toLowerCase()) {
                case "bytes": {
                    return StandardUnit.Bytes;
                }
                case "milliseconds": {
                    return StandardUnit.Milliseconds;
                }
                case "count": {
                    return StandardUnit.Count;
                }
            }
            return StandardUnit.None;
        }

        private List<Dimension> toDimensions(List<Tag> tags) {
            return tags.stream().map(tag -> new Dimension().withName(tag.getKey()).withValue(tag.getValue())).collect(Collectors.toList());
        }
    }
}

