package com.sksamuel.kotest.engine.spec.config

import io.kotest.common.ExperimentalKotest
import io.kotest.core.spec.style.FreeSpec
import io.kotest.matchers.shouldBe
import java.util.concurrent.atomic.AtomicInteger
import kotlin.time.Duration
import kotlin.time.ExperimentalTime

/**
 * A test that just ensures the syntax for test configs does not break between releases.
 * The actual functionality of things like tags and timeouts is tested elsewhere.
 */
@ExperimentalTime
@ExperimentalKotest
class FreeSpecConfigSyntaxTest : FreeSpec() {
   init {

      val counter = AtomicInteger(0)

      afterSpec {
         counter.get() shouldBe 22
      }

      "a test disabled by an enabled flag".config(enabled = false) {
         error("boom")
      }

      "a test disabled by an enabled function".config(enabledIf = { System.currentTimeMillis() == 0L }) {
         error("boom")
      }

      "a test with multiple invocations".config(invocations = 2) {
         counter.incrementAndGet()
      }

      "a test with multiple threads".config(threads = 2, invocations = 3) {
         counter.incrementAndGet()
      }

      "a test with timeout".config(timeout = Duration.seconds(1)) {
         counter.incrementAndGet()
      }

      "a test with tags".config(tags = setOf(Tag1)) {
         counter.incrementAndGet()
      }

      "a test with multiple tags".config(tags = setOf(Tag1, Tag2)) {
         counter.incrementAndGet()
      }

      "an outer context with timeout".config(timeout = Duration.seconds(2)) - {
         counter.incrementAndGet()
         "an inner test" {
            counter.incrementAndGet()
         }
      }

      "an outer context with tags".config(tags = setOf(Tag1)) - {
         counter.incrementAndGet()
         "an inner test" {
            counter.incrementAndGet()
         }
      }

      "an outer context with multiple tags".config(tags = setOf(Tag1, Tag2)) - {
         counter.incrementAndGet()
         "an inner test" {
            counter.incrementAndGet()
         }
      }

      "an outer context disabled by an enabled flag".config(enabled = false) - {
         error("boom")
         "an inner test" { error("boom") }
      }

      "an outer context disabled by an enabled function".config(enabledIf = { System.currentTimeMillis() == 0L }) - {
         error("boom")
         "an inner test" { error("boom") }
      }

      "an outer context" - {

         "inner" {
            counter.incrementAndGet()
         }

         counter.incrementAndGet()
         "an inner context with timeout".config(timeout = Duration.seconds(2)) - {
            counter.incrementAndGet()
            "an inner test" {
               counter.incrementAndGet()
            }
         }

         "an inner context with tags".config(tags = setOf(Tag1)) - {
            counter.incrementAndGet()
            "an inner test" {
               counter.incrementAndGet()
            }
         }

         "an inner context with multiple tags".config(tags = setOf(Tag1, Tag2)) - {
            counter.incrementAndGet()
            "an inner test" {
               counter.incrementAndGet()
            }
         }

         "an inner context disabled by an enabled flag".config(enabled = false) - {
            error("boom")
            "an inner test" { error("boom") }
         }

         "an inner context disabled by an enabled function".config(enabledIf = { System.currentTimeMillis() == 0L }) - {
            error("boom")
            "an inner test" { error("boom") }
         }
      }
   }
}
