/*
 * Decompiled with CFR 0.152.
 */
package com.nirima.jenkins.plugins.docker;

import com.github.dockerjava.api.DockerClient;
import com.github.dockerjava.api.model.Container;
import com.nirima.jenkins.plugins.docker.ContainerNodeNameMap;
import com.nirima.jenkins.plugins.docker.DockerCloud;
import com.nirima.jenkins.plugins.docker.DockerContainerLabelKeys;
import com.nirima.jenkins.plugins.docker.DockerDisabled;
import com.nirima.jenkins.plugins.docker.DockerTemplateBase;
import com.nirima.jenkins.plugins.docker.utils.JenkinsUtils;
import hudson.Extension;
import hudson.model.AsyncPeriodicWork;
import hudson.model.Node;
import hudson.model.TaskListener;
import hudson.slaves.SlaveComputer;
import io.jenkins.docker.DockerTransientNode;
import io.jenkins.docker.client.DockerAPI;
import java.io.IOException;
import java.time.Clock;
import java.time.Duration;
import java.time.Instant;
import java.util.Collection;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import jenkins.model.Jenkins;
import org.kohsuke.accmod.Restricted;
import org.kohsuke.accmod.restrictions.NoExternalUse;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

@Extension
public class DockerContainerWatchdog
extends AsyncPeriodicWork {
    private Clock clock = Clock.systemUTC();
    private static final Logger LOGGER = LoggerFactory.getLogger(DockerContainerWatchdog.class);
    private static final long RECURRENCE_PERIOD_IN_MS = JenkinsUtils.getSystemPropertyLong(DockerContainerWatchdog.class.getName() + ".recurrenceInSeconds", 300L) * 1000L;
    private static final Duration PROCESSING_TIMEOUT_IN_MS = Duration.ofMillis(RECURRENCE_PERIOD_IN_MS * 4L / 5L);
    private static final Statistics executionStatistics = new Statistics();

    public DockerContainerWatchdog() {
        super(String.format("%s Asynchronous Periodic Work", DockerContainerWatchdog.class.getSimpleName()));
    }

    @Restricted(value={NoExternalUse.class})
    public void setClock(Clock clock) {
        this.clock = clock;
    }

    public long getRecurrencePeriod() {
        return RECURRENCE_PERIOD_IN_MS;
    }

    protected List<DockerCloud> getAllClouds() {
        return DockerCloud.instances();
    }

    protected List<Node> getAllNodes() {
        return Jenkins.getInstance().getNodes();
    }

    protected String getJenkinsInstanceId() {
        return DockerTemplateBase.getJenkinsInstanceIdForContainerLabel();
    }

    protected void removeNode(DockerTransientNode dtn) throws IOException {
        Jenkins.getInstance().removeNode((Node)dtn);
    }

    protected boolean stopAndRemoveContainer(DockerAPI dockerApi, Logger logger, String description, boolean removeVolumes, String containerId, boolean stop) {
        return DockerTransientNode.stopAndRemoveContainer(dockerApi, logger, description, removeVolumes, containerId, stop);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    protected void execute(TaskListener listener) throws IOException, InterruptedException {
        if (!JenkinsUtils.getSystemPropertyBoolean(DockerContainerWatchdog.class.getName() + ".enabled", true)) {
            LOGGER.info("Docker Container Watchdog is disabled based on system configuration");
            return;
        }
        LOGGER.info("Docker Container Watchdog has been triggered");
        executionStatistics.writeStatisticsToLog();
        DockerContainerWatchdog.executionStatistics.addExecution();
        Instant start = this.clock.instant();
        try {
            ContainerNodeNameMap csmMerged = new ContainerNodeNameMap();
            Instant snapshotInstance = this.clock.instant();
            Map<String, Node> nodeMap = this.loadNodeMap();
            try {
                for (DockerCloud dc : this.getAllClouds()) {
                    String uri = dc.getDockerApi().getDockerHost().getUri();
                    if (uri == null) {
                        LOGGER.info("Skipping unconfigured Docker Cloud {}", (Object)dc.getDisplayName());
                        continue;
                    }
                    LOGGER.info("Checking Docker Cloud {} at {}", (Object)dc.getDisplayName(), (Object)uri);
                    listener.getLogger().println(String.format("Checking Docker Cloud %s", dc.getDisplayName()));
                    csmMerged = this.processCloud(dc, nodeMap, csmMerged, snapshotInstance);
                }
                if (csmMerged.isContainerListIncomplete()) {
                    LOGGER.info("Not checking the list of nodes, as list of containers is known to be incomplete");
                } else {
                    this.cleanUpSuperfluousComputer(nodeMap, csmMerged, snapshotInstance);
                }
            }
            catch (WatchdogProcessingTimeout timeout) {
                LOGGER.warn("Processing of cleanup watchdog took too long; current timeout value: {} ms, watchdog started on {}", new Object[]{PROCESSING_TIMEOUT_IN_MS, start.toString(), timeout});
                DockerContainerWatchdog.executionStatistics.addProcessingTimeout();
                DockerContainerWatchdog.executionStatistics.addOverallRuntime(Duration.between(start, this.clock.instant()).toMillis());
                Instant stop = this.clock.instant();
                DockerContainerWatchdog.executionStatistics.addOverallRuntime(Duration.between(start, stop).toMillis());
                return;
            }
        }
        finally {
            Instant stop = this.clock.instant();
            DockerContainerWatchdog.executionStatistics.addOverallRuntime(Duration.between(start, stop).toMillis());
        }
        LOGGER.info("Docker Container Watchdog check has been completed");
    }

    private Map<String, Node> loadNodeMap() {
        HashMap<String, Node> nodeMap = new HashMap<String, Node>();
        for (Node n : this.getAllNodes()) {
            nodeMap.put(n.getNodeName(), n);
        }
        LOGGER.info("We currently have {} nodes assigned to this Jenkins instance, which we will check", (Object)nodeMap.size());
        return nodeMap;
    }

    private ContainerNodeNameMap processCloud(DockerCloud dc, Map<String, Node> nodeMap, ContainerNodeNameMap csmMerged, Instant snapshotInstant) {
        DockerAPI dockerApi = dc.getDockerApi();
        try (DockerClient client = dockerApi.getClient();){
            ContainerNodeNameMap csm = this.retrieveContainers(dc, client);
            DockerDisabled dcDisabled = dc.getDisabled();
            if (dcDisabled.isDisabled()) {
                LOGGER.info("Will not cleanup superfluous containers on DockerCloud [name={}, dockerURI={}], as it is disabled", (Object)dc.getDisplayName(), (Object)dc.getDockerApi().getDockerHost().getUri());
            } else {
                this.cleanUpSuperfluousContainers(client, nodeMap, csm, dc, snapshotInstant);
            }
            csmMerged = csmMerged.merge(csm);
        }
        catch (IOException e) {
            LOGGER.warn("Failed to properly close a DockerClient instance after reading the list of containers and cleaning them up; ignoring", (Throwable)e);
        }
        catch (ContainersRetrievalException e) {
            csmMerged.setContainerListIncomplete(true);
        }
        return csmMerged;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private ContainerNodeNameMap retrieveContainers(DockerCloud dc, DockerClient client) throws ContainersRetrievalException {
        HashMap<String, String> labelFilter = new HashMap<String, String>();
        labelFilter.put(DockerContainerLabelKeys.JENKINS_INSTANCE_ID, this.getJenkinsInstanceId());
        ContainerNodeNameMap result = new ContainerNodeNameMap();
        Instant start = this.clock.instant();
        try {
            List containerList = null;
            try {
                containerList = (List)client.listContainersCmd().withShowAll(Boolean.valueOf(true)).withLabelFilter(labelFilter).exec();
            }
            catch (Exception e) {
                LOGGER.warn("Unable to retrieve list of containers available on DockerCloud [name={}, dockerURI={}] while reading list of containers (showAll=true, labelFilters={})", new Object[]{dc.getDisplayName(), dc.getDockerApi().getDockerHost().getUri(), ((Object)labelFilter).toString(), e});
                throw new ContainersRetrievalException((Throwable)e);
            }
            for (Container container : containerList) {
                String containerId = container.getId();
                String status = container.getStatus();
                if (status == null) {
                    LOGGER.warn("Container {} has a null-status and thus cannot be checked; ignoring container", (Object)containerId);
                    continue;
                }
                Map containerLabels = container.getLabels();
                String containerNodeName = (String)containerLabels.get(DockerContainerLabelKeys.NODE_NAME);
                if (containerNodeName == null) {
                    LOGGER.warn("Container {} is said to be created by this Jenkins instance, but does not have any node name label assigned; manual cleanup is required for this", (Object)containerId);
                    continue;
                }
                result.registerMapping(container, containerNodeName);
            }
        }
        finally {
            Instant stop = this.clock.instant();
            DockerContainerWatchdog.executionStatistics.addRetrieveContainerRuntime(Duration.between(start, stop).toMillis());
        }
        return result;
    }

    private void cleanUpSuperfluousContainers(DockerClient client, Map<String, Node> nodeMap, ContainerNodeNameMap csm, DockerCloud dc, Instant snapshotInstant) {
        Collection<Container> allContainers = csm.getAllContainers();
        for (Container container : allContainers) {
            String nodeName = csm.getNodeName(container.getId());
            Node node = nodeMap.get(nodeName);
            if (node != null || this.isStillTooYoung(container.getCreated(), snapshotInstant)) continue;
            this.checkForTimeout(snapshotInstant);
            LOGGER.info("Container {}, which is reported to be assigned to node {}, is no longer associated (node might be gone already?). The container's last status is {}; it was created on {}", new Object[]{container.getId(), nodeName, container.getStatus(), container.getCreated()});
            try {
                this.terminateContainer(dc, client, container);
            }
            catch (Exception e) {
                LOGGER.warn("Graceful termination of Container {} failed; terminating directly via API - this may cause remnants to be left behind", (Object)container.getId(), (Object)e);
            }
        }
    }

    private boolean isStillTooYoung(Long created, Instant snapshotInstant) {
        Instant createdInstant = Instant.ofEpochSecond(created);
        Duration containerLifetime = Duration.between(createdInstant, snapshotInstant);
        Duration graceDurationForContainers = Duration.ofSeconds(JenkinsUtils.getSystemPropertyLong(DockerContainerWatchdog.class.getName() + ".initialGraceDurationForContainersInSeconds", 60L));
        Duration untilMayBeCleanedUp = containerLifetime.minus(graceDurationForContainers);
        return untilMayBeCleanedUp.isNegative();
    }

    private void terminateContainer(DockerCloud dc, DockerClient client, Container container) {
        boolean gracefulFailed = false;
        try {
            this.terminateContainerGracefully(dc, container);
        }
        catch (TerminationException e) {
            gracefulFailed = true;
        }
        catch (ContainerIsTaintedException e) {
            LOGGER.warn("Container {} has been tampered with; skipping cleanup", (Object)container.getId(), (Object)e);
            return;
        }
        if (gracefulFailed) {
            try {
                Instant start = this.clock.instant();
                client.removeContainerCmd(container.getId()).withForce(Boolean.valueOf(true)).exec();
                Instant stop = this.clock.instant();
                DockerContainerWatchdog.executionStatistics.addContainerRemovalForce(Duration.between(start, stop).toMillis());
            }
            catch (RuntimeException e) {
                LOGGER.warn("Forced termination of container {} failed with RuntimeException", (Object)container.getId(), (Object)e);
                DockerContainerWatchdog.executionStatistics.addContainerRemovalFailed();
            }
        }
    }

    private void terminateContainerGracefully(DockerCloud dc, Container container) throws TerminationException, ContainerIsTaintedException {
        String containerId = container.getId();
        Map containerLabels = container.getLabels();
        String nodeName = (String)containerLabels.get(DockerContainerLabelKeys.NODE_NAME);
        if (this.nodeExistsBypassingCache(nodeName)) {
            LOGGER.warn("Attempt to terminate container ID {} gracefully, but a node for it suddenly has appeared", (Object)container.getId());
            throw new ContainerIsTaintedException(String.format("Node for container ID %s has appeared", container.getId()));
        }
        containerLabels.get(DockerContainerLabelKeys.REMOVE_VOLUMES);
        String removeVolumesString = (String)containerLabels.get(DockerContainerLabelKeys.REMOVE_VOLUMES);
        if (removeVolumesString == null) {
            throw new ContainerIsTaintedException(String.format("Container ID %s has no '%s' label; skipping.", container.getId(), DockerContainerLabelKeys.REMOVE_VOLUMES));
        }
        boolean removeVolumes = Boolean.parseBoolean(removeVolumesString);
        boolean containerRunning = true;
        if (container.getStatus().startsWith("Dead") || container.getStatus().startsWith("Exited") || container.getStatus().startsWith("Created")) {
            containerRunning = false;
        }
        DockerAPI dockerApi = dc.getDockerApi();
        Instant start = this.clock.instant();
        boolean success = this.stopAndRemoveContainer(dockerApi, LOGGER, String.format("(orphaned container found by %s)", DockerContainerWatchdog.class.getSimpleName()), removeVolumes, container.getId(), !containerRunning);
        Instant stop = this.clock.instant();
        if (!success) {
            throw new TerminationException("Graceful termination failed.");
        }
        DockerContainerWatchdog.executionStatistics.addContainerRemovalGracefully(Duration.between(start, stop).toMillis());
        LOGGER.info("Successfully terminated orphaned container {}", (Object)containerId);
    }

    private boolean nodeExistsBypassingCache(String nodeName) {
        for (Node node : this.getAllNodes()) {
            if (!nodeName.equals(node.getNodeName())) continue;
            return true;
        }
        return false;
    }

    private void cleanUpSuperfluousComputer(Map<String, Node> nodeMap, ContainerNodeNameMap csmMerged, Instant snapshotInstant) {
        for (Node node : nodeMap.values()) {
            SlaveComputer computer;
            if (!(node instanceof DockerTransientNode)) continue;
            this.checkForTimeout(snapshotInstant);
            DockerTransientNode dtn = (DockerTransientNode)node;
            boolean seenOnDockerInstance = csmMerged.isContainerIdRegistered(dtn.getContainerId());
            if (seenOnDockerInstance || (computer = dtn.getComputer()) == null || !computer.isOffline()) continue;
            LOGGER.info("{} has container ID {}, but the container does not exist in any docker cloud. Will remove node.", (Object)dtn, (Object)dtn.getContainerId());
            try {
                this.removeNode(dtn);
                DockerContainerWatchdog.executionStatistics.addNodeRemoved();
            }
            catch (IOException e) {
                LOGGER.warn("Failed to remove orphaned node {}", (Object)dtn.toString(), (Object)e);
                DockerContainerWatchdog.executionStatistics.addNodeRemovedFailed();
            }
        }
    }

    private void checkForTimeout(Instant startedTimestamp) {
        Instant now = this.clock.instant();
        Duration runtime = Duration.between(startedTimestamp, now);
        if (runtime.compareTo(PROCESSING_TIMEOUT_IN_MS) > 0) {
            throw new WatchdogProcessingTimeout();
        }
    }

    private static class Statistics {
        private long executions;
        private long containersRemovedGracefully;
        private long containersRemovedGracefullyRuntimeSum;
        private long containersRemovedForce;
        private long containersRemovedForceRuntimeSum;
        private long containersRemovedFailed;
        private long nodesRemoved;
        private long nodesRemovedFailed;
        private long processingTimeout;
        private long overallRuntime;
        private long retrieveContainersRuntime;
        private long retrieveContainersCalls;

        private Statistics() {
        }

        public void writeStatisticsToLog() {
            LOGGER.info("Watchdog Statistics: Number of overall executions: {}, Executions with processing timeout: {}, Containers removed gracefully: {}, Containers removed with force: {}, Containers removal failed: {}, Nodes removed successfully: {}, Nodes removal failed: {}, Container removal average duration (gracefully): {} ms, Container removal average duration (force): {} ms, Average overall runtime of watchdog: {} ms, Average runtime of container retrieval: {} ms", new Object[]{this.executions, this.processingTimeout, this.containersRemovedGracefully, this.containersRemovedForce, this.containersRemovedFailed, this.nodesRemoved, this.nodesRemovedFailed, this.getContainerRemovalAverageDurationGracefully(), this.getContainerRemovalAverageDurationForce(), this.getAverageOverallRuntime(), this.getAverageRetrieveContainerRuntime()});
        }

        private void addExecution() {
            ++this.executions;
        }

        private void addContainerRemovalGracefully(long runtime) {
            ++this.containersRemovedGracefully;
            this.containersRemovedGracefullyRuntimeSum += runtime;
        }

        private void addContainerRemovalForce(long runtime) {
            ++this.containersRemovedForce;
            this.containersRemovedForceRuntimeSum += runtime;
        }

        private void addContainerRemovalFailed() {
            ++this.containersRemovedFailed;
        }

        private void addNodeRemoved() {
            ++this.nodesRemoved;
        }

        private void addNodeRemovedFailed() {
            ++this.nodesRemovedFailed;
        }

        private void addProcessingTimeout() {
            ++this.processingTimeout;
        }

        private void addOverallRuntime(long runtime) {
            this.overallRuntime += runtime;
        }

        private void addRetrieveContainerRuntime(long runtime) {
            this.retrieveContainersRuntime += runtime;
            ++this.retrieveContainersCalls;
        }

        private String getAverageOverallRuntime() {
            if (this.executions == 0L) {
                return "0";
            }
            return new Long(this.overallRuntime / this.executions).toString();
        }

        private String getContainerRemovalAverageDurationForce() {
            if (this.containersRemovedForce == 0L) {
                return "0";
            }
            return new Long(this.containersRemovedForceRuntimeSum / this.containersRemovedForce).toString();
        }

        private String getContainerRemovalAverageDurationGracefully() {
            if (this.containersRemovedGracefully == 0L) {
                return "0";
            }
            return new Long(this.containersRemovedGracefullyRuntimeSum / this.containersRemovedGracefully).toString();
        }

        private String getAverageRetrieveContainerRuntime() {
            if (this.retrieveContainersCalls == 0L) {
                return "0";
            }
            return new Long(this.retrieveContainersRuntime / this.retrieveContainersCalls).toString();
        }
    }

    private static class ContainerIsTaintedException
    extends Exception {
        private static final long serialVersionUID = -8500246547989418166L;

        public ContainerIsTaintedException(String message) {
            super(message);
        }
    }

    private static class TerminationException
    extends Exception {
        private static final long serialVersionUID = -7259431101547222511L;

        public TerminationException(String message) {
            super(message);
        }
    }

    private class ContainersRetrievalException
    extends Exception {
        private static final long serialVersionUID = -3370783213009509440L;

        public ContainersRetrievalException(Throwable cause) {
            super(cause);
        }
    }

    private static class WatchdogProcessingTimeout
    extends Error {
        private static final long serialVersionUID = 2162341066478288340L;
    }
}

