package io.intercom.android.nexus;

import androidx.annotation.NonNull;
import androidx.annotation.Nullable;

import com.intercom.twig.Twig;

import java.util.ArrayList;
import java.util.List;
import java.util.concurrent.Executors;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.ScheduledFuture;
import java.util.concurrent.ThreadFactory;
import java.util.concurrent.TimeUnit;

import okhttp3.OkHttpClient;

public class NexusClient implements NexusTopicProvider {
    private final List<NexusSocket> sockets = new ArrayList<>();
    private final List<String> topics = new ArrayList<>();
    private final Twig twig;
    private final OkHttpClient client;
    private final NexusEventPropagator eventPropagator;

    private @Nullable ScheduledExecutorService backgroundTaskExecutor;
    private ScheduledFuture future;
    private long presenceInterval;

    public static OkHttpClient.Builder defaultOkHttpClientBuilder() {
        // allow for 2 minute heartbeats
        return new OkHttpClient.Builder()
                .readTimeout(130, TimeUnit.SECONDS)
                .writeTimeout(130, TimeUnit.SECONDS)
                .connectTimeout(20, TimeUnit.SECONDS);
    }

    public NexusClient(Twig twig) {
        this(twig, defaultOkHttpClientBuilder().build());
    }

    public NexusClient(Twig twig, OkHttpClient okHttpClient) {
        this(twig, okHttpClient, new NexusEventPropagator(twig));
    }

    NexusClient(Twig twig, OkHttpClient client, NexusEventPropagator eventPropagator) {
        this.twig = twig;
        this.eventPropagator = eventPropagator;
        this.client = client;
    }

    public void connect(NexusConfig config, boolean shouldSendPresence) {
        if (config.getEndpoints().isEmpty()) {
            twig.e("No endpoints present");
            return;
        }

        if (backgroundTaskExecutor == null) {
            // one thread per connection + 1 for the client itself
            ThreadFactory threadFactory = new NexusThreadFactory();
            backgroundTaskExecutor = Executors.newScheduledThreadPool(config.getEndpoints().size() + 1, threadFactory);
        }

        for (String url : config.getEndpoints()) {
            twig.i("Adding socket");
            NexusSocket socket = new NexusSocket(
                    url,
                    config.getConnectionTimeout(),
                    shouldSendPresence,
                    twig,
                    backgroundTaskExecutor,
                    client,
                    eventPropagator,
                    this);
            socket.connect();
            sockets.add(socket);
        }

        presenceInterval = config.getPresenceHeartbeatInterval();

        if (shouldSendPresence) {
            schedulePresence();
        }
    }

    public synchronized void disconnect() {
        if (!sockets.isEmpty()) {
            for (NexusSocket nexusSocket : sockets) {
                twig.i("disconnecting socket");
                nexusSocket.disconnect();
            }
            sockets.clear();
            twig.i("client disconnected");
        }

        if (future != null) {
            future.cancel(true);
        }
    }

    public synchronized void fire(NexusEvent event) {
        eventPropagator.cacheEvent(event);
        String data = event.toStringEncodedJsonObject();
        if (!data.isEmpty()) {
            for (NexusSocket nexusSocket : sockets) {
                nexusSocket.fire(data);
            }
        }
    }

    public synchronized void localUpdate(@NonNull NexusEvent event) {
        eventPropagator.notifyEvent(event);
    }

    public synchronized boolean isConnected() {
        for (NexusSocket socket : sockets) {
            if (socket.isConnected()) {
                return true;
            }
        }
        return false;
    }

    public void addEventListener(@NonNull NexusListener listener) {
        eventPropagator.addListener(listener);
    }

    public void removeEventListener(@NonNull NexusListener listener) {
        eventPropagator.removeListener(listener);
    }

    @Override @NonNull public synchronized List<String> getTopics() {
        return topics;
    }

    public synchronized void setTopics(@NonNull List<String> newTopics) {
        List<String> topicsToSubscribe = new ArrayList<>(newTopics);
        topicsToSubscribe.removeAll(topics);

        List<String> topicsToUnsubscribe = new ArrayList<>(topics);
        topicsToUnsubscribe.removeAll(newTopics);

        subscribeToTopics(topicsToSubscribe);
        unSubscribeFromTopics(topicsToUnsubscribe);

        topics.clear();
        topics.addAll(newTopics);
    }

    public synchronized void addTopics(@NonNull List<String> newTopics) {
        List<String> topicsToSubscribe = new ArrayList<>(newTopics);
        topicsToSubscribe.removeAll(topics);
        subscribeToTopics(topicsToSubscribe);
        topics.addAll(topicsToSubscribe);
    }

    public synchronized void removeTopics(@NonNull List<String> topicsToRemove) {
        List<String> topicsToUnsubscribe = new ArrayList<>();
        for (String topic : topicsToRemove) {
            if (topics.contains(topic)) {
                topicsToUnsubscribe.add(topic);
            }
        }
        unSubscribeFromTopics(topicsToUnsubscribe);
        topics.removeAll(topicsToUnsubscribe);
    }

    public synchronized void clearTopics() {
        unSubscribeFromTopics(topics);
        topics.clear();
    }

    private void subscribeToTopics(@NonNull List<String> topics) {
        if (!topics.isEmpty()) {
            fire(NexusEvent.getSubscribeEvent(topics));
        }
    }

    private void unSubscribeFromTopics(@NonNull List<String> topics) {
        if (!topics.isEmpty()) {
            fire(NexusEvent.getUnsubscribeEvent(topics));
        }
    }

    private void schedulePresence() {
        if (presenceInterval > 0) {
            future = backgroundTaskExecutor.schedule(new Runnable() {
                @Override public void run() {
                    fire(NexusEvent.getUserPresenceEvent());
                    schedulePresence();
                }
            }, presenceInterval, TimeUnit.SECONDS);
        }
    }

    private static class NexusThreadFactory implements ThreadFactory {

        private final ThreadFactory defaultFactory = Executors.defaultThreadFactory();

        private int threadCount = 0;

        @Override public Thread newThread(@NonNull Runnable r) {
            Thread thread = defaultFactory.newThread(r);
            threadCount++;
            thread.setName("IntercomNexus-" + threadCount);
            return thread;
        }
    }
}
