package io.intercom.android.sdk.identity;

import android.support.annotation.NonNull;
import android.support.annotation.Nullable;
import android.text.TextUtils;

import com.intercom.twig.Twig;

import io.intercom.android.sdk.Intercom;
import io.intercom.android.sdk.UserAttributes;
import io.intercom.android.sdk.logger.LumberMill;

/**
 * <p>The Registration object is used for registering identified users with Intercom.</p>
 * <p>This is an example of how to register a user with a user ID</p>
 * <pre>
 * {@code
 * Intercom.client().registerIdentifiedUser(Registration.create().withUserId("12345"));
 * }
 * </pre>
 *
 * <p>You can chain methods on the Registration object if you want to provide more data with the registration.</p>
 *
 * <pre>
 * {@code
 * Map<String, Object> userAttributes = new HashMap<>();
 * userAttributes.put("name", "Admin name");
 *
 * Registration.create()
 *     .withUserId("12345")
 *     .withEmail("test@email.com")
 *     .withUserAttributes(userAttributes);
 * Intercom.client().registerIdentifiedUser(registration);
 * }
 * </pre>
 */
public class Registration {
    private final Twig twig = LumberMill.getLogger();

    private String email = "";
    private String userId = "";
    @Nullable private UserAttributes attributes;

    private Validity validity = Validity.NOT_SET;

    /**
     * Factory to create a new, empty Registration object.
     *
     * @return a new Registration
     */
    public static Registration create() {
        return new Registration();
    }

    /**
     * Add an email address to this registration.
     *
     * @param email the user's email address
     * @return this registration object
     */
    public Registration withEmail(@NonNull String email) {
        boolean validEmail = !TextUtils.isEmpty(email);
        if (validEmail) {
            this.email = email;
        } else {
            twig.e("Email cannot be null or empty");
        }
        updateState(validEmail);
        return this;
    }

    /**
     * Add a unique user identifier to this registration.
     *
     * @param userId A unique identifier for your user
     * @return this registration object
     */
    public Registration withUserId(@NonNull String userId) {
        boolean validUserId = !TextUtils.isEmpty(userId);
        if (validUserId) {
            this.userId = userId;
        } else {
            twig.e("UserId cannot be null or empty");
        }
        updateState(validUserId);
        return this;
    }

    /**
     * <p>Add a map of user attributes to this registration.</p>
     *
     * <p>For full documentation see {@link Intercom#updateUser(UserAttributes)}</p>
     *
     * @param userAttributes the UserAttributes object
     * @return this registration object
     */
    public Registration withUserAttributes(@NonNull UserAttributes userAttributes) {
        if (userAttributes == null) {
            validity = Validity.INVALID;
            twig.e("Registration.withUserAttributes method failed: "
                    + "the attributes Map provided is null");
        } else if (userAttributes.isEmpty()) {
            validity = Validity.INVALID;
            twig.e("Registration.withUserAttributes method failed: "
                    + "the attributes Map provided is empty");
        } else {
            this.attributes = userAttributes;
        }
        return this;
    }

    public String getEmail() {
        return email;
    }

    public String getUserId() {
        return userId;
    }

    @Nullable public UserAttributes getAttributes() {
        return attributes;
    }

    boolean isValidRegistration() {
        return Validity.VALID.equals(validity);
    }

    private void updateState(boolean newState) {
        if (validity == Validity.NOT_SET || validity == Validity.VALID) {
            validity = newState ? Validity.VALID : Validity.INVALID;
        }
    }

    private enum Validity {
        NOT_SET,
        INVALID,
        VALID
    }
}
