/*
 * Decompiled with CFR 0.152.
 */
package io.prestosql.plugin.jdbc;

import com.google.common.base.Joiner;
import com.google.common.base.Preconditions;
import com.google.common.base.Strings;
import com.google.common.base.VerifyException;
import com.google.common.collect.ImmutableList;
import com.google.common.collect.Iterables;
import io.airlift.log.Logger;
import io.airlift.slice.Slice;
import io.prestosql.plugin.jdbc.BlockWriteFunction;
import io.prestosql.plugin.jdbc.BooleanWriteFunction;
import io.prestosql.plugin.jdbc.DoubleWriteFunction;
import io.prestosql.plugin.jdbc.JdbcClient;
import io.prestosql.plugin.jdbc.JdbcColumnHandle;
import io.prestosql.plugin.jdbc.JdbcTypeHandle;
import io.prestosql.plugin.jdbc.LongWriteFunction;
import io.prestosql.plugin.jdbc.SliceWriteFunction;
import io.prestosql.plugin.jdbc.WriteFunction;
import io.prestosql.spi.block.Block;
import io.prestosql.spi.connector.ColumnHandle;
import io.prestosql.spi.connector.ConnectorSession;
import io.prestosql.spi.predicate.Domain;
import io.prestosql.spi.predicate.Range;
import io.prestosql.spi.predicate.TupleDomain;
import io.prestosql.spi.type.Type;
import java.sql.Connection;
import java.sql.PreparedStatement;
import java.sql.SQLException;
import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.Function;
import java.util.stream.Collectors;

public class QueryBuilder {
    private static final Logger log = Logger.get(QueryBuilder.class);
    private static final String ALWAYS_TRUE = "1=1";
    private static final String ALWAYS_FALSE = "1=0";
    private final String identifierQuote;
    private boolean isPushDown;

    public QueryBuilder(String identifierQuote) {
        this.identifierQuote = Objects.requireNonNull(identifierQuote, "identifierQuote is null");
    }

    public QueryBuilder(String identifierQuote, boolean isPushDown) {
        this(identifierQuote);
        this.isPushDown = isPushDown;
    }

    public PreparedStatement buildSql(JdbcClient client, ConnectorSession session, Connection connection, String catalog, String schema, String table, List<JdbcColumnHandle> columns, TupleDomain<ColumnHandle> tupleDomain, Optional<String> additionalPredicate, Function<String, String> sqlFunction) throws SQLException {
        StringBuilder sql = new StringBuilder();
        String columnNames = columns.stream().map(JdbcColumnHandle::getColumnName).map(this::quote).collect(Collectors.joining(", "));
        sql.append("SELECT ");
        sql.append(columnNames);
        if (columns.isEmpty()) {
            sql.append("null");
        }
        sql.append(" FROM ");
        if (!Strings.isNullOrEmpty((String)catalog)) {
            sql.append(this.quote(catalog)).append('.');
        }
        if (!Strings.isNullOrEmpty((String)schema)) {
            sql.append(this.quote(schema)).append('.');
        }
        if (this.isPushDown) {
            sql.append("(").append(table).append(") pushdown");
        } else {
            sql.append(this.quote(table));
        }
        ArrayList<TypeAndValue> accumulator = new ArrayList<TypeAndValue>();
        ImmutableList clauses = this.toConjuncts(client, session, connection, columns, tupleDomain, accumulator);
        if (additionalPredicate.isPresent()) {
            clauses = ImmutableList.builder().addAll(clauses).add((Object)additionalPredicate.get()).build();
        }
        if (!clauses.isEmpty()) {
            sql.append(" WHERE ").append(Joiner.on((String)" AND ").join((Iterable)clauses));
        }
        String query = sqlFunction.apply(sql.toString());
        PreparedStatement statement = client.getPreparedStatement(connection, query);
        for (int i = 0; i < accumulator.size(); ++i) {
            TypeAndValue typeAndValue = (TypeAndValue)accumulator.get(i);
            int parameterIndex = i + 1;
            Type type = typeAndValue.getType();
            WriteFunction writeFunction = client.toPrestoType(session, connection, typeAndValue.getTypeHandle()).orElseThrow(() -> new VerifyException(String.format("Unsupported type %s with handle %s", type, typeAndValue.getTypeHandle()))).getWriteFunction();
            Class javaType = type.getJavaType();
            Object value = typeAndValue.getValue();
            if (javaType == Boolean.TYPE) {
                ((BooleanWriteFunction)writeFunction).set(statement, parameterIndex, (Boolean)value);
                continue;
            }
            if (javaType == Long.TYPE) {
                ((LongWriteFunction)writeFunction).set(statement, parameterIndex, (Long)value);
                continue;
            }
            if (javaType == Double.TYPE) {
                ((DoubleWriteFunction)writeFunction).set(statement, parameterIndex, (Double)value);
                continue;
            }
            if (javaType == Slice.class) {
                ((SliceWriteFunction)writeFunction).set(statement, parameterIndex, (Slice)value);
                continue;
            }
            if (javaType == Block.class) {
                ((BlockWriteFunction)writeFunction).set(statement, parameterIndex, (Block)value);
                continue;
            }
            throw new VerifyException(String.format("Unexpected type %s with java type %s", type, javaType.getName()));
        }
        return statement;
    }

    private static Domain pushDownDomain(JdbcClient client, ConnectorSession session, Connection connection, JdbcColumnHandle column, Domain domain) {
        return (Domain)client.toPrestoType(session, connection, column.getJdbcTypeHandle()).orElseThrow(() -> new IllegalStateException(String.format("Unsupported type %s with handle %s", column.getColumnType(), column.getJdbcTypeHandle()))).getPushdownConverter().apply(domain);
    }

    private List<String> toConjuncts(JdbcClient client, ConnectorSession session, Connection connection, List<JdbcColumnHandle> columns, TupleDomain<ColumnHandle> tupleDomain, List<TypeAndValue> accumulator) {
        if (tupleDomain.isNone()) {
            return ImmutableList.of((Object)ALWAYS_FALSE);
        }
        ImmutableList.Builder builder = ImmutableList.builder();
        for (JdbcColumnHandle column : columns) {
            Domain domain = (Domain)((Map)tupleDomain.getDomains().get()).get(column);
            if (domain == null) continue;
            domain = QueryBuilder.pushDownDomain(client, session, connection, column, domain);
            builder.add((Object)this.toPredicate(column.getColumnName(), domain, column, accumulator));
        }
        return builder.build();
    }

    private String toPredicate(String columnName, Domain domain, JdbcColumnHandle column, List<TypeAndValue> accumulator) {
        if (domain.getValues().isNone()) {
            return domain.isNullAllowed() ? this.quote(columnName) + " IS NULL" : ALWAYS_FALSE;
        }
        if (domain.getValues().isAll()) {
            return domain.isNullAllowed() ? ALWAYS_TRUE : this.quote(columnName) + " IS NOT NULL";
        }
        ArrayList<String> disjuncts = new ArrayList<String>();
        ArrayList<Object> singleValues = new ArrayList<Object>();
        for (Range range : domain.getValues().getRanges().getOrderedRanges()) {
            Preconditions.checkState((!range.isAll() ? 1 : 0) != 0);
            if (range.isSingleValue()) {
                singleValues.add(range.getLow().getValue());
                continue;
            }
            ArrayList<String> rangeConjuncts = new ArrayList<String>();
            if (!range.getLow().isLowerUnbounded()) {
                switch (range.getLow().getBound()) {
                    case ABOVE: {
                        rangeConjuncts.add(this.toPredicate(columnName, ">", range.getLow().getValue(), column, accumulator));
                        break;
                    }
                    case EXACTLY: {
                        rangeConjuncts.add(this.toPredicate(columnName, ">=", range.getLow().getValue(), column, accumulator));
                        break;
                    }
                    case BELOW: {
                        throw new IllegalArgumentException("Low marker should never use BELOW bound");
                    }
                    default: {
                        throw new AssertionError((Object)("Unhandled bound: " + range.getLow().getBound()));
                    }
                }
            }
            if (!range.getHigh().isUpperUnbounded()) {
                switch (range.getHigh().getBound()) {
                    case ABOVE: {
                        throw new IllegalArgumentException("High marker should never use ABOVE bound");
                    }
                    case EXACTLY: {
                        rangeConjuncts.add(this.toPredicate(columnName, "<=", range.getHigh().getValue(), column, accumulator));
                        break;
                    }
                    case BELOW: {
                        rangeConjuncts.add(this.toPredicate(columnName, "<", range.getHigh().getValue(), column, accumulator));
                        break;
                    }
                    default: {
                        throw new AssertionError((Object)("Unhandled bound: " + range.getHigh().getBound()));
                    }
                }
            }
            Preconditions.checkState((!rangeConjuncts.isEmpty() ? 1 : 0) != 0);
            disjuncts.add("(" + Joiner.on((String)" AND ").join(rangeConjuncts) + ")");
        }
        if (singleValues.size() == 1) {
            disjuncts.add(this.toPredicate(columnName, "=", Iterables.getOnlyElement(singleValues), column, accumulator));
        } else if (singleValues.size() > 1) {
            for (Object e : singleValues) {
                QueryBuilder.bindValue(e, column, accumulator);
            }
            String values = Joiner.on((String)",").join(Collections.nCopies(singleValues.size(), "?"));
            disjuncts.add(this.quote(columnName) + " IN (" + values + ")");
        }
        Preconditions.checkState((!disjuncts.isEmpty() ? 1 : 0) != 0);
        if (domain.isNullAllowed()) {
            disjuncts.add(this.quote(columnName) + " IS NULL");
        }
        return "(" + Joiner.on((String)" OR ").join(disjuncts) + ")";
    }

    private String toPredicate(String columnName, String operator, Object value, JdbcColumnHandle column, List<TypeAndValue> accumulator) {
        QueryBuilder.bindValue(value, column, accumulator);
        return this.quote(columnName) + " " + operator + " ?";
    }

    private String quote(String name) {
        return this.identifierQuote + name.replace(this.identifierQuote, this.identifierQuote + this.identifierQuote) + this.identifierQuote;
    }

    private static void bindValue(Object value, JdbcColumnHandle column, List<TypeAndValue> accumulator) {
        Type type = column.getColumnType();
        accumulator.add(new TypeAndValue(type, column.getJdbcTypeHandle(), value));
    }

    private static class TypeAndValue {
        private final Type type;
        private final JdbcTypeHandle typeHandle;
        private final Object value;

        public TypeAndValue(Type type, JdbcTypeHandle typeHandle, Object value) {
            this.type = Objects.requireNonNull(type, "type is null");
            this.typeHandle = Objects.requireNonNull(typeHandle, "typeHandle is null");
            this.value = Objects.requireNonNull(value, "value is null");
        }

        public Type getType() {
            return this.type;
        }

        public JdbcTypeHandle getTypeHandle() {
            return this.typeHandle;
        }

        public Object getValue() {
            return this.value;
        }
    }
}

