/*
 * Decompiled with CFR 0.152.
 */
package io.helidon.dbclient.jdbc;

import io.helidon.dbclient.DbClientException;
import io.helidon.dbclient.DbClientServiceContext;
import io.helidon.dbclient.DbExecuteContext;
import io.helidon.dbclient.DbIndexedStatementParameters;
import io.helidon.dbclient.DbNamedStatementParameters;
import io.helidon.dbclient.DbStatement;
import io.helidon.dbclient.DbStatementBase;
import io.helidon.dbclient.DbStatementParameters;
import io.helidon.dbclient.jdbc.JdbcConnectionPool;
import io.helidon.dbclient.jdbc.JdbcExecuteContext;
import io.helidon.dbclient.jdbc.NamedStatementParser;
import java.io.ByteArrayInputStream;
import java.io.CharArrayReader;
import java.io.InputStream;
import java.io.Reader;
import java.math.BigDecimal;
import java.math.BigInteger;
import java.sql.Connection;
import java.sql.Date;
import java.sql.PreparedStatement;
import java.sql.SQLException;
import java.sql.SQLXML;
import java.sql.Time;
import java.sql.Timestamp;
import java.time.LocalDate;
import java.time.LocalDateTime;
import java.time.LocalTime;
import java.time.OffsetDateTime;
import java.time.OffsetTime;
import java.util.ArrayList;
import java.util.Calendar;
import java.util.List;
import java.util.Map;
import java.util.UUID;

public abstract class JdbcStatement<S extends DbStatement<S>>
extends DbStatementBase<S> {
    private static final System.Logger LOGGER = System.getLogger(JdbcStatement.class.getName());
    private final JdbcConnectionPool connectionPool;
    private Connection connection;

    JdbcStatement(JdbcConnectionPool connectionPool, JdbcExecuteContext context) {
        super((DbExecuteContext)context);
        this.connectionPool = connectionPool;
    }

    void closeConnection() {
        try {
            if (this.connection != null) {
                this.connection.close();
            }
        }
        catch (SQLException e) {
            LOGGER.log(System.Logger.Level.WARNING, String.format("Could not close connection: %s", e.getMessage()), (Throwable)e);
        }
    }

    private JdbcExecuteContext jdbcContext() {
        return (JdbcExecuteContext)this.context(JdbcExecuteContext.class);
    }

    protected PreparedStatement prepareStatement(DbClientServiceContext serviceContext) {
        String stmtName = serviceContext.statementName();
        String stmt = serviceContext.statement();
        DbStatementParameters stmtParams = serviceContext.statementParameters();
        LOGGER.log(System.Logger.Level.DEBUG, () -> String.format("Building SQL statement: %s", stmt));
        if (stmtParams instanceof DbIndexedStatementParameters) {
            DbIndexedStatementParameters indexed = (DbIndexedStatementParameters)stmtParams;
            List params = indexed.parameters();
            return this.prepareIndexedStatement(stmtName, stmt, params);
        }
        if (stmtParams instanceof DbNamedStatementParameters) {
            DbNamedStatementParameters named = (DbNamedStatementParameters)stmtParams;
            Map params = named.parameters();
            return this.prepareNamedStatement(stmtName, stmt, params);
        }
        return this.prepareStatement(stmtName, stmt);
    }

    protected PreparedStatement prepareStatement(String stmtName, String stmt) {
        Connection connection = this.connectionPool.connection();
        try {
            connection.setAutoCommit(true);
        }
        catch (SQLException e) {
            throw new DbClientException("Failed to set autocommit to true", (Throwable)e);
        }
        return this.prepareStatement(connection, stmtName, stmt);
    }

    protected PreparedStatement prepareStatement(Connection connection, String stmtName, String stmt) {
        try {
            this.connection = connection;
            return connection.prepareStatement(stmt);
        }
        catch (SQLException e) {
            throw new DbClientException(String.format("Failed to prepare statement: %s", stmtName), (Throwable)e);
        }
    }

    private PreparedStatement prepareNamedStatement(String stmtName, String stmt, Map<String, Object> parameters) {
        PreparedStatement preparedStatement = null;
        try {
            NamedStatementParser parser = new NamedStatementParser(stmt);
            String convertedStmt = parser.convert();
            LOGGER.log(System.Logger.Level.TRACE, () -> String.format("Converted statement: %s", convertedStmt));
            preparedStatement = this.prepareStatement(stmtName, convertedStmt);
            List<String> namesOrder = parser.namesOrder();
            int i = 1;
            for (String name : namesOrder) {
                if (parameters.containsKey(name)) {
                    Object value = parameters.get(name);
                    LOGGER.log(System.Logger.Level.TRACE, String.format("Mapped parameter %d: %s -> %s", i, name, value));
                    this.setParameter(preparedStatement, i, value);
                    ++i;
                    continue;
                }
                throw new DbClientException(JdbcStatement.namedStatementErrorMessage(namesOrder, parameters));
            }
            return preparedStatement;
        }
        catch (SQLException e) {
            this.closePreparedStatement(preparedStatement);
            throw new DbClientException("Failed to prepare statement with named parameters: " + stmtName, (Throwable)e);
        }
    }

    private PreparedStatement prepareIndexedStatement(String stmtName, String stmt, List<Object> parameters) {
        PreparedStatement preparedStatement = null;
        try {
            preparedStatement = this.prepareStatement(stmtName, stmt);
            int i = 1;
            for (Object value : parameters) {
                LOGGER.log(System.Logger.Level.TRACE, String.format("Indexed parameter %d: %s", i, value));
                this.setParameter(preparedStatement, i, value);
                ++i;
            }
            return preparedStatement;
        }
        catch (SQLException e) {
            this.closePreparedStatement(preparedStatement);
            throw new DbClientException(String.format("Failed to prepare statement with indexed params: %s", stmtName), (Throwable)e);
        }
    }

    private void closePreparedStatement(PreparedStatement preparedStatement) {
        if (preparedStatement != null) {
            try {
                preparedStatement.close();
            }
            catch (SQLException e) {
                LOGGER.log(System.Logger.Level.WARNING, String.format("Could not close PreparedStatement: %s", e.getMessage()), (Throwable)e);
            }
        }
    }

    private static String namedStatementErrorMessage(List<String> names, Map<String, Object> parameters) {
        ArrayList<String> notInParams = new ArrayList<String>(names.size());
        for (String name : names) {
            if (parameters.containsKey(name)) continue;
            notInParams.add(name);
        }
        StringBuilder sb = new StringBuilder();
        sb.append("Query parameters missing in Map: ");
        boolean first = true;
        for (String name : notInParams) {
            if (first) {
                first = false;
            } else {
                sb.append(", ");
            }
            sb.append(name);
        }
        return sb.toString();
    }

    private void setParameter(PreparedStatement statement, int index, Object parameter) throws SQLException {
        if (parameter instanceof String) {
            String s = (String)parameter;
            if (this.jdbcContext().parametersConfig().useStringBinding() && s.length() > this.jdbcContext().parametersConfig().stringBindingSize()) {
                CharArrayReader reader = new CharArrayReader(s.toCharArray());
                statement.setCharacterStream(index, (Reader)reader, s.length());
            } else if (this.jdbcContext().parametersConfig().useNString()) {
                statement.setNString(index, s);
            } else {
                statement.setString(index, s);
            }
        } else if (parameter instanceof Number) {
            Number number = (Number)parameter;
            if (number instanceof Integer) {
                Integer i = (Integer)number;
                statement.setInt(index, i);
            } else if (number instanceof Long) {
                Long l = (Long)number;
                statement.setLong(index, l);
            } else if (number instanceof BigDecimal) {
                BigDecimal bd = (BigDecimal)number;
                statement.setBigDecimal(index, bd);
            } else if (number instanceof Double) {
                Double d = (Double)number;
                statement.setDouble(index, d);
            } else if (number instanceof Float) {
                Float f = (Float)number;
                statement.setFloat(index, f.floatValue());
            } else if (number instanceof Short) {
                Short s = (Short)number;
                statement.setShort(index, s);
            } else if (number instanceof Byte) {
                Byte b = (Byte)number;
                statement.setByte(index, b);
            } else if (number instanceof BigInteger) {
                BigInteger bi = (BigInteger)number;
                statement.setBigDecimal(index, new BigDecimal(bi));
            } else {
                statement.setObject(index, number);
            }
        } else if (parameter instanceof Date) {
            Date d = (Date)parameter;
            statement.setDate(index, d);
        } else if (parameter instanceof Time) {
            Time t = (Time)parameter;
            statement.setTime(index, t);
        } else if (parameter instanceof Timestamp) {
            Timestamp ts = (Timestamp)parameter;
            statement.setTimestamp(index, ts);
        } else if (parameter instanceof LocalDate) {
            LocalDate ld = (LocalDate)parameter;
            if (this.jdbcContext().parametersConfig().setObjectForJavaTime()) {
                statement.setObject(index, ld);
            } else {
                statement.setDate(index, Date.valueOf(ld));
            }
        } else if (parameter instanceof LocalDateTime) {
            LocalDateTime ldt = (LocalDateTime)parameter;
            if (this.jdbcContext().parametersConfig().setObjectForJavaTime()) {
                statement.setObject(index, ldt);
            } else {
                statement.setTimestamp(index, Timestamp.valueOf(ldt));
            }
        } else if (parameter instanceof OffsetDateTime) {
            OffsetDateTime odt = (OffsetDateTime)parameter;
            if (this.jdbcContext().parametersConfig().setObjectForJavaTime()) {
                statement.setObject(index, odt);
            } else {
                statement.setTimestamp(index, Timestamp.from(odt.toInstant()));
            }
        } else if (parameter instanceof LocalTime) {
            LocalTime lt = (LocalTime)parameter;
            if (this.jdbcContext().parametersConfig().setObjectForJavaTime()) {
                statement.setObject(index, lt);
            } else if (this.jdbcContext().parametersConfig().timestampForLocalTime()) {
                statement.setTimestamp(index, Timestamp.valueOf(LocalDateTime.of(LocalDate.ofEpochDay(0L), lt)));
            } else {
                statement.setTime(index, Time.valueOf(lt));
            }
        } else if (parameter instanceof OffsetTime) {
            OffsetTime ot = (OffsetTime)parameter;
            if (this.jdbcContext().parametersConfig().setObjectForJavaTime()) {
                statement.setObject(index, ot);
            } else {
                statement.setTimestamp(index, Timestamp.valueOf(LocalDateTime.of(LocalDate.ofEpochDay(0L), ot.toLocalTime())));
            }
        } else if (parameter instanceof Boolean) {
            Boolean b = (Boolean)parameter;
            statement.setBoolean(index, b);
        } else if (parameter == null) {
            statement.setNull(index, 0);
        } else if (parameter instanceof byte[]) {
            byte[] b = (byte[])parameter;
            if (this.jdbcContext().parametersConfig().useByteArrayBinding()) {
                ByteArrayInputStream inputStream = new ByteArrayInputStream(b);
                statement.setBinaryStream(index, (InputStream)inputStream, b.length);
            } else {
                statement.setBytes(index, b);
            }
        } else if (parameter instanceof Calendar) {
            Calendar c = (Calendar)parameter;
            statement.setTimestamp(index, JdbcStatement.timestampFromDate(c.getTime()));
        } else if (parameter instanceof java.util.Date) {
            java.util.Date d = (java.util.Date)parameter;
            statement.setTimestamp(index, JdbcStatement.timestampFromDate(d));
        } else if (parameter instanceof Character) {
            Character c = (Character)parameter;
            statement.setString(index, String.valueOf(c));
        } else if (parameter instanceof char[]) {
            char[] c = (char[])parameter;
            statement.setString(index, new String(c));
        } else if (parameter instanceof Character[]) {
            Character[] c = (Character[])parameter;
            statement.setString(index, String.valueOf(JdbcStatement.characterArrayToCharArray(c)));
        } else if (parameter instanceof Byte[]) {
            Byte[] b = (Byte[])parameter;
            statement.setBytes(index, JdbcStatement.byteArrayToByteArray(b));
        } else if (parameter instanceof SQLXML) {
            SQLXML s = (SQLXML)parameter;
            statement.setSQLXML(index, s);
        } else if (parameter instanceof UUID) {
            UUID uuid = (UUID)parameter;
            statement.setString(index, uuid.toString());
        } else {
            statement.setObject(index, parameter);
        }
    }

    private static Timestamp timestampFromLong(long millis) {
        Timestamp timestamp = new Timestamp(millis);
        if (millis % 1000L > 0L) {
            timestamp.setNanos((int)(millis % 1000L) * 1000000);
        } else if (millis % 1000L < 0L) {
            timestamp.setNanos((int)(1000000000L - Math.abs(millis % 1000L * 1000000L)));
        }
        return timestamp;
    }

    private static Timestamp timestampFromDate(java.util.Date date) {
        return JdbcStatement.timestampFromLong(date.getTime());
    }

    private static char[] characterArrayToCharArray(Character[] source) {
        char[] chars = new char[source.length];
        for (int i = 0; i < source.length; ++i) {
            chars[i] = source[i].charValue();
        }
        return chars;
    }

    private static byte[] byteArrayToByteArray(Byte[] source) {
        byte[] bytes = new byte[source.length];
        for (int i = 0; i < source.length; ++i) {
            Byte value = source[i];
            if (value == null) continue;
            bytes[i] = value;
        }
        return bytes;
    }
}

