/*
 * Decompiled with CFR 0.152.
 */
package io.grpc.okhttp;

import com.google.common.base.Preconditions;
import io.grpc.okhttp.OkHttpClientStream;
import io.grpc.okhttp.OkHttpClientTransport;
import io.grpc.okhttp.internal.framed.FrameWriter;
import java.io.IOException;
import java.util.ArrayDeque;
import java.util.Queue;
import javax.annotation.Nullable;
import okio.Buffer;

class OutboundFlowController {
    private final OkHttpClientTransport transport;
    private final FrameWriter frameWriter;
    private int initialWindowSize = 65535;
    private final OutboundFlowState connectionState = new OutboundFlowState(0);

    OutboundFlowController(OkHttpClientTransport transport, FrameWriter frameWriter) {
        this.transport = (OkHttpClientTransport)Preconditions.checkNotNull((Object)transport, (Object)"transport");
        this.frameWriter = (FrameWriter)Preconditions.checkNotNull((Object)frameWriter, (Object)"frameWriter");
    }

    void initialOutboundWindowSize(int newWindowSize) {
        if (newWindowSize < 0) {
            throw new IllegalArgumentException("Invalid initial window size: " + newWindowSize);
        }
        int delta = newWindowSize - this.initialWindowSize;
        this.initialWindowSize = newWindowSize;
        for (OkHttpClientStream stream : this.transport.getActiveStreams()) {
            OutboundFlowState state = (OutboundFlowState)stream.getOutboundFlowState();
            if (state == null) {
                state = new OutboundFlowState(stream);
                stream.setOutboundFlowState(state);
                continue;
            }
            state.incrementStreamWindow(delta);
        }
        if (delta > 0) {
            this.writeStreams();
        }
    }

    int windowUpdate(@Nullable OkHttpClientStream stream, int delta) {
        int updatedWindow;
        if (stream == null) {
            updatedWindow = this.connectionState.incrementStreamWindow(delta);
            this.writeStreams();
        } else {
            OutboundFlowState state = this.state(stream);
            updatedWindow = state.incrementStreamWindow(delta);
            WriteStatus writeStatus = new WriteStatus();
            state.writeBytes(state.writableWindow(), writeStatus);
            if (writeStatus.hasWritten()) {
                this.flush();
            }
        }
        return updatedWindow;
    }

    void data(boolean outFinished, int streamId, Buffer source, boolean flush) {
        Preconditions.checkNotNull((Object)source, (Object)"source");
        OkHttpClientStream stream = this.transport.getStream(streamId);
        if (stream == null) {
            return;
        }
        OutboundFlowState state = this.state(stream);
        int window = state.writableWindow();
        boolean framesAlreadyQueued = state.hasFrame();
        OutboundFlowState.Frame frame = state.newFrame(source, outFinished);
        if (!framesAlreadyQueued && window >= frame.size()) {
            frame.write();
            if (flush) {
                this.flush();
            }
            return;
        }
        frame.enqueue();
        if (framesAlreadyQueued || window <= 0) {
            if (flush) {
                this.flush();
            }
            return;
        }
        frame.split(window).write();
        if (flush) {
            this.flush();
        }
    }

    void flush() {
        try {
            this.frameWriter.flush();
        }
        catch (IOException e) {
            throw new RuntimeException(e);
        }
    }

    private OutboundFlowState state(OkHttpClientStream stream) {
        OutboundFlowState state = (OutboundFlowState)stream.getOutboundFlowState();
        if (state == null) {
            state = new OutboundFlowState(stream);
            stream.setOutboundFlowState(state);
        }
        return state;
    }

    private void writeStreams() {
        OutboundFlowState state;
        OkHttpClientStream[] streams = this.transport.getActiveStreams();
        int connectionWindow = this.connectionState.window();
        int numStreams = streams.length;
        while (numStreams > 0 && connectionWindow > 0) {
            int nextNumStreams = 0;
            int windowSlice = (int)Math.ceil((float)connectionWindow / (float)numStreams);
            for (int index = 0; index < numStreams && connectionWindow > 0; ++index) {
                OkHttpClientStream stream = streams[index];
                state = this.state(stream);
                int bytesForStream = Math.min(connectionWindow, Math.min(state.unallocatedBytes(), windowSlice));
                if (bytesForStream > 0) {
                    state.allocateBytes(bytesForStream);
                    connectionWindow -= bytesForStream;
                }
                if (state.unallocatedBytes() <= 0) continue;
                streams[nextNumStreams++] = stream;
            }
            numStreams = nextNumStreams;
        }
        WriteStatus writeStatus = new WriteStatus();
        for (OkHttpClientStream stream : this.transport.getActiveStreams()) {
            state = this.state(stream);
            state.writeBytes(state.allocatedBytes(), writeStatus);
            state.clearAllocatedBytes();
        }
        if (writeStatus.hasWritten()) {
            this.flush();
        }
    }

    private final class OutboundFlowState {
        final Queue<Frame> pendingWriteQueue;
        final int streamId;
        int queuedBytes;
        int window;
        int allocatedBytes;
        OkHttpClientStream stream;

        OutboundFlowState(int streamId) {
            this.window = OutboundFlowController.this.initialWindowSize;
            this.streamId = streamId;
            this.pendingWriteQueue = new ArrayDeque<Frame>(2);
        }

        OutboundFlowState(OkHttpClientStream stream) {
            this(stream.id());
            this.stream = stream;
        }

        int window() {
            return this.window;
        }

        void allocateBytes(int bytes) {
            this.allocatedBytes += bytes;
        }

        int allocatedBytes() {
            return this.allocatedBytes;
        }

        int unallocatedBytes() {
            return this.streamableBytes() - this.allocatedBytes;
        }

        void clearAllocatedBytes() {
            this.allocatedBytes = 0;
        }

        int incrementStreamWindow(int delta) {
            if (delta > 0 && Integer.MAX_VALUE - delta < this.window) {
                throw new IllegalArgumentException("Window size overflow for stream: " + this.streamId);
            }
            this.window += delta;
            return this.window;
        }

        int writableWindow() {
            return Math.min(this.window, OutboundFlowController.this.connectionState.window());
        }

        int streamableBytes() {
            return Math.max(0, Math.min(this.window, this.queuedBytes));
        }

        Frame newFrame(Buffer data, boolean endStream) {
            return new Frame(data, endStream);
        }

        boolean hasFrame() {
            return !this.pendingWriteQueue.isEmpty();
        }

        private Frame peek() {
            return this.pendingWriteQueue.peek();
        }

        int writeBytes(int bytes, WriteStatus writeStatus) {
            int bytesAttempted = 0;
            int maxBytes = Math.min(bytes, this.writableWindow());
            while (this.hasFrame()) {
                Frame pendingWrite = this.peek();
                if (maxBytes >= pendingWrite.size()) {
                    writeStatus.incrementNumWrites();
                    bytesAttempted += pendingWrite.size();
                    pendingWrite.write();
                } else {
                    if (maxBytes <= 0) break;
                    Frame partialFrame = pendingWrite.split(maxBytes);
                    writeStatus.incrementNumWrites();
                    bytesAttempted += partialFrame.size();
                    partialFrame.write();
                }
                maxBytes = Math.min(bytes - bytesAttempted, this.writableWindow());
            }
            return bytesAttempted;
        }

        private final class Frame {
            final Buffer data;
            final boolean endStream;
            boolean enqueued;

            Frame(Buffer data, boolean endStream) {
                this.data = data;
                this.endStream = endStream;
            }

            int size() {
                return (int)this.data.size();
            }

            void enqueue() {
                if (!this.enqueued) {
                    this.enqueued = true;
                    OutboundFlowState.this.pendingWriteQueue.offer(this);
                    OutboundFlowState.this.queuedBytes += this.size();
                }
            }

            void write() {
                do {
                    int bytesToWrite;
                    int frameBytes;
                    if ((frameBytes = Math.min(bytesToWrite = this.size(), OutboundFlowController.this.frameWriter.maxDataLength())) == bytesToWrite) {
                        OutboundFlowController.this.connectionState.incrementStreamWindow(-bytesToWrite);
                        OutboundFlowState.this.incrementStreamWindow(-bytesToWrite);
                        try {
                            OutboundFlowController.this.frameWriter.data(this.endStream, OutboundFlowState.this.streamId, this.data, bytesToWrite);
                        }
                        catch (IOException e) {
                            throw new RuntimeException(e);
                        }
                        OutboundFlowState.this.stream.transportState().onSentBytes(bytesToWrite);
                        if (this.enqueued) {
                            OutboundFlowState.this.queuedBytes -= bytesToWrite;
                            OutboundFlowState.this.pendingWriteQueue.remove(this);
                        }
                        return;
                    }
                    Frame frame = this.split(frameBytes);
                    frame.write();
                } while (this.size() > 0);
            }

            Frame split(int maxBytes) {
                assert (maxBytes < this.size()) : "Attempting to split a frame for the full size.";
                int dataSplit = Math.min(maxBytes, (int)this.data.size());
                Buffer splitSlice = new Buffer();
                splitSlice.write(this.data, (long)dataSplit);
                Frame frame = new Frame(splitSlice, false);
                if (this.enqueued) {
                    OutboundFlowState.this.queuedBytes -= dataSplit;
                }
                return frame;
            }
        }
    }

    private static final class WriteStatus {
        int numWrites;

        private WriteStatus() {
        }

        void incrementNumWrites() {
            ++this.numWrites;
        }

        boolean hasWritten() {
            return this.numWrites > 0;
        }
    }
}

