/*
 * Decompiled with CFR 0.152.
 */
package io.grpc.testing.integration;

import com.google.common.annotations.VisibleForTesting;
import com.google.common.base.Joiner;
import com.google.common.base.Objects;
import com.google.common.base.Preconditions;
import com.google.common.base.Splitter;
import com.google.common.collect.Iterators;
import com.google.common.util.concurrent.Futures;
import com.google.common.util.concurrent.ListenableFuture;
import com.google.common.util.concurrent.ListeningExecutorService;
import com.google.common.util.concurrent.MoreExecutors;
import io.grpc.BindableService;
import io.grpc.ManagedChannel;
import io.grpc.ManagedChannelBuilder;
import io.grpc.Server;
import io.grpc.ServerBuilder;
import io.grpc.Status;
import io.grpc.StatusException;
import io.grpc.internal.testing.TestUtils;
import io.grpc.netty.GrpcSslContexts;
import io.grpc.netty.NegotiationType;
import io.grpc.netty.NettyChannelBuilder;
import io.grpc.stub.StreamObserver;
import io.grpc.testing.integration.AbstractInteropTest;
import io.grpc.testing.integration.Metrics;
import io.grpc.testing.integration.MetricsServiceGrpc;
import io.grpc.testing.integration.TestCases;
import io.netty.handler.ssl.SslContext;
import java.io.IOException;
import java.net.InetAddress;
import java.net.InetSocketAddress;
import java.net.SocketAddress;
import java.net.URI;
import java.net.URISyntaxException;
import java.net.UnknownHostException;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.TimeUnit;
import java.util.logging.Level;
import java.util.logging.Logger;

public class StressTestClient {
    private static final Logger log = Logger.getLogger(StressTestClient.class.getName());
    private static final int WORKER_GRACE_PERIOD_SECS = 30;
    private List<InetSocketAddress> addresses = Collections.singletonList(new InetSocketAddress("localhost", 8080));
    private List<TestCaseWeightPair> testCaseWeightPairs = new ArrayList<TestCaseWeightPair>();
    private String serverHostOverride;
    private boolean useTls = false;
    private boolean useTestCa = false;
    private int durationSecs = -1;
    private int channelsPerServer = 1;
    private int stubsPerChannel = 1;
    private int metricsPort = 8081;
    private Server metricsServer;
    private final Map<String, Metrics.GaugeResponse> gauges = new ConcurrentHashMap<String, Metrics.GaugeResponse>();
    private volatile boolean shutdown;
    private final List<ListenableFuture<?>> workerFutures = new ArrayList();
    private final List<ManagedChannel> channels = new ArrayList<ManagedChannel>();
    private ListeningExecutorService threadpool;

    public static void main(String ... args) throws Exception {
        final StressTestClient client = new StressTestClient();
        client.parseArgs(args);
        Runtime.getRuntime().addShutdownHook(new Thread(){

            @Override
            public void run() {
                client.shutdown();
            }
        });
        try {
            client.startMetricsService();
            client.runStressTest();
            client.blockUntilStressTestComplete();
        }
        catch (Exception e) {
            log.log(Level.WARNING, "The stress test client encountered an error!", e);
        }
        finally {
            client.shutdown();
        }
    }

    @VisibleForTesting
    void parseArgs(String[] args) {
        boolean usage = false;
        String serverAddresses = "";
        for (String arg : args) {
            if (!arg.startsWith("--")) {
                System.err.println("All arguments must start with '--': " + arg);
                usage = true;
                break;
            }
            String[] parts = arg.substring(2).split("=", 2);
            String key = parts[0];
            if ("help".equals(key)) {
                usage = true;
                break;
            }
            if (parts.length != 2) {
                System.err.println("All arguments must be of the form --arg=value");
                usage = true;
                break;
            }
            String value = parts[1];
            if ("server_addresses".equals(key)) {
                serverAddresses = value;
                continue;
            }
            if ("server_host_override".equals(key)) {
                this.serverHostOverride = value;
                continue;
            }
            if ("use_tls".equals(key)) {
                this.useTls = Boolean.parseBoolean(value);
                continue;
            }
            if ("use_test_ca".equals(key)) {
                this.useTestCa = Boolean.parseBoolean(value);
                continue;
            }
            if ("test_cases".equals(key)) {
                this.testCaseWeightPairs = StressTestClient.parseTestCases(value);
                continue;
            }
            if ("test_duration_secs".equals(key)) {
                this.durationSecs = Integer.valueOf(value);
                continue;
            }
            if ("num_channels_per_server".equals(key)) {
                this.channelsPerServer = Integer.valueOf(value);
                continue;
            }
            if ("num_stubs_per_channel".equals(key)) {
                this.stubsPerChannel = Integer.valueOf(value);
                continue;
            }
            if ("metrics_port".equals(key)) {
                this.metricsPort = Integer.valueOf(value);
                continue;
            }
            System.err.println("Unknown argument: " + key);
            usage = true;
            break;
        }
        if (!usage && !serverAddresses.isEmpty()) {
            this.addresses = this.parseServerAddresses(serverAddresses);
            usage = this.addresses.isEmpty();
        }
        if (usage) {
            StressTestClient c = new StressTestClient();
            System.err.println("Usage: [ARGS...]\n\n  --server_host_override=HOST    Claimed identification expected of server.\n                                 Defaults to server host\n  --server_addresses=<name_1>:<port_1>,<name_2>:<port_2>...<name_N>:<port_N>\n    Default: " + StressTestClient.serverAddressesToString(c.addresses) + "\n  --test_cases=<testcase_1:w_1>,<testcase_2:w_2>...<testcase_n:w_n>\n    List of <testcase,weight> tuples. Weight is the relative frequency at which testcase is run.\n    Valid Testcases:" + StressTestClient.validTestCasesHelpText() + "\n  --use_tls=true|false           Whether to use TLS. Default: " + c.useTls + "\n  --use_test_ca=true|false       Whether to trust our fake CA. Requires --use_tls=true\n                                 to have effect. Default: " + c.useTestCa + "\n  --test_duration_secs=SECONDS   '-1' for no limit. Default: " + c.durationSecs + "\n  --num_channels_per_server=INT  Number of connections to each server address. Default: " + c.channelsPerServer + "\n  --num_stubs_per_channel=INT    Default: " + c.stubsPerChannel + "\n  --metrics_port=PORT            Listening port of the metrics server. Default: " + c.metricsPort);
            System.exit(1);
        }
    }

    @VisibleForTesting
    void startMetricsService() throws IOException {
        Preconditions.checkState((!this.shutdown ? 1 : 0) != 0, (Object)"client was shutdown.");
        this.metricsServer = ServerBuilder.forPort((int)this.metricsPort).addService((BindableService)new MetricsServiceImpl()).build().start();
    }

    @VisibleForTesting
    void runStressTest() throws Exception {
        Preconditions.checkState((!this.shutdown ? 1 : 0) != 0, (Object)"client was shutdown.");
        if (this.testCaseWeightPairs.isEmpty()) {
            return;
        }
        int numChannels = this.addresses.size() * this.channelsPerServer;
        int numThreads = numChannels * this.stubsPerChannel;
        this.threadpool = MoreExecutors.listeningDecorator((ExecutorService)Executors.newFixedThreadPool(numThreads));
        int serverIdx = -1;
        for (InetSocketAddress address : this.addresses) {
            ++serverIdx;
            for (int i = 0; i < this.channelsPerServer; ++i) {
                ManagedChannel channel = this.createChannel(address);
                this.channels.add(channel);
                for (int j = 0; j < this.stubsPerChannel; ++j) {
                    String gaugeName = String.format("/stress_test/server_%d/channel_%d/stub_%d/qps", serverIdx, i, j);
                    Worker worker = new Worker(channel, this.testCaseWeightPairs, this.durationSecs, gaugeName);
                    this.workerFutures.add(this.threadpool.submit((Runnable)worker));
                }
            }
        }
    }

    @VisibleForTesting
    void blockUntilStressTestComplete() throws Exception {
        Preconditions.checkState((!this.shutdown ? 1 : 0) != 0, (Object)"client was shutdown.");
        ListenableFuture f = Futures.allAsList(this.workerFutures);
        if (this.durationSecs == -1) {
            f.get();
        } else {
            f.get((long)(this.durationSecs + 30), TimeUnit.SECONDS);
        }
    }

    @VisibleForTesting
    void shutdown() {
        if (this.shutdown) {
            return;
        }
        this.shutdown = true;
        for (ManagedChannel ch : this.channels) {
            try {
                ch.shutdownNow();
                ch.awaitTermination(1L, TimeUnit.SECONDS);
            }
            catch (Throwable t) {
                log.log(Level.WARNING, "Error shutting down channel!", t);
            }
        }
        try {
            this.metricsServer.shutdownNow();
        }
        catch (Throwable t) {
            log.log(Level.WARNING, "Error shutting down metrics service!", t);
        }
        try {
            if (this.threadpool != null) {
                this.threadpool.shutdownNow();
            }
        }
        catch (Throwable t) {
            log.log(Level.WARNING, "Error shutting down threadpool.", t);
        }
    }

    @VisibleForTesting
    int getMetricServerPort() {
        return this.metricsServer.getPort();
    }

    private List<InetSocketAddress> parseServerAddresses(String addressesStr) {
        ArrayList<InetSocketAddress> addresses = new ArrayList<InetSocketAddress>();
        for (List<String> namePort : StressTestClient.parseCommaSeparatedTuples(addressesStr)) {
            InetAddress address;
            String name = namePort.get(0);
            int port = Integer.valueOf(namePort.get(1));
            try {
                address = InetAddress.getByName(name);
                if (this.serverHostOverride != null) {
                    address = InetAddress.getByAddress(this.serverHostOverride, address.getAddress());
                }
            }
            catch (UnknownHostException ex) {
                throw new RuntimeException(ex);
            }
            addresses.add(new InetSocketAddress(address, port));
        }
        return addresses;
    }

    private static List<TestCaseWeightPair> parseTestCases(String testCasesStr) {
        ArrayList<TestCaseWeightPair> testCaseWeightPairs = new ArrayList<TestCaseWeightPair>();
        for (List<String> nameWeight : StressTestClient.parseCommaSeparatedTuples(testCasesStr)) {
            TestCases testCase = TestCases.fromString(nameWeight.get(0));
            int weight = Integer.valueOf(nameWeight.get(1));
            testCaseWeightPairs.add(new TestCaseWeightPair(testCase, weight));
        }
        return testCaseWeightPairs;
    }

    private static List<List<String>> parseCommaSeparatedTuples(String str) {
        ArrayList<List<String>> tuples = new ArrayList<List<String>>();
        for (String tupleStr : Splitter.on((char)',').split((CharSequence)str)) {
            int splitIdx = tupleStr.lastIndexOf(58);
            if (splitIdx == -1) {
                throw new IllegalArgumentException("Illegal tuple format: '" + tupleStr + "'");
            }
            String part0 = tupleStr.substring(0, splitIdx);
            String part1 = tupleStr.substring(splitIdx + 1);
            tuples.add(Arrays.asList(part0, part1));
        }
        return tuples;
    }

    private ManagedChannel createChannel(InetSocketAddress address) {
        SslContext sslContext = null;
        if (this.useTestCa) {
            try {
                sslContext = GrpcSslContexts.forClient().trustManager(TestUtils.loadCert((String)"ca.pem")).build();
            }
            catch (Exception ex) {
                throw new RuntimeException(ex);
            }
        }
        return NettyChannelBuilder.forAddress((SocketAddress)address).negotiationType(this.useTls ? NegotiationType.TLS : NegotiationType.PLAINTEXT).sslContext(sslContext).build();
    }

    private static String serverAddressesToString(List<InetSocketAddress> addresses) {
        ArrayList<String> tmp = new ArrayList<String>();
        for (InetSocketAddress address : addresses) {
            URI uri;
            try {
                uri = new URI(null, null, address.getHostName(), address.getPort(), null, null, null);
            }
            catch (URISyntaxException e) {
                throw new RuntimeException(e);
            }
            tmp.add(uri.getAuthority());
        }
        return Joiner.on((char)',').join(tmp);
    }

    private static String validTestCasesHelpText() {
        StringBuilder builder = new StringBuilder();
        for (TestCases testCase : TestCases.values()) {
            String strTestcase = testCase.name().toLowerCase();
            builder.append("\n      ").append(strTestcase).append(": ").append(testCase.description());
        }
        return builder.toString();
    }

    @VisibleForTesting
    List<InetSocketAddress> addresses() {
        return Collections.unmodifiableList(this.addresses);
    }

    @VisibleForTesting
    String serverHostOverride() {
        return this.serverHostOverride;
    }

    @VisibleForTesting
    boolean useTls() {
        return this.useTls;
    }

    @VisibleForTesting
    boolean useTestCa() {
        return this.useTestCa;
    }

    @VisibleForTesting
    List<TestCaseWeightPair> testCaseWeightPairs() {
        return this.testCaseWeightPairs;
    }

    @VisibleForTesting
    int durationSecs() {
        return this.durationSecs;
    }

    @VisibleForTesting
    int channelsPerServer() {
        return this.channelsPerServer;
    }

    @VisibleForTesting
    int stubsPerChannel() {
        return this.stubsPerChannel;
    }

    @VisibleForTesting
    int metricsPort() {
        return this.metricsPort;
    }

    @VisibleForTesting
    static class TestCaseWeightPair {
        final TestCases testCase;
        final int weight;

        TestCaseWeightPair(TestCases testCase, int weight) {
            Preconditions.checkArgument((weight >= 0 ? 1 : 0) != 0, (Object)"weight must be positive.");
            this.testCase = (TestCases)((Object)Preconditions.checkNotNull((Object)((Object)testCase), (Object)"testCase"));
            this.weight = weight;
        }

        public boolean equals(Object other) {
            if (!(other instanceof TestCaseWeightPair)) {
                return false;
            }
            TestCaseWeightPair that = (TestCaseWeightPair)other;
            return this.testCase.equals((Object)that.testCase) && this.weight == that.weight;
        }

        public int hashCode() {
            return Objects.hashCode((Object[])new Object[]{this.testCase, this.weight});
        }
    }

    private class MetricsServiceImpl
    extends MetricsServiceGrpc.MetricsServiceImplBase {
        private MetricsServiceImpl() {
        }

        @Override
        public void getAllGauges(Metrics.EmptyMessage request, StreamObserver<Metrics.GaugeResponse> responseObserver) {
            for (Metrics.GaugeResponse gauge : StressTestClient.this.gauges.values()) {
                responseObserver.onNext((Object)gauge);
            }
            responseObserver.onCompleted();
        }

        @Override
        public void getGauge(Metrics.GaugeRequest request, StreamObserver<Metrics.GaugeResponse> responseObserver) {
            String gaugeName = request.getName();
            Metrics.GaugeResponse gauge = (Metrics.GaugeResponse)StressTestClient.this.gauges.get(gaugeName);
            if (gauge != null) {
                responseObserver.onNext((Object)gauge);
                responseObserver.onCompleted();
            } else {
                responseObserver.onError((Throwable)new StatusException(Status.NOT_FOUND));
            }
        }
    }

    private class Worker
    implements Runnable {
        private static final long METRICS_COLLECTION_INTERVAL_SECS = 5L;
        private final ManagedChannel channel;
        private final List<TestCaseWeightPair> testCaseWeightPairs;
        private final Integer durationSec;
        private final String gaugeName;

        Worker(ManagedChannel channel, List<TestCaseWeightPair> testCaseWeightPairs, int durationSec, String gaugeName) {
            Preconditions.checkArgument((durationSec >= -1 ? 1 : 0) != 0, (Object)"durationSec must be gte -1.");
            this.channel = (ManagedChannel)Preconditions.checkNotNull((Object)channel, (Object)"channel");
            this.testCaseWeightPairs = (List)Preconditions.checkNotNull(testCaseWeightPairs, (Object)"testCaseWeightPairs");
            this.durationSec = durationSec == -1 ? null : Integer.valueOf(durationSec);
            this.gaugeName = (String)Preconditions.checkNotNull((Object)gaugeName, (Object)"gaugeName");
        }

        @Override
        public void run() {
            Thread.currentThread().setName(this.gaugeName);
            Tester tester = new Tester();
            tester.setUp();
            WeightedTestCaseSelector testCaseSelector = new WeightedTestCaseSelector(this.testCaseWeightPairs);
            Long endTime = this.durationSec == null ? null : Long.valueOf(System.nanoTime() + TimeUnit.SECONDS.toNanos(StressTestClient.this.durationSecs));
            long lastMetricsCollectionTime = this.initLastMetricsCollectionTime();
            long testCasesSinceLastMetricsCollection = 0L;
            while (!(Thread.currentThread().isInterrupted() || StressTestClient.this.shutdown || endTime != null && endTime - System.nanoTime() <= 0L)) {
                try {
                    this.runTestCase(tester, testCaseSelector.nextTestCase());
                }
                catch (Exception e) {
                    throw new RuntimeException(e);
                }
                ++testCasesSinceLastMetricsCollection;
                double durationSecs = this.computeDurationSecs(lastMetricsCollectionTime);
                if (!(durationSecs >= 5.0)) continue;
                long qps = (long)Math.ceil((double)testCasesSinceLastMetricsCollection / durationSecs);
                Metrics.GaugeResponse gauge = Metrics.GaugeResponse.newBuilder().setName(this.gaugeName).setLongValue(qps).build();
                StressTestClient.this.gauges.put(this.gaugeName, gauge);
                lastMetricsCollectionTime = System.nanoTime();
                testCasesSinceLastMetricsCollection = 0L;
            }
        }

        private long initLastMetricsCollectionTime() {
            return System.nanoTime() - TimeUnit.SECONDS.toNanos(5L);
        }

        private double computeDurationSecs(long lastMetricsCollectionTime) {
            return (double)(System.nanoTime() - lastMetricsCollectionTime) / 1.0E9;
        }

        private void runTestCase(Tester tester, TestCases testCase) throws Exception {
            switch (testCase) {
                case EMPTY_UNARY: {
                    tester.emptyUnary();
                    break;
                }
                case LARGE_UNARY: {
                    tester.largeUnary();
                    break;
                }
                case CLIENT_STREAMING: {
                    tester.clientStreaming();
                    break;
                }
                case SERVER_STREAMING: {
                    tester.serverStreaming();
                    break;
                }
                case PING_PONG: {
                    tester.pingPong();
                    break;
                }
                case EMPTY_STREAM: {
                    tester.emptyStream();
                    break;
                }
                case UNIMPLEMENTED_METHOD: {
                    tester.unimplementedMethod();
                    break;
                }
                case UNIMPLEMENTED_SERVICE: {
                    tester.unimplementedService();
                    break;
                }
                case CANCEL_AFTER_BEGIN: {
                    tester.cancelAfterBegin();
                    break;
                }
                case CANCEL_AFTER_FIRST_RESPONSE: {
                    tester.cancelAfterFirstResponse();
                    break;
                }
                case TIMEOUT_ON_SLEEPING_SERVER: {
                    tester.timeoutOnSleepingServer();
                    break;
                }
                default: {
                    throw new IllegalArgumentException("Unknown test case: " + (Object)((Object)testCase));
                }
            }
        }

        class WeightedTestCaseSelector {
            final Iterator<TestCases> testCases;

            WeightedTestCaseSelector(List<TestCaseWeightPair> testCaseWeightPairs) {
                Preconditions.checkNotNull(testCaseWeightPairs, (Object)"testCaseWeightPairs");
                Preconditions.checkArgument((testCaseWeightPairs.size() > 0 ? 1 : 0) != 0);
                ArrayList<TestCases> testCases = new ArrayList<TestCases>();
                for (TestCaseWeightPair testCaseWeightPair : testCaseWeightPairs) {
                    for (int i = 0; i < testCaseWeightPair.weight; ++i) {
                        testCases.add(testCaseWeightPair.testCase);
                    }
                }
                Collections.shuffle(testCases);
                this.testCases = Iterators.cycle(testCases);
            }

            TestCases nextTestCase() {
                return this.testCases.next();
            }
        }

        class Tester
        extends AbstractInteropTest {
            Tester() {
            }

            @Override
            protected ManagedChannel createChannel() {
                return Worker.this.channel;
            }

            @Override
            protected ManagedChannelBuilder<?> createChannelBuilder() {
                throw new UnsupportedOperationException();
            }

            @Override
            protected int operationTimeoutMillis() {
                return Integer.MAX_VALUE;
            }

            @Override
            protected boolean metricsExpected() {
                return false;
            }
        }
    }
}

