/*
 * Decompiled with CFR 0.152.
 */
package io.grpc.gcp.observability;

import com.google.cloud.ServiceOptions;
import com.google.common.base.Preconditions;
import com.google.common.collect.ImmutableList;
import com.google.common.collect.ImmutableMap;
import com.google.common.collect.ImmutableSet;
import io.grpc.gcp.observability.ObservabilityConfig;
import io.grpc.internal.JsonParser;
import io.grpc.internal.JsonUtil;
import io.opencensus.trace.Sampler;
import io.opencensus.trace.samplers.Samplers;
import java.io.IOException;
import java.nio.charset.StandardCharsets;
import java.nio.file.Files;
import java.nio.file.Paths;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.logging.Level;
import java.util.logging.Logger;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

final class ObservabilityConfigImpl
implements ObservabilityConfig {
    private static final Logger logger = Logger.getLogger(ObservabilityConfigImpl.class.getName());
    private static final String CONFIG_ENV_VAR_NAME = "GRPC_GCP_OBSERVABILITY_CONFIG";
    private static final String CONFIG_FILE_ENV_VAR_NAME = "GRPC_GCP_OBSERVABILITY_CONFIG_FILE";
    private static final double EPSILON = 1.0E-6;
    private static final Pattern METHOD_NAME_REGEX = Pattern.compile("^([*])|((([\\w.]+)/((?:\\w+)|[*])))$");
    private boolean enableCloudLogging = false;
    private boolean enableCloudMonitoring = false;
    private boolean enableCloudTracing = false;
    private String projectId = null;
    private List<ObservabilityConfig.LogFilter> clientLogFilters;
    private List<ObservabilityConfig.LogFilter> serverLogFilters;
    private Sampler sampler;
    private Map<String, String> customTags;

    ObservabilityConfigImpl() {
    }

    static ObservabilityConfigImpl getInstance() throws IOException {
        ObservabilityConfigImpl config = new ObservabilityConfigImpl();
        String configFile = System.getenv(CONFIG_FILE_ENV_VAR_NAME);
        if (configFile != null) {
            config.parseFile(configFile);
        } else {
            config.parse(System.getenv(CONFIG_ENV_VAR_NAME));
        }
        return config;
    }

    void parseFile(String configFile) throws IOException {
        String configFileContent = new String(Files.readAllBytes(Paths.get(configFile, new String[0])), StandardCharsets.UTF_8);
        Preconditions.checkArgument((!configFileContent.isEmpty() ? 1 : 0) != 0, (Object)"GRPC_GCP_OBSERVABILITY_CONFIG_FILE is empty!");
        this.parse(configFileContent);
    }

    void parse(String config) throws IOException {
        Preconditions.checkArgument((config != null ? 1 : 0) != 0, (Object)"GRPC_GCP_OBSERVABILITY_CONFIG value is null!");
        this.parseConfig((Map)JsonParser.parse((String)config));
    }

    private void parseConfig(Map<String, ?> config) {
        Map rawCustomTagsObject;
        Map rawCloudTracingObject;
        Map rawCloudMonitoringObject;
        Preconditions.checkArgument((config != null ? 1 : 0) != 0, (Object)"Invalid configuration");
        if (config.isEmpty()) {
            this.clientLogFilters = Collections.emptyList();
            this.serverLogFilters = Collections.emptyList();
            this.customTags = Collections.emptyMap();
            return;
        }
        this.projectId = ObservabilityConfigImpl.fetchProjectId(JsonUtil.getString(config, (String)"project_id"));
        Map rawCloudLoggingObject = JsonUtil.getObject(config, (String)"cloud_logging");
        if (rawCloudLoggingObject != null) {
            this.enableCloudLogging = true;
            ImmutableList.Builder clientFiltersBuilder = new ImmutableList.Builder();
            ImmutableList.Builder serverFiltersBuilder = new ImmutableList.Builder();
            ObservabilityConfigImpl.parseLoggingObject(rawCloudLoggingObject, (ImmutableList.Builder<ObservabilityConfig.LogFilter>)clientFiltersBuilder, (ImmutableList.Builder<ObservabilityConfig.LogFilter>)serverFiltersBuilder);
            this.clientLogFilters = clientFiltersBuilder.build();
            this.serverLogFilters = serverFiltersBuilder.build();
        }
        if ((rawCloudMonitoringObject = JsonUtil.getObject(config, (String)"cloud_monitoring")) != null) {
            this.enableCloudMonitoring = true;
        }
        if ((rawCloudTracingObject = JsonUtil.getObject(config, (String)"cloud_trace")) != null) {
            this.enableCloudTracing = true;
            this.sampler = ObservabilityConfigImpl.parseTracingObject(rawCloudTracingObject);
        }
        if ((rawCustomTagsObject = JsonUtil.getObject(config, (String)"labels")) != null) {
            this.customTags = ObservabilityConfigImpl.parseCustomTags(rawCustomTagsObject);
        }
        if (this.clientLogFilters == null) {
            this.clientLogFilters = Collections.emptyList();
        }
        if (this.serverLogFilters == null) {
            this.serverLogFilters = Collections.emptyList();
        }
        if (this.customTags == null) {
            this.customTags = Collections.emptyMap();
        }
    }

    private static String fetchProjectId(String configProjectId) {
        String projectId = configProjectId != null ? configProjectId : ObservabilityConfigImpl.getDefaultGcpProjectId();
        Preconditions.checkArgument((projectId != null ? 1 : 0) != 0, (Object)"Unable to detect project_id");
        logger.log(Level.FINEST, "Found project ID : ", projectId);
        return projectId;
    }

    private static String getDefaultGcpProjectId() {
        return ServiceOptions.getDefaultProjectId();
    }

    private static void parseLoggingObject(Map<String, ?> rawLoggingConfig, ImmutableList.Builder<ObservabilityConfig.LogFilter> clientFilters, ImmutableList.Builder<ObservabilityConfig.LogFilter> serverFilters) {
        ObservabilityConfigImpl.parseRpcEvents(JsonUtil.getList(rawLoggingConfig, (String)"client_rpc_events"), clientFilters);
        ObservabilityConfigImpl.parseRpcEvents(JsonUtil.getList(rawLoggingConfig, (String)"server_rpc_events"), serverFilters);
    }

    private static Sampler parseTracingObject(Map<String, ?> rawCloudTracingConfig) {
        Sampler defaultSampler = Samplers.probabilitySampler((double)0.0);
        Double samplingRate = JsonUtil.getNumberAsDouble(rawCloudTracingConfig, (String)"sampling_rate");
        if (samplingRate == null) {
            return defaultSampler;
        }
        Preconditions.checkArgument((samplingRate >= 0.0 && samplingRate <= 1.0 ? 1 : 0) != 0, (Object)"'sampling_rate' needs to be between [0.0, 1.0]");
        return 1.0 - samplingRate < 1.0E-6 ? Samplers.alwaysSample() : Samplers.probabilitySampler((double)samplingRate);
    }

    private static Map<String, String> parseCustomTags(Map<String, ?> rawCustomTags) {
        ImmutableMap.Builder builder = new ImmutableMap.Builder();
        for (Map.Entry<String, ?> entry : rawCustomTags.entrySet()) {
            Preconditions.checkArgument((boolean)(entry.getValue() instanceof String), (Object)"'labels' needs to be a map of <string, string>");
            builder.put((Object)entry.getKey(), (Object)((String)entry.getValue()));
        }
        return builder.build();
    }

    private static void parseRpcEvents(List<?> rpcEvents, ImmutableList.Builder<ObservabilityConfig.LogFilter> filters) {
        if (rpcEvents == null) {
            return;
        }
        List jsonRpcEvents = JsonUtil.checkObjectList(rpcEvents);
        for (Map jsonClientRpcEvent : jsonRpcEvents) {
            filters.add((Object)ObservabilityConfigImpl.parseJsonLogFilter(jsonClientRpcEvent));
        }
    }

    private static ObservabilityConfig.LogFilter parseJsonLogFilter(Map<String, ?> logFilterMap) {
        ImmutableSet.Builder servicesSetBuilder = new ImmutableSet.Builder();
        ImmutableSet.Builder methodsSetBuilder = new ImmutableSet.Builder();
        boolean wildCardFilter = false;
        boolean excludeFilter = Boolean.TRUE.equals(JsonUtil.getBoolean(logFilterMap, (String)"exclude"));
        List methodsList = JsonUtil.getListOfStrings(logFilterMap, (String)"methods");
        if (methodsList != null) {
            wildCardFilter = ObservabilityConfigImpl.extractMethodOrServicePattern(methodsList, excludeFilter, (ImmutableSet.Builder<String>)servicesSetBuilder, (ImmutableSet.Builder<String>)methodsSetBuilder);
        }
        Integer maxHeaderBytes = JsonUtil.getNumberAsInteger(logFilterMap, (String)"max_metadata_bytes");
        Integer maxMessageBytes = JsonUtil.getNumberAsInteger(logFilterMap, (String)"max_message_bytes");
        return new ObservabilityConfig.LogFilter((Set<String>)servicesSetBuilder.build(), (Set<String>)methodsSetBuilder.build(), wildCardFilter, maxHeaderBytes != null ? maxHeaderBytes : 0, maxMessageBytes != null ? maxMessageBytes : 0, excludeFilter);
    }

    private static boolean extractMethodOrServicePattern(List<String> patternList, boolean exclude, ImmutableSet.Builder<String> servicesSetBuilder, ImmutableSet.Builder<String> methodsSetBuilder) {
        boolean globalFilter = false;
        for (String methodOrServicePattern : patternList) {
            Matcher matcher = METHOD_NAME_REGEX.matcher(methodOrServicePattern);
            Preconditions.checkArgument((boolean)matcher.matches(), (Object)("invalid service or method filter : " + methodOrServicePattern));
            if ("*".equals(methodOrServicePattern)) {
                Preconditions.checkArgument((!exclude ? 1 : 0) != 0, (Object)"cannot have 'exclude' and '*' wildcard in the same filter");
                globalFilter = true;
                continue;
            }
            if ("*".equals(matcher.group(5))) {
                String service = matcher.group(4);
                servicesSetBuilder.add((Object)service);
                continue;
            }
            methodsSetBuilder.add((Object)methodOrServicePattern);
        }
        return globalFilter;
    }

    @Override
    public boolean isEnableCloudLogging() {
        return this.enableCloudLogging;
    }

    @Override
    public boolean isEnableCloudMonitoring() {
        return this.enableCloudMonitoring;
    }

    @Override
    public boolean isEnableCloudTracing() {
        return this.enableCloudTracing;
    }

    @Override
    public String getProjectId() {
        return this.projectId;
    }

    @Override
    public List<ObservabilityConfig.LogFilter> getClientLogFilters() {
        return this.clientLogFilters;
    }

    @Override
    public List<ObservabilityConfig.LogFilter> getServerLogFilters() {
        return this.serverLogFilters;
    }

    @Override
    public Sampler getSampler() {
        return this.sampler;
    }

    @Override
    public Map<String, String> getCustomTags() {
        return this.customTags;
    }
}

