/*
 * Decompiled with CFR 0.152.
 */
package io.gravitee.rest.api.service.impl;

import com.fasterxml.jackson.databind.ObjectMapper;
import io.gravitee.common.data.domain.Page;
import io.gravitee.repository.management.model.Event;
import io.gravitee.rest.api.model.EventEntity;
import io.gravitee.rest.api.model.EventQuery;
import io.gravitee.rest.api.model.EventType;
import io.gravitee.rest.api.model.InstanceEntity;
import io.gravitee.rest.api.model.InstanceListItem;
import io.gravitee.rest.api.model.InstanceQuery;
import io.gravitee.rest.api.model.InstanceState;
import io.gravitee.rest.api.model.PluginEntity;
import io.gravitee.rest.api.service.EventService;
import io.gravitee.rest.api.service.InstanceService;
import io.gravitee.rest.api.service.common.ExecutionContext;
import io.gravitee.rest.api.service.exceptions.EventNotFoundException;
import io.gravitee.rest.api.service.exceptions.InstanceNotFoundException;
import java.io.IOException;
import java.time.Duration;
import java.time.Instant;
import java.time.temporal.ChronoUnit;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.Date;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.function.Function;
import java.util.function.Predicate;
import java.util.regex.Pattern;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.stereotype.Component;
import org.springframework.util.StringUtils;

@Component
public class InstanceServiceImpl
implements InstanceService {
    private static final Logger LOGGER = LoggerFactory.getLogger(InstanceServiceImpl.class);
    private static final Pattern PROPERTY_SPLITTER = Pattern.compile(", ");
    private final EventService eventService;
    private final ObjectMapper objectMapper;
    @Value(value="${gateway.unknown-expire-after:604800}")
    private long unknownExpireAfterInSec;
    private static final List<EventType> instancesAllState = new ArrayList<EventType>();
    private static final List<EventType> instancesRunningOnly = new ArrayList<EventType>();

    public InstanceServiceImpl(EventService eventService, ObjectMapper objectMapper) {
        instancesAllState.add(EventType.GATEWAY_STARTED);
        instancesAllState.add(EventType.GATEWAY_STOPPED);
        instancesRunningOnly.add(EventType.GATEWAY_STARTED);
        this.eventService = eventService;
        this.objectMapper = objectMapper;
    }

    @Override
    public Page<InstanceListItem> search(ExecutionContext executionContext, InstanceQuery query) {
        List<EventType> types = query.isIncludeStopped() ? instancesAllState : instancesRunningOnly;
        ExpiredPredicate filter = new ExpiredPredicate(Duration.ofSeconds(this.unknownExpireAfterInSec));
        return this.eventService.search(executionContext, types, query.getProperties(), query.getFrom(), query.getTo(), query.getPage(), query.getSize(), new Function<EventEntity, InstanceListItem>(){

            @Override
            public InstanceListItem apply(EventEntity eventEntity) {
                InstanceEntity instanceEntity = InstanceServiceImpl.this.convert(eventEntity);
                InstanceListItem item = new InstanceListItem();
                item.setId(instanceEntity.getId());
                item.setEvent(instanceEntity.getEvent());
                item.setHostname(instanceEntity.getHostname());
                item.setIp(instanceEntity.getIp());
                item.setPort(instanceEntity.getPort());
                item.setLastHeartbeatAt(instanceEntity.getLastHeartbeatAt());
                item.setStartedAt(instanceEntity.getStartedAt());
                item.setStoppedAt(instanceEntity.getStoppedAt());
                item.setVersion(instanceEntity.getVersion());
                item.setTags(instanceEntity.getTags());
                item.setTenant(instanceEntity.getTenant());
                item.setOperatingSystemName((String)instanceEntity.getSystemProperties().get("os.name"));
                item.setState(instanceEntity.getState());
                return item;
            }
        }, filter, Collections.singletonList(executionContext.getEnvironmentId()));
    }

    @Override
    public InstanceEntity findById(ExecutionContext executionContext, String instanceId) {
        EventQuery query = new EventQuery();
        query.setId(instanceId);
        query.setTypes(instancesAllState);
        Collection<EventEntity> events = this.eventService.search(executionContext, query);
        if (events == null || events.isEmpty()) {
            throw new InstanceNotFoundException(instanceId);
        }
        return this.convert(events.iterator().next());
    }

    @Override
    public InstanceEntity findByEvent(ExecutionContext executionContext, String eventId) {
        try {
            LOGGER.debug("Find instance by event ID: {}", (Object)eventId);
            EventEntity event = this.eventService.findById(executionContext, eventId);
            List<String> environments = this.extractProperty(event, Event.EventProperties.ENVIRONMENTS_HRIDS_PROPERTY.getValue());
            List<String> organizations = this.extractProperty(event, Event.EventProperties.ORGANIZATIONS_HRIDS_PROPERTY.getValue());
            return this.convert(event, environments, organizations);
        }
        catch (EventNotFoundException enfe) {
            throw new InstanceNotFoundException(eventId);
        }
    }

    @Override
    public List<InstanceEntity> findAllStarted(ExecutionContext executionContext) {
        LOGGER.debug("Find started instances by event");
        EventQuery query = new EventQuery();
        query.setTypes(instancesRunningOnly);
        Collection<EventEntity> events = this.eventService.search(executionContext, query);
        return events.stream().map(event -> {
            List<String> environments = this.extractProperty((EventEntity)event, Event.EventProperties.ENVIRONMENTS_HRIDS_PROPERTY.getValue());
            List<String> organizations = this.extractProperty((EventEntity)event, Event.EventProperties.ORGANIZATIONS_HRIDS_PROPERTY.getValue());
            return this.convert((EventEntity)event, environments, organizations);
        }).collect(Collectors.toList());
    }

    private List<String> extractProperty(EventEntity event, String property) {
        String extractedProperty = (String)event.getProperties().get(property);
        return extractedProperty == null ? List.of() : Stream.of(PROPERTY_SPLITTER.split(extractedProperty)).filter(StringUtils::hasText).collect(Collectors.toList());
    }

    private InstanceEntity convert(EventEntity event) {
        return this.convert(event, null, null);
    }

    private InstanceEntity convert(EventEntity event, List<String> environments, List<String> organizations) {
        Instant nowMinusXMinutes = Instant.now().minus(5L, ChronoUnit.MINUTES);
        Map props = event.getProperties();
        InstanceEntity instance = new InstanceEntity((String)props.get("id"));
        instance.setEvent(event.getId());
        if (!org.apache.commons.lang3.StringUtils.isBlank((CharSequence)((CharSequence)props.get("last_heartbeat_at")))) {
            instance.setLastHeartbeatAt(new Date(Long.parseLong((String)props.get("last_heartbeat_at"))));
        }
        if (!org.apache.commons.lang3.StringUtils.isBlank((CharSequence)((CharSequence)props.get("started_at")))) {
            instance.setStartedAt(new Date(Long.parseLong((String)props.get("started_at"))));
        }
        instance.setEnvironments(event.getEnvironments());
        instance.setEnvironmentsHrids(environments);
        instance.setOrganizationsHrids(organizations);
        if (event.getPayload() != null) {
            try {
                InstanceInfo info = (InstanceInfo)this.objectMapper.readValue(event.getPayload(), InstanceInfo.class);
                instance.setHostname(info.getHostname());
                instance.setIp(info.getIp());
                instance.setPort(info.getPort());
                instance.setTenant(info.getTenant());
                instance.setVersion(info.getVersion());
                instance.setTags(info.getTags());
                instance.setSystemProperties(info.getSystemProperties());
                instance.setPlugins(info.getPlugins());
            }
            catch (IOException ioe) {
                LOGGER.error("Unexpected error while getting instance data from event payload", (Throwable)ioe);
            }
        }
        if (event.getType() == EventType.GATEWAY_STARTED) {
            if (instance.getLastHeartbeatAt() == null || Instant.ofEpochMilli(instance.getLastHeartbeatAt().getTime()).isBefore(nowMinusXMinutes)) {
                instance.setState(InstanceState.UNKNOWN);
            } else {
                instance.setState(InstanceState.STARTED);
            }
        } else {
            instance.setState(InstanceState.STOPPED);
            if (!org.apache.commons.lang3.StringUtils.isBlank((CharSequence)((CharSequence)props.get("stopped_at")))) {
                instance.setStoppedAt(new Date(Long.parseLong((String)props.get("stopped_at"))));
            }
        }
        return instance;
    }

    public static final class ExpiredPredicate
    implements Predicate<InstanceListItem> {
        private Duration threshold;

        public ExpiredPredicate(Duration threshold) {
            this.threshold = threshold;
        }

        @Override
        public boolean test(InstanceListItem instance) {
            boolean result = true;
            if (instance.getState().equals((Object)InstanceState.UNKNOWN)) {
                Instant now = Instant.now();
                result = now.toEpochMilli() - instance.getLastHeartbeatAt().getTime() <= this.threshold.toMillis();
            }
            return result;
        }
    }

    private static class InstanceInfo {
        private String id;
        private String version;
        private List<String> tags;
        private Set<PluginEntity> plugins;
        private String hostname;
        private String ip;
        private String port;
        private String tenant;
        private Map<String, String> systemProperties;

        private InstanceInfo() {
        }

        public String getHostname() {
            return this.hostname;
        }

        public void setHostname(String hostname) {
            this.hostname = hostname;
        }

        public String getId() {
            return this.id;
        }

        public void setId(String id) {
            this.id = id;
        }

        public String getIp() {
            return this.ip;
        }

        public void setIp(String ip) {
            this.ip = ip;
        }

        public String getPort() {
            return this.port;
        }

        public void setPort(String port) {
            this.port = port;
        }

        public Map<String, String> getSystemProperties() {
            return this.systemProperties;
        }

        public void setSystemProperties(Map<String, String> systemProperties) {
            this.systemProperties = systemProperties;
        }

        public List<String> getTags() {
            return this.tags;
        }

        public void setTags(List<String> tags) {
            this.tags = tags;
        }

        public String getVersion() {
            return this.version;
        }

        public void setVersion(String version) {
            this.version = version;
        }

        public Set<PluginEntity> getPlugins() {
            return this.plugins;
        }

        public void setPlugins(Set<PluginEntity> plugins) {
            this.plugins = plugins;
        }

        public String getTenant() {
            return this.tenant;
        }

        public void setTenant(String tenant) {
            this.tenant = tenant;
        }
    }
}

