/*
 * Decompiled with CFR 0.152.
 */
package io.github.jeremylong.openvulnerability.client.ghsa;

import java.io.Closeable;
import java.io.IOException;
import java.time.Instant;
import java.util.Objects;
import java.util.concurrent.CancellationException;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.Executors;
import java.util.concurrent.RejectedExecutionException;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.Semaphore;
import java.util.concurrent.TimeUnit;
import java.util.function.Supplier;
import org.apache.hc.client5.http.async.methods.SimpleBody;
import org.apache.hc.client5.http.async.methods.SimpleHttpRequest;
import org.apache.hc.client5.http.async.methods.SimpleHttpResponse;
import org.apache.hc.client5.http.async.methods.SimpleRequestProducer;
import org.apache.hc.client5.http.async.methods.SimpleResponseConsumer;
import org.apache.hc.client5.http.impl.async.CloseableHttpAsyncClient;
import org.apache.hc.core5.concurrent.FutureCallback;
import org.apache.hc.core5.http.Header;
import org.apache.hc.core5.http.nio.AsyncRequestProducer;
import org.apache.hc.core5.http.nio.AsyncResponseConsumer;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public final class GithubRateLimitedAsyncClient
implements Closeable {
    private static final Logger LOG = LoggerFactory.getLogger(GithubRateLimitedAsyncClient.class);
    private final CloseableHttpAsyncClient delegate;
    private final ScheduledExecutorService scheduler;
    private final Semaphore concurrency;
    private final int maxSecondaryRetries;
    private final long minDelayBetweenRequestsMs;
    private final Object lock = new Object();
    private long remaining = Long.MAX_VALUE;
    private long resetEpochSeconds = 0L;
    private long nextAllowedRequestTimeMs = 0L;

    public GithubRateLimitedAsyncClient(CloseableHttpAsyncClient delegate, int maxConcurrentRequests, int maxRequestsPerMinute, int maxSecondaryRetries) {
        this.delegate = Objects.requireNonNull(delegate, "delegate");
        this.concurrency = new Semaphore(Math.max(1, maxConcurrentRequests));
        this.maxSecondaryRetries = Math.max(0, maxSecondaryRetries);
        this.minDelayBetweenRequestsMs = maxRequestsPerMinute <= 0 ? 0L : 60000L / (long)maxRequestsPerMinute;
        this.scheduler = Executors.newSingleThreadScheduledExecutor(r -> {
            Thread t = new Thread(r, "github-rate-limit-scheduler");
            t.setDaemon(true);
            return t;
        });
    }

    public CompletableFuture<SimpleHttpResponse> execute(Supplier<SimpleHttpRequest> requestSupplier) {
        Objects.requireNonNull(requestSupplier, "requestSupplier must not be null");
        CompletableFuture<SimpleHttpResponse> future = new CompletableFuture<SimpleHttpResponse>();
        this.scheduleAttempt(requestSupplier, future, 0);
        return future;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void scheduleAttempt(Supplier<SimpleHttpRequest> requestSupplier, CompletableFuture<SimpleHttpResponse> future, int attempt) {
        long delayMs;
        Object object = this.lock;
        synchronized (object) {
            long waitSec;
            long nowMs = System.currentTimeMillis();
            long nowSec = nowMs / 1000L;
            if (this.resetEpochSeconds > 0L && nowSec >= this.resetEpochSeconds) {
                this.remaining = Long.MAX_VALUE;
                this.resetEpochSeconds = 0L;
            }
            long primaryWaitMs = 0L;
            if (this.remaining <= 0L && this.resetEpochSeconds > 0L && (waitSec = this.resetEpochSeconds - nowSec) > 0L) {
                primaryWaitMs = waitSec * 1000L;
            }
            long smoothingWaitMs = Math.max(0L, this.nextAllowedRequestTimeMs - nowMs);
            delayMs = Math.max(primaryWaitMs, smoothingWaitMs);
            if (this.remaining > 0L) {
                --this.remaining;
            }
            if (this.minDelayBetweenRequestsMs > 0L) {
                long sendTime = nowMs + delayMs;
                this.nextAllowedRequestTimeMs = sendTime + this.minDelayBetweenRequestsMs;
            }
        }
        if (future.isDone()) {
            return;
        }
        try {
            this.scheduler.schedule(() -> this.doExecute(requestSupplier, future, attempt), delayMs, TimeUnit.MILLISECONDS);
        }
        catch (RejectedExecutionException e) {
            future.completeExceptionally(new CancellationException("Client closed"));
        }
    }

    private void doExecute(final Supplier<SimpleHttpRequest> requestSupplier, final CompletableFuture<SimpleHttpResponse> future, final int attempt) {
        SimpleHttpRequest request;
        if (future.isDone()) {
            return;
        }
        try {
            this.concurrency.acquire();
        }
        catch (InterruptedException e) {
            Thread.currentThread().interrupt();
            future.completeExceptionally(e);
            return;
        }
        try {
            request = requestSupplier.get();
        }
        catch (Throwable t) {
            this.concurrency.release();
            future.completeExceptionally(t);
            return;
        }
        this.delegate.execute((AsyncRequestProducer)SimpleRequestProducer.create((SimpleHttpRequest)request), (AsyncResponseConsumer)SimpleResponseConsumer.create(), (FutureCallback)new FutureCallback<SimpleHttpResponse>(){

            public void completed(SimpleHttpResponse response) {
                try {
                    GithubRateLimitedAsyncClient.this.handleCompleted(response, requestSupplier, future, attempt);
                }
                finally {
                    GithubRateLimitedAsyncClient.this.concurrency.release();
                }
            }

            public void failed(Exception ex) {
                try {
                    future.completeExceptionally(ex);
                }
                finally {
                    GithubRateLimitedAsyncClient.this.concurrency.release();
                }
            }

            public void cancelled() {
                try {
                    future.cancel(true);
                }
                finally {
                    GithubRateLimitedAsyncClient.this.concurrency.release();
                }
            }
        });
    }

    private void handleCompleted(SimpleHttpResponse response, Supplier<SimpleHttpRequest> requestSupplier, CompletableFuture<SimpleHttpResponse> future, int attempt) {
        long backoffMs;
        boolean canRetry;
        this.updatePrimaryRateLimitFromHeaders(response);
        int status = response.getCode();
        boolean rateLimitedStatus = this.isRateLimited(response, status);
        boolean bl = canRetry = attempt < this.maxSecondaryRetries;
        if (rateLimitedStatus && canRetry && (backoffMs = this.computeSecondaryBackoffMillis(response, attempt)) > 0L) {
            try {
                this.scheduler.schedule(() -> this.scheduleAttempt(requestSupplier, future, attempt + 1), backoffMs, TimeUnit.MILLISECONDS);
                if (LOG.isDebugEnabled()) {
                    LOG.debug("Request was rate limited (HTTP {}), retrying attempt {} in {} ms", new Object[]{status, attempt + 1, backoffMs});
                }
            }
            catch (RejectedExecutionException e) {
                future.completeExceptionally(new CancellationException("Client closed"));
            }
            return;
        }
        if (rateLimitedStatus) {
            String message = this.extractMessage(response);
            future.completeExceptionally(new IOException("GitHub rate limited the request (HTTP " + status + "): " + message));
        } else {
            future.complete(response);
        }
    }

    private boolean isRateLimited(SimpleHttpResponse response, int status) {
        if (status == 429) {
            return true;
        }
        if (status != 403) {
            return false;
        }
        Header retryAfter = response.getFirstHeader("Retry-After");
        if (retryAfter != null) {
            return true;
        }
        Header remainingHeader = response.getFirstHeader("X-RateLimit-Remaining");
        Header resetHeader = response.getFirstHeader("X-RateLimit-Reset");
        if (remainingHeader != null && "0".equals(remainingHeader.getValue()) && resetHeader != null) {
            return true;
        }
        String message = this.extractMessage(response);
        String lowerMessage = message.toLowerCase();
        return lowerMessage.contains("secondary rate limit") || lowerMessage.contains("wait a few minutes") || lowerMessage.contains("https://docs.github.com/en/site-policy/github-terms/github-terms-of-service");
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void updatePrimaryRateLimitFromHeaders(SimpleHttpResponse response) {
        Header remainingHeader = response.getFirstHeader("X-RateLimit-Remaining");
        Header resetHeader = response.getFirstHeader("X-RateLimit-Reset");
        if (remainingHeader == null && resetHeader == null) {
            return;
        }
        Object object = this.lock;
        synchronized (object) {
            if (remainingHeader != null) {
                try {
                    this.remaining = Long.parseLong(remainingHeader.getValue());
                    LOG.debug("Rate limit remaining: {}", (Object)this.remaining);
                }
                catch (NumberFormatException numberFormatException) {
                    // empty catch block
                }
            }
            if (resetHeader != null) {
                try {
                    this.resetEpochSeconds = Long.parseLong(resetHeader.getValue());
                    if (LOG.isDebugEnabled()) {
                        Instant resetInstant = Instant.ofEpochSecond(this.resetEpochSeconds);
                        LOG.debug("Rate limit resets in {} seconds", (Object)Math.max(0L, this.resetEpochSeconds - Instant.now().getEpochSecond()));
                    }
                }
                catch (NumberFormatException numberFormatException) {
                    // empty catch block
                }
            }
        }
    }

    private long computeSecondaryBackoffMillis(SimpleHttpResponse response, int attempt) {
        long baseBackoffSec = 60L;
        Header retryAfter = response.getFirstHeader("Retry-After");
        Header remainingHeader = response.getFirstHeader("X-RateLimit-Remaining");
        Header resetHeader = response.getFirstHeader("X-RateLimit-Reset");
        if (retryAfter != null) {
            try {
                if (LOG.isDebugEnabled()) {
                    LOG.debug("Using Retry-After header for secondary rate limit backoff: {} seconds", (Object)retryAfter.getValue());
                }
                baseBackoffSec = Long.parseLong(retryAfter.getValue());
            }
            catch (NumberFormatException numberFormatException) {}
        } else if (remainingHeader != null && "0".equals(remainingHeader.getValue()) && resetHeader != null) {
            try {
                long reset = Long.parseLong(resetHeader.getValue());
                long now = Instant.now().getEpochSecond();
                long diff = reset - now;
                if (LOG.isDebugEnabled()) {
                    LOG.debug("Using X-RateLimit-Reset header for secondary rate limit backoff: {} seconds", (Object)diff);
                }
                if (diff > 0L) {
                    baseBackoffSec = diff;
                }
            }
            catch (NumberFormatException reset) {
                // empty catch block
            }
        }
        long factor = 1L << attempt;
        return baseBackoffSec * factor * 1000L;
    }

    private String extractMessage(SimpleHttpResponse response) {
        SimpleBody body = response.getBody();
        if (body == null) {
            return "";
        }
        try {
            return body.getBodyText();
        }
        catch (UnsupportedOperationException e) {
            return "";
        }
    }

    @Override
    public void close() throws IOException {
        this.scheduler.shutdownNow();
        try {
            if (!this.scheduler.awaitTermination(5L, TimeUnit.SECONDS)) {
                LOG.warn("Scheduler did not terminate within timeout");
            }
        }
        catch (InterruptedException e) {
            Thread.currentThread().interrupt();
            LOG.warn("Interrupted while waiting for scheduler termination", (Throwable)e);
        }
        this.delegate.close();
    }
}

